#include "mbed.h"
#include "x_nucleo_53l0a1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/* This VL53L0X Expansion board test application performs a range measurement in polling mode
   on the sensors. The demo supports the centre sensor and up to 2 satelites boards. 
   
   The User Blue button controls the selected sensor. Measurements from this sensor are displayed 
   on the 4-digit display.
   Data collected from all sensors is also sent on the com port.
   
   The Reset button is used to restart the program. 
 */

#define VL53L0_I2C_SDA   D14 
#define VL53L0_I2C_SCL   D15 

static X_NUCLEO_53L0A1 *board=NULL;
VL53L0X_RangingMeasurementData_t data_sensor;
OperatingMode operating_mode;
    
/* Current sensor number*/
int currentSensor = 0;
/* Installed sensors count */ 
int sensorCnt = 0; 

/* array storing pointers to the sensors and their prefixes for the display purpouse*/
struct Sensors
{
    char prefix;
    VL53L0X *sensorPtr;
} installedSensors[3];

int InitSensorsArray()
{
   int status = 1;
   /* start the measure on sensor top */
   if (NULL != board->sensor_centre)
   {
       installedSensors[sensorCnt].prefix = 'C';
       installedSensors[sensorCnt].sensorPtr = board->sensor_centre;
       ++sensorCnt;
       status=board->sensor_centre->StartMeasurement(operating_mode, NULL);
   }
   /* start the measure on sensor left if present */
   if (NULL != board->sensor_left)
   {
        installedSensors[sensorCnt].prefix='L';
        installedSensors[sensorCnt].sensorPtr = board->sensor_left;
        ++sensorCnt;    
        status=board->sensor_left->StartMeasurement(operating_mode, NULL);
   }
   /* start the measure on sensor right if present */
   if (NULL != board->sensor_right)
   {
        installedSensors[sensorCnt].prefix = 'R';
        installedSensors[sensorCnt].sensorPtr = board->sensor_right;
        ++sensorCnt;
        status=board->sensor_right->StartMeasurement(operating_mode, NULL);
   }
   /* set first displayed sensor to embedded top */
   currentSensor = 0;
   return status;
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void SwitchMeasuringSensorIRQ(void)
{
    ++currentSensor;
    if (currentSensor == sensorCnt)
        currentSensor = 0;
    printf("Sensor changed to %c\r\n",installedSensors[currentSensor].prefix);
}

/* On board 4 digit local display refresh*/
void DisplayRefresh(OperatingMode op_mode)
{   
  char str[4];
   
  for (int t=0; t < sensorCnt; t++)
  {
     installedSensors[t].sensorPtr->GetMeasurement(op_mode, &data_sensor);
     if (data_sensor.RangeStatus == 0) // we have a valid range.
     {
      printf("%c %4d; ", installedSensors[t].prefix,data_sensor.RangeMilliMeter);
      if (currentSensor == t)
      {
          sprintf(str,"%c%3d", installedSensors[t].prefix ,data_sensor.RangeMilliMeter);
      }
     }
     else
     {
      printf("%c ----; ", installedSensors[t].prefix);
      if (currentSensor == t)
      {
       sprintf(str,"%c%s", installedSensors[t].prefix, "---");   
      }
     } 
  }
  printf("\r\n");
  board->display->DisplayString(str);
}

void RangeMeasure(DevI2C *device_i2c) {
   int status;

   /* creates the 53L0A1 expansion board singleton obj */
   board=X_NUCLEO_53L0A1::Instance(device_i2c, A2, D8, D2);
    
   board->display->DisplayString("53L0");
   /* init the 53L0A1 expansion board with default values */
   status=board->InitBoard();
   if(status)
      printf("Failed to init board!\n\r");   
   operating_mode=range_single_shot_polling;

   status = InitSensorsArray();
   
   if(!status)
   {
      while(1)
      {
         DisplayRefresh(operating_mode);
      }
   }
   board->display->DisplayString("BYE");
   delete board;        
}    

/*=================================== Main ==================================
 Press the blue user button to switch the sensor.    
=============================================================================*/
int main()
{   
#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-f401 
   InterruptIn stop_button (USER_BUTTON);
   stop_button.rise (&SwitchMeasuringSensorIRQ);  
#endif   
   DevI2C *device_i2c =new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);        
   RangeMeasure(device_i2c);  // start continuous measures
}

