#include "mbed.h"
#include "x_nucleo_53l0a1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/* This VL53L0X Expansion board test application performs a range measurement in interrupt mode
   on the center embedded sensor. 
   The measured data is displayed on the on-board 4-digit display, and sent to com port

   User Blue button stops the current measurement and the entire program, releasing all resources.
   Reset button is used to restart the program. 

   Interrupt mode requires callback function which handles IRQ from given sensor.
*/

#define VL53L0_I2C_SDA   D14 
#define VL53L0_I2C_SCL   D15 

static X_NUCLEO_53L0A1 *board=NULL;
VL53L0X_RangingMeasurementData_t data_sensor;
OperatingMode operating_mode;
    
/* flags that handle interrupt request for sensor and user blue button*/
volatile bool int_sensor_centre=false;
volatile bool int_measuring_stop = false;


/* ISR callback function of the center sensor */
void SensorCenterIRQ(void)
{
   int_sensor_centre=true;
   board->sensor_centre->DisableInterruptMeasureDetectionIRQ();
}   

/* ISR callback function of the user blue button to switch measuring sensor. */
void MeasuringStopIRQ(void)
{
    int_measuring_stop = true;
}

/* On board 4 digit local display refresh */
void DisplayRefresh(const VL53L0X_RangingMeasurementData_t &data)
{   
    char str[4];
    if (data.RangeStatus == 0) // we have a valid range.
    {
        printf("%4d; ", data.RangeMilliMeter);
        sprintf(str,"%4d", data.RangeMilliMeter);
    }
    else
    {
        sprintf(str,"%s", "----");   
    }       
    board->display->DisplayString(str);
}

void RangeMeasure(DevI2C *device_i2c) {
   int status;

   /* creates the 53L0A1 expansion board singleton obj */
   board=X_NUCLEO_53L0A1::Instance(device_i2c, A2, D8, D2);
    
   board->display->DisplayString("53L0");
   
   /* setting operating mode to continuous interrupt */
   operating_mode=range_continuous_interrupt;
   
   /* init the 53L0A1 expansion board with default values */
   status=board->InitBoard();
   if(status)
   {
        printf("Failed to init board!\n\r");   
        return;
   }
   //Stop any measurement before setting sensor
   status=board->sensor_centre->StopMeasurement(operating_mode);
   status=board->sensor_centre->StartMeasurement(operating_mode, &SensorCenterIRQ); 
   
   if(!status)
   {
     printf ("Entering loop mode\r\n");
     while (true)
     { 
        if (int_sensor_centre)
        {
            int_sensor_centre = false;
            status = board->sensor_centre->HandleIRQ(operating_mode, &data_sensor);
            DisplayRefresh(data_sensor);
        }

        if (int_measuring_stop)
        {
            printf("\r\nEnding loop mode \r\n");
            break;
        }
     }
   }
   board->display->DisplayString("BYE");
   delete board;        
}    

/*=================================== Main ==================================
 Press the blue user button to stop the measurements in progress    
=============================================================================*/
int main()
{   
#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-f401 
   InterruptIn stop_button (USER_BUTTON);
   stop_button.rise (&MeasuringStopIRQ);  
#endif   
   DevI2C *device_i2c =new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);        
   RangeMeasure(device_i2c);  // start continuous measures
}


