#include "mbed.h"
#include "rtos.h"
#include "dbg.h"
#include "DebouncedInterrupt.h"
#include "system.h"
#include "files.h"
#include "alp.h"

// Semaphore for notifiying button presses
Semaphore button_user(1);

alp_err_t d7_write_file(const uint8_t file_id,
                        const uint16_t offset,
                        const uint16_t file_size,
                        const uint8_t* const content)
{
    alp_err_t err = ALP_ERR_NO;
    void* file = NULL;
    
    //err = write_file(file_id, offset, file_size, content);
    PRINT("D7 File %d Write (offset %d, size %d)\r\n", file_id, offset, file_size);
    switch (file_id)
    {
        case REVISION_DEVICE_FILE_ID:
            file = &revision;
            break;
        case ALARM_DATA_FILE_ID:
            file = &alarm_data;
            break;
        case ALARM_CMD_FILE_ID:
            file = &alarm_cmd;
            // Update alarm file
            alarm_data.status = !content[0];
            // Simulate button press
            button_user.release();
            break;
        case TEMP_DATA_FILE_ID:
            file = &temp_data;
            break;
        default:
            PRINT("Unknown file id %d\r\n", file_id);
            err = ALP_ERR_FILE_NOT_FOUND;
            break;
    }
    
    if (file != NULL)
    {
        memcpy(file+offset, content, file_size);
    }
    
    return err;
}

alp_err_t d7_read_file( const uint8_t file_id,
                        const uint16_t offset,
                        const uint16_t file_size,
                        uint8_t* buf)
{
    alp_err_t err = ALP_ERR_NO;
    void* file = NULL;

    PRINT("D7 File %d Read (offset %d, size %d)\r\n", file_id, offset, file_size);

    switch (file_id)
    {
        case REVISION_DEVICE_FILE_ID:
            file = &revision;
            break;
        case ALARM_DATA_FILE_ID:
            file = &alarm_data;
            break;
        case ALARM_CMD_FILE_ID:
            file = &alarm_cmd;
            break;
        case TEMP_DATA_FILE_ID:
            file = &temp_data;
            break;
        default:
            PRINT("Unknown file id %d\r\n", file_id);
            err = ALP_ERR_FILE_NOT_FOUND;
            break;
    }
    
    if (file != NULL)
    {
        memcpy(buf, file+offset, file_size);
    }
    
    return err;
}

const d7_config_t shield1001_config = {
    .tx = PB_6,
    .rx = PA_10,
    .rts = PA_5,
    .cts = PC_7,
    .rx_buffer_size = 256,
    .local_timeout = 200,
    .distant_timeout = 3000,
    .write = d7_write_file,
    .read = d7_read_file,
};


// Checks the status of the report send.
void check_status( uint8_t status, char* message )
{
    switch (status)
    {
        case ALP_ERR_NO: // message is send and acknowleged
            PRINT("%s OK\r\n", message);
        break;
        default:
            PRINT("%s ERROR %d\r\n", message, status);
        break;
    }
}

// Interrupt Service Routine on button press.
void button_push_isr( void )
{
    button_user.release();
}

// This Thread monitors the user button
// and reports the alarm status
void alarm_thread( const void* args )
{    
    // Get modem
    D7_modem* wm1001 = (D7_modem*)args;

    // Enable interrupt on User button
    DebouncedInterrupt button(USER_BUTTON);
    button.attach(&button_push_isr, IRQ_FALL, 500, true);
            
    while(true)
    {
        // Wait for button press
        button_user.wait();
        
        // Invert alarm status
        alarm_data.status = !alarm_data.status;
        
        PRINT("NOTIFY ALARM STATE CHANGE %d\r\n", alarm_data.status);
        FLUSH();
        
        // Notify alarm status
        check_status(wm1001->notify_filechange(ALARM_DATA_FILE_ID), "ALARM REPORT");
    }
}

// This Thread reads the chip temperature
// and reports it every 5 minutes
void temp_thread( const void* args )
{
    // Get modem
    D7_modem* wm1001 = (D7_modem*)args;

    while(true)
    {
        temp_data.value = system_get_temperature();
        
        PRINT("NOTIFY TEMPERATURE %d C\r\n", temp_data.value);
        
        // Notify temp value
        check_status(wm1001->notify_filechange(TEMP_DATA_FILE_ID), "TEMP REPORT");

        // Wait 5 minutes
        // The function Thread::wait(...) takes a uin32_t as parameter
        // but the maximum value is uin16_t (65535)
        for (uint8_t i=0 ; i<5 ; i++) { Thread::wait(60000); }
    }
}

int main()
{
    // ----- Debug session over USB Serial ----- //
    DBG_OPEN();
        
    // Clear some lines on the terminal
    PRINT("\r\n\nBOOT\r\n");
    
    // Initialize system functions
    system_open();

    // Initialize ALP modem
    D7_modem wm1001(&shield1001_config, PB_0);
    
    // Wait for modem to notify its files if there is any
    Thread::wait(500);
    
    // Register the files
    wm1001.register_file(&revision_device_fh);
    wm1001.register_file(&alarm_data_fh);
    wm1001.register_file(&alarm_cmd_fh);
    wm1001.register_file(&temp_data_fh);    
    
    // Send the Revision report
    PRINT("NOTIFY REVISION\r\n");
    check_status(wm1001.notify_filechange(REVISION_DEVICE_FILE_ID), "REPORT REVISION");
    
    // Init Threads
    Thread t_alarm_thread(alarm_thread, &wm1001);
    Thread t_temp_thread(temp_thread, &wm1001);
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityIdle);
    while(true)
    {
        // Wait to avoid beeing stuck in loop
        Thread::wait(osWaitForever);
    }
}