#include "mbed.h"
#include "rtos.h"
#include "hts221.h"
#include "LPS25H.h"
#include "DataGenerator.h"
#include <string>
#include <time.h>

//Complex.Variables
DigitalOut myled(LED1);
I2C i2c2(I2C_SDA, I2C_SCL);
Ticker dataTimer;

//Method Headers
void readData();
void createDataThread();
void printData();
void storeNewRecord();

//Prim.Variables
float tempCelsius = 25.50;
float humi = 55;
float barometerPressure = 0;
float barometerTemperature = 0;
float dataReadTime = 2.0;
int humiMax = 100; 
char cmd=0;
uint32_t seconds = 0, minutes=0, hours=0; 
int headNode = 0;
int currentNode = 0;
const int maxNumRecords = 120;

//Can't print data in a ticker method (not interrupt safe)
//This is set once we have got new data, so it can be printed safely later
bool isNewData=false;

// Test Data generator setup
DataGenerator dataGen;

struct dataEntry{
        float airTemp;
        float airPress;
        float barTemp;
        float barPress;
        time_t dt;
    }tempEntry;
    
dataEntry storedDataArray [maxNumRecords];

int main(){
    printf("SOFT253 simple Temperature Humidity and Pressure Sensor Monitor\n\r");
    printf("Using the X-NUCLEO-IKS01A1 shield and MBED Libraries\n\r");
    
    dataTimer.attach(&readData, dataReadTime); //Attach timer to data reader.
    
    
    while(1) 
    {
        if (isNewData == true){
            storeNewRecord();
            printData();   
        }
        
        
        /* Flicker the LED. */
        myled = 1; // LED is ON
        Thread::wait(200); // 200 ms NB 'Thread::wait(int d);' !!! d is in milliseconds! 
        myled = 0; // LED is OFF
        Thread::wait(100); // 100 ms
    }
}

void storeNewRecord(){
    
    if(headNode>=120){
        headNode = 0;
    }
    
    tempEntry.dt = time(0);
    
    storedDataArray[headNode] = tempEntry;
    currentNode = headNode;
    headNode++;
}

/* Reads all the data in whenever the ticker interrupt is called */  
void readData(){
    //Check if old data has been printed if it hasn't don't do anything
    //As you have interrupted printdata section.
    if (isNewData == false){
        //Random data CHANGE ME
        
        dataGen.ReadTempHumi(&tempCelsius, &humi);

        barometerPressure = dataGen.pressure();
        barometerTemperature = dataGen.temperature();
    
        tempEntry.airTemp = tempCelsius;
        tempEntry.airPress = humi;
        tempEntry.barTemp = barometerTemperature;
        tempEntry.barPress = barometerPressure; 
        
        isNewData = true;
    }
}
/* Prints the data that was read */
void printData(){
    //Location where string is copied when using strftime
    char buff[20];
    strftime(buff, 20, "%Y-%m-%d %H:%M:%S", localtime(&tempEntry.dt));
    printf("%s \t", buff);  
    printf("%i: %4.2fC %3.1f%%", headNode, tempEntry.airTemp, tempEntry.airPress);
    printf(" %6.1f %4.1f\r\n", tempEntry.barPress, tempEntry.barTemp);
    isNewData = false;
}   



/* Old Code
LPS25H barometer(i2c2, LPS25H_V_CHIP_ADDR);
HTS221 humidity(I2C_SDA, I2C_SCL);

Humidity sensor setup.
humidity.init();
humidity.calib();

//printf("%#x\n\r",barometer.read_id());

barometer.get();
printf(" %6.1f %4.1f\r\n", barometer.pressure(), barometer.temperature());
humidity.ReadTempHumi(&tempCelsius, &humi);    
printf("%4.2fC %3.1f%%", tempCelsius, humi);
*/