#include "mbed.h"
#include "EthernetInterface.h"
#include "rtos.h"
#include "vfd.h"
#include "LPD8806.h"
#include "HTTPD.h"
#include "mDNSResponder.h"
#include "lwip/inet.h"
#include "lwip/netif.h"

DigitalOut myled(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);

LPD8806 strip = LPD8806(32);
EthernetInterface eth;
HTTPD *httpd;

//LocalFileSystem local("local");

void strip_clear() {
    int i;

    for (i = 0 ; i < strip.numPixels() ; i++) {
        // clear the strip
        strip.setPixelColor(i, strip.Color(0, 0, 0));
    }
}

void setPixelsTop(int start, int end, int colour) {
    int i;

    for (i = start; i < end + 1 ; i++) {
        strip.setPixelColor(i, colour);
    }
}

void setPixelsBottom(int start, int end, int colour) {
    int i;

    for (i = start; i < end + 1 ; i++) {
        strip.setPixelColor(16 + i, colour);
    }
}

/* 0 - 16 */
void top_strip(int quantity) {
    if (quantity < 16) {
        // blank unused bits.
        setPixelsTop(quantity, 15, 0);
    }

    if (quantity == 0) return;

    quantity --;

    setPixelsTop(0, quantity < 12 ? quantity : 11, strip.Color(0, 127, 0));

    if (quantity > 11)
        setPixelsTop(12, quantity < 14 ? quantity : 14, strip.Color(127, 127, 0));

    if (quantity > 13)
        setPixelsTop(14, quantity < 16 ? quantity : 16, strip.Color(127, 0, 0));
}

void bottom_strip(int quantity) {
    if (quantity < 16) {
        // blank unused bits.
        setPixelsBottom(quantity, 15, 0);
    }

    if (quantity == 0) return;
    quantity --;

    setPixelsBottom(0, quantity < 12 ? quantity : 11, strip.Color(0, 127, 0));

    if (quantity > 11)
        setPixelsBottom(12, quantity < 14 ? quantity : 14, strip.Color(127, 127, 0));

    if (quantity > 13)
        setPixelsBottom(14, quantity < 16 ? quantity : 16, strip.Color(127, 0, 0));
}

void emf_blue() {
        setPixelsBottom(0, 15, strip.Color(0, 161, 228));
        setPixelsTop(0, 15, strip.Color(0, 161, 228));
}

const static char *ok = "Ok";
const static char *fail = "Fail";

const char* const index_page =
    "<html><head>"
    "<title>LHS Bandwidth Meter</title>"
    "<LINK REL=\"SHORTCUT ICON\" HREF=\"/favicon.ico\">"
    "</head><body>"
    "<h1>Bandwidth Meter</h1>"
    "<h1>Commands:</h1>"

    "<h2>Led Strips:</h2>"
    "<ul>"
    "<li>/s/clear - clear the led strips</li>"
    "<li>/s/t?<i>amount</i> - set the top strip to <i>amount</i> (0-16).</li>"
    "<li>/s/b?<i>amount</i> - set the bottom strip to <i>amount</i> (0-16).</li>"
    "<li>/s/set?&lt;t,b&gt;hhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhhh - set the LEDs in the top ('t') or bottom ('b') strip to hhh to colours.</li>"
    "</ul>"
    "<p><a href=\"/s/clear\">clear</a></p>"

    "<h2>VFD</h2>"
    "<ul>"
    "<li>/v/r - reset the vfd.</li>"
    "<li>/v/c - clear the vfd.</li>"
    "<li>/v/p?<i>x,y</i> - set the cursor position to <i>x,y</i>. 0,0 is top left.</li>"
    "<li>/v/w?<i>text</i> - write <i>text</i> at the current cursor position</li>"
    "</ul>"

    "</body>"
    "</html>";

const char* const favicon = 
"\x00\x00\x01\x00\x01\x00\x10\x10\x02\x00\x01\x00\x01\x00\xb0\x00\x00\x00"
"\x16\x00\x00\x00(\x00\x00\x00\x10\x00\x00\x00 \x00\x00\x00\x01\x00\x01"
"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\x00\xf0\xf0\x00"
"\x00\xf0\xf0\x00\x00\xf0\xf0\x00\x00\xf0\xf0\x00\x00\x0f\x0f\x00\x00\x0f"
"\x0f\x00\x00\x0f\x0f\x00\x00\x0f\x0f\x00\x00\xf0\xf0\x00\x00\xf0\xf0\x00"
"\x00\xf0\xf0\x00\x00\xf0\xf0\x00\x00\x0f\x0f\x00\x00\x0f\x0f\x00\x00\x0f"
"\x0f\x00\x00\x0f\x0f\x00\x00\xf0\x0f\x00\x00\xc0\x03\x00\x00\x80\x01\x00"
"\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
"\x00\x80\x01\x00\x00\x80\x01\x00\x00\xc0\x03\x00\x00\xf0\x0f\x00\x00";

const char* const text_plain = "Content-Type: text/plain\r\n";

void callback_static (int id) {
    char buf[256];
 
    strcpy(buf, httpd->getFilename(id));
    printf("static %d %s\r\n", id, buf);
 
    if (strcmp(buf, "") == 0 || strcmp(buf, "index.html") == 0) {
        httpd->send(id, index_page, strlen(index_page), "Content-Type: text/html\r\n");
    } else if (strcmp(buf, "favicon.ico") == 0) {
        httpd->send(id, favicon, 198, "image/vnd.microsoft.icon\r\n");
    } else {
        httpd->httpdError(id, 404);
    }
}

void s_clear(int id) {
    httpd->send(id, ok, strlen(ok), text_plain);
    strip_clear();
    strip.show();
}

void s_top(int id) {
    int q;

    q = atoi(httpd->getQueryString(id));
    if (q < 0 || q > 16) {
        httpd->send(id, fail, strlen(fail), text_plain);
        return;
    }

    httpd->send(id, ok, strlen(ok), text_plain);
    top_strip(q);
    strip.show();
}

void s_bottom(int id) {
    int q;
    
    q = atoi(httpd->getQueryString(id));
    if (q < 0 || q > 16) {
        httpd->send(id, fail, strlen(fail), text_plain);
        return;
    }

    httpd->send(id, ok, strlen(ok), text_plain);
    bottom_strip(q);
    strip.show();
}

int hex_char_to_int(char h) {
    int ret;
    if (h >= '0' || h <= '9') {
        ret = h - 48;
    } else if (h >= 'A' || h <= 'F') {
        ret = (h - 'A') + 10;
    } else if (h >= 'a' || h <= 'f') {
        ret = (h - 'a') + 10;
    }
    return ret;
}

void s_set(int id) {
    int p, r, g, b, poff=0;
    const char* buf;

    buf = httpd->getQueryString(id);
    
    if (strlen(buf) != 49) {
        httpd->send(id, fail, strlen(fail), text_plain);
    }
    
    httpd->send(id, ok, strlen(ok), text_plain);

    if (buf[0] == 'b') {
        poff = 16;
    }

    for (p = 0; p < 16 ; p ++) {
        r = hex_char_to_int(buf[(p * 3) + 1]);
        g = hex_char_to_int(buf[(p * 3) + 2]);
        b = hex_char_to_int(buf[(p * 3) + 3]);
        strip.setPixelColor(p + poff , strip.Color(r,g,b));
    }

    strip.show();
}

void s_emf(int id) {
    emf_blue();
    strip.show();
}

void v_reset(int id) {
    httpd->send(id, ok, strlen(ok), text_plain);
    vfd_reset();
}

void v_clear(int id) {
    httpd->send(id, ok, strlen(ok), text_plain);
    vfd_init();
}

void v_write(int id) {
    char buf[128];
    
    if (strlen(httpd->getQueryString(id)) > 127) {
        httpd->send(id, fail, strlen(fail), text_plain);
        return;
    }

    httpd->send(id, ok, strlen(ok), text_plain);
    httpd->urldecode(httpd->getQueryString(id), buf, 128);
    send_text(buf);
}

void v_position(int id) {
    unsigned int v,h;

    httpd->send(id, ok, strlen(ok), text_plain);

    sscanf(httpd->getQueryString(id), "%d,%d", &h,&v);
    if (v > 3) v = 0;
    if (h > 19) h = 0;
    vfd_pos(h, v);
}

void v_logo(int id) {
    vfd_init();
    logo();
}

mDNSResponder mdns;

void mdns_thread(void const *args) {
    ip_addr_t ip, *z;
    z = (ip_addr_t *)args;

    ip.addr = z->addr;

    mdns.announce(ip, "net-o-meter", "_http._tcp", 80, "LHS Bandwidth Meter", 
    (char *[]) {                // NULL terminated list of KV's = see
            "path=/",         // http://www.zeroconf.org/Rendezvous/txtrecords.html
            NULL
        }
    );
}

int main() {
    ip_addr_t ip;

    led3 = 1;
    
    printf("Hello World\n\r");

    wait_ms(10);
    vfd_reset();
    wait_ms(10);
    vfd_init();
    wait_ms(10);
    logo();

    strip.begin();
    strip_clear();
    strip.show();

    eth.init(); //Use DHCP
    printf("Mac address is %s\n\r", eth.getMACAddress());
    eth.connect();
    printf("IP Address is %s\n\r", eth.getIPAddress());

    inet_aton(eth.getIPAddress(), &ip);
    
    Thread thread(mdns_thread, &ip);

    httpd = new HTTPD;

    led2 = 1;

    // LED strips
    httpd->attach("/s/t", &s_top);
    httpd->attach("/s/b", &s_bottom);
    httpd->attach("/s/clear", &s_clear);
    httpd->attach("/s/s", &s_set);
    httpd->attach("/s/e", &s_emf);

    // VFD
    httpd->attach("/v/r", &v_reset);
    httpd->attach("/v/c", &v_clear);
    httpd->attach("/v/w", &v_write);
    httpd->attach("/v/p", &v_position);
    httpd->attach("/v/l", &v_logo);

//    httpd->attach("/local/", "/local/");

    httpd->attach("/", &callback_static);
    httpd->start(80);
    printf("httpd ready\r\n");
 
    for (;;) {
        httpd->poll();
        if (myled == 0) {
            myled = 1;
        } else {
            myled = 0;
        }
    }    
}
