#include "mbed.h"
#include "freemaster_class.h"
//Initialize FreeMASTER driver
Freemaster fm(USBTX, USBRX);

//Add global variables
/* Test variables, will be displayed in the FreeMASTER application */
volatile unsigned char  var8;
volatile unsigned short var16;
volatile unsigned short var16inc = 100;
volatile unsigned long  var32;
volatile unsigned long  var32inc = 100;
volatile unsigned char  nAppCmdCounter;

/*
 * Test structure types - demonstrates the "TSA" feature thanks to which the 
 * FreeMASTER is able to load a variable and type information directly from 
 * the embedded application. 
 *
 */

typedef struct {
  
  unsigned short  aa;
  unsigned long bb[2];
  unsigned short  cc;
  unsigned long dd[3];
  unsigned char  ee;
  unsigned char  ff[5];
} INNER_STRUCT;

typedef struct {
  
  unsigned short  a;
  unsigned long b;
  INNER_STRUCT inA[4];
  INNER_STRUCT inB;
} OUTER_STRUCT;

/* Structure type information will be available in the FreeMASTER application (TSA) */
OUTER_STRUCT so1, so2;
INNER_STRUCT si1, si2;

/* buffer for runtime definition of the TSA table */
static unsigned long tsatable[(4*20)]; //up to 20 variables to register

/////////////////////////////////////////////////////////////////////////

FMSTR_APPCMD_RESULT myhandler(FMSTR_APPCMD_CODE /*nAppcmd*/, FMSTR_APPCMD_PDATA /*pData*/, FMSTR_SIZE /*nDataLen*/);

/*
 * With TSA enabled, the user describes the global and static variables using 
 * so-called TSA tables. There can be any number of tables defined in 
 * the project files. Each table does have the identifier which should be
 * unique across the project. 
 *
 * Note that you can declare variables as Read-Only or Read-Write.
 * The FreeMASTER driver denies any write access to the Read-Only variables
 * when TSA_SAFETY is enabled.
 */

FMSTR_TSA_TABLE_BEGIN(first_table)
    FMSTR_TSA_RW_VAR(var8,     FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(var32,    FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(var32inc, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(so1,      FMSTR_TSA_USERTYPE(OUTER_STRUCT))
    FMSTR_TSA_RW_VAR(si1,      FMSTR_TSA_USERTYPE(INNER_STRUCT))

    FMSTR_TSA_STRUCT(OUTER_STRUCT)
    FMSTR_TSA_MEMBER(OUTER_STRUCT, a,   FMSTR_TSA_UINT16)
    FMSTR_TSA_MEMBER(OUTER_STRUCT, b,   FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(OUTER_STRUCT, inA, FMSTR_TSA_USERTYPE(INNER_STRUCT))
    FMSTR_TSA_MEMBER(OUTER_STRUCT, inB, FMSTR_TSA_USERTYPE(INNER_STRUCT))
    
    FMSTR_TSA_STRUCT(INNER_STRUCT)
    FMSTR_TSA_MEMBER(INNER_STRUCT, aa, FMSTR_TSA_UINT16)
    FMSTR_TSA_MEMBER(INNER_STRUCT, bb, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(INNER_STRUCT, cc, FMSTR_TSA_SINT16)
    FMSTR_TSA_MEMBER(INNER_STRUCT, dd, FMSTR_TSA_SINT32)
    FMSTR_TSA_MEMBER(INNER_STRUCT, ee, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(INNER_STRUCT, ff, FMSTR_TSA_SINT8)
FMSTR_TSA_TABLE_END()

/*
 * This is an example of another TSA table. Typically, you put one table
 * to each .c file where your global or static variables are instantiated.
 */

FMSTR_TSA_TABLE_BEGIN(next_table)
    FMSTR_TSA_RO_VAR(so2, FMSTR_TSA_USERTYPE(OUTER_STRUCT))
    FMSTR_TSA_RO_VAR(si2, FMSTR_TSA_USERTYPE(INNER_STRUCT))
FMSTR_TSA_TABLE_END()

//Registration of all variables to observe in the FreeMASTER tool
FMSTR_TSA_TABLE_LIST_BEGIN()
//Register all TSA tables
    FMSTR_TSA_TABLE(first_table)
    FMSTR_TSA_TABLE(next_table)
FMSTR_TSA_TABLE_LIST_END()

/*
 * This function is registerred as a application command handler (see 
 * main() below. It gets automatically invoked when the FreeMASTER
 * application sends appropriate application command.
 *
 */

FMSTR_APPCMD_RESULT myhandler(FMSTR_APPCMD_CODE nAppcmd, FMSTR_APPCMD_PDATA pData, FMSTR_SIZE nDataLen)
{
    // the return value is used as the application command result code
    return 0x10;
}


int main() {
  //change baudrate to 9600
  fm.baud(9600);
  //register global or static variables to FreeMASTER driver
  fm.TsaAddVar(FMSTR_TSA_RW_VAR_CFG(var16inc,FMSTR_TSA_UINT16));
  //register read only variable to FreeMASTER driver
  fm.TsaAddVar(FMSTR_TSA_RO_VAR_CFG(var16,FMSTR_TSA_UINT8));

  while(1) {
    //execute demo code
    unsigned short nAppCmdCode;
    // scope variables
    var16 += var16inc;
    var32 += var32inc;

    // the application commands not registered with callback handlers
    // can be detected and processed using the API calls below
        
    // first, check if a new command has been received
    nAppCmdCode = fm.GetAppCmd();

    // when a new command arrives, the nAppCmdCode contains the application 
    // command code. In other case, the "NOCMD" special value is returned
    if (nAppCmdCode != FMSTR_APPCMDRESULT_NOCMD)
    {
        nAppCmdCounter++;
            
        // each command may have different processing and different 
        // result code. The command processing is finished by 
        // calling FMSTR_AppCmdAck() with the result code value
        switch(nAppCmdCode)
        {
            case 1: fm.AppCmdAck(var8); break;
            case 2: fm.AppCmdAck((unsigned char) ~var8); break;
            default: fm.AppCmdAck(0); break;
        }
    }

    // This call should be placed in the timer interrupt or anywhere where
    // the recorder sampling should occur.
    fm.Recorder();

    // The FreeMASTER poll call must be called in the main application loop
    // to handle the communication interface and protocol. 
    // In LONG_INTR FreeMASTER interrupt mode, all the processing is done 
    // during the communication interrupt routine and the FMSTR_Poll() is 
    // compiled empty.
    fm.Poll();
    wait(0.0025);
  }
}