#ifndef PAUSE_H
#define PAUSE_H

#include "mbed.h"
#include "Gamepad.h"
#include "ClassicEngine.h"
#include "N5110.h"

/** Enum for game mode*/
enum Mode{
    CLASSIC_MODE, 
    BRICKBREAKER_MODE
    };

/** Enum for Pause Options*/
enum PauseOption{
    RESUME,
    RESTART,
    QUIT,
    HELP
    };

/**Pause selection struct*/
struct PauseSelection{
    int output;                 /**<Integer output for line to print arrows*/
    PauseOption next_state[3];  /**<Array of enums for possible next option*/
    };

/** Pause Class
@brief Library to power the pause menu
@brief Can pass frame values to control the flow of the game modes they're
@brief utilised in, enabling features like restarting and quitting

@author James Cummins

@code

#include "mbed.h"
#include "Pause.h"

Pause pause;        //create a pause object for the pause menu
Gamepad gamepad;
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);

int main(){
    //first initialise the pause menu so it points to top item first
    pause.init();
    int game_length = 90;   //desired game mode length in seconds
    int fps = 30;           //frames per sec of the game mode
    
    //use either one or the other of these (for the correct game mode)
    //to display instructions before the game commences
    pause.brickbreaker_help(gamepad, lcd);
    pause.classic_help(gamepad, lcd);
    
    for(int i = 0; i < game_length*fps; i++){
        //
        //
        //code to operate a time based game mode
        //
        //
        
        if(gamepad.check_event(gamepad.BACK_PRESSED)){
        //retrieve the user's choice from the pause menu
            PauseOption choice = pause.pause_menu(gamepad, lcd, fps);
        //jump to the appropriate point in the game mode
            i = pause.brickbreaker_action(choice, gamepad, lcd, i, fps);
        }
    }
}

@endcode
*/

class Pause {

public:
    /** 
    * @brief Create a Pause object
    */
    Pause();
    /** 
    * @brief Delete Pause object to free memory
    */
    ~Pause();
    /** 
    * @brief Initialise the pause menu with the first menu option highlighted
    */
    void init();
    /** 
    * @brief Read the user input and update the displayed menu until an option is selected
    * @param gamepad - Gamepad object to sense user interaction
    * @param lcd - N5110 object to render menu on LCD screen
    * @param fps - integer value of frames per second game is ran at
    * @returns an Enum of the pause option selected by user (RESUME, RESTART, QUIT or HELP)
    */
    PauseOption pause_menu(Gamepad &gamepad, N5110 &lcd, int fps);
    /** 
    * @brief Jump to a specified frame in the game based on user's menu choice
    * @param choice - option selected in the Pause menu
    * @param gamepad - Gamepad object to check for button press
    * @param lcd - N5110 object to interact with LCD screen
    * @param frame - integer of current no. of frames iterated through in gameplay
    * @param fps - integer of gameplay's frames per second
    * @returns i - integer of frame to jump to
    */
    int brickbreaker_action(PauseOption choice, Gamepad &gamepad, N5110 &lcd, int frame, int fps);
    /** 
    * @brief Display the help menu for classic game mode
    * @param gamepad - Gamepad object to check for button press
    * @param lcd - N5110 object to render help menu on LCD
    */
    void classic_help(Gamepad &gamepad, N5110 &lcd);
    /** 
    * @brief Display the help menu for brickbreaker game mode
    * @param gamepad - Gamepad object to check for button press
    * @param lcd - N5110 object to render help menu on LCD
    */
    void brickbreaker_help(Gamepad &gamepad, N5110 &lcd);

private:

    void display_pause_options(N5110 &lcd);
    PauseOption pause_selection(Gamepad &gamepad, N5110 &lcd);
    PauseOption _state;
    int _next_state;
};
#endif