#ifndef OPTIONSENGINE_H
#define OPTIONSENGINE_H

#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Ball.h"
#include "Pause.h"
#include "SDFileSystem.h"

/** Enum for choice currently selected in Options Menu*/
enum Option{
    BRIGHTNESS, /**<Top menu option*/
    BALL_SPEED, /**<Middle menu option*/
    HIGH_SCORES /**<Bottom menu option*/
    };

/** Option Selection struct*/
struct OptionSelection{
    int output;             /**< Integer output for line to print arrows */
    Option next_state[3];   /**< Array of enums for possible next option */
    };

/** Options Engine Class
@brief Library to power the options menu
@brief Features methods 

@author James Cummins

@code

#include "mbed.h"
#include "OptionsEngine.h"

//Create an engine object for the Options Menu
OptionsEngine opt;
Gamepad gamepad;
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Ball ball;

int main(){
    //Enum for the user's option choice initialised to first menu item (brightness)
    Option choice = BRIGHTNESS;
    while(!(gamepad.check_event(gamepad.A_PRESSED))){
    //display options renders the options menu
    //option selection renders the arrows which point to the current selection
        lcd.clear();
        opt.display_options(lcd);
        choice = opt.option_selection(gamepad, lcd);
        lcd.refresh();
    //longer wait time than normal game fps to compensate for button bounce
        wait(0.2);
    }
    //each menu option called by its respective enum
    if(choice == BRIGHTNESS){ opt.change_brightness(gamepad, lcd); }
    if(choice == BALL_SPEED){ opt.change_ball_speed(gamepad, lcd, ball); }
    if(choice == HIGH_SCORES){ opt.view_high_scores(gamepad, lcd); }
}

@endcode

*/

class OptionsEngine {

public:
    /** 
    * @brief Create an engine object for the Options menu
    */
    OptionsEngine();
    /** 
    * @brief Delete an Options Engine object to free up memory
    */
    ~OptionsEngine();
    /** 
    * @brief Initialise Options menu settings. Set brightness to 50%, set initial
    * option to the top one in the list, and set ball speed sensitivity to 5/10
    */
    void init();
    /** 
    * @brief Render the options menu on the LCD screen
    * @param lcd - N5110 object to interact with LCD screen
    */
    void display_options(N5110 &lcd);
    /** 
    * @brief Read the user input, highlight and store the current option on the screen
    * @param gamepad - Gamepad object to read the user input
    * @param lcd - N5110 object to display arrows on LCD screen
    * @returns an Enum of the option selected - BRIGHTNESS, BALL_SPEED or HIGH_SCORES
    */
    Option option_selection(Gamepad &gamepad, N5110 &lcd);
    /** 
    * @brief Set the brightness of the LCD screen
    * @param gamepad - Gamepad object to detect button press
    * @param lcd - N5110 object to alter brightness and display graphic interface
    */
    void change_brightness(Gamepad &gamepad, N5110 &lcd);
    /** 
    * @brief Set the ball speed multiplier
    * @param gamepad - Gamepad object to detect button press
    * @param lcd - N5110 object to display graphic interface
    * @details ball speed is a continual multiplier to the values sensed by the 
    *          accelerometer, altering the sensitivity of the sensed input
    *          (i.e the same change in tilt causes a greater change in the
    *          ball's position
    */
    void change_ball_speed(Gamepad &gamepad, N5110 &lcd, Ball &ball);
    /** 
    * @brief Display the high scores for the selected game mode on the LCD screen
    * @param gamepad - Gamepad object to sense user's selection
    * @param lcd - N5110 object to display chosen leaderboard on LCD Screen
    */
    void view_high_scores(Gamepad &gamepad, N5110 &lcd);

private:
    void read_brightness_input(Gamepad &gamepad);
    void read_ball_speed_input(Gamepad &gamepad);
    void print_high_scores(N5110 &lcd);
    void read_classic_high_scores();
    void read_bb_high_scores();
    Option _state;
    int _next_state;
    float _brightness;
    int _ball_speed;
    Mode _leaderboard;
    int _classic_index[6];
    float _classic_values[6];
    int _bb_index[6];
    float _bb_values[6];
    
};
#endif