#include "OptionsEngine.h"

//constructor
OptionsEngine::OptionsEngine(){
}

//destructor
OptionsEngine::~OptionsEngine(){
}

//initialiser
void OptionsEngine::init(){
    _state = BRIGHTNESS;        //first item in menu
    _brightness = 0.5;          //medium brightness
    _ball_speed = 5;            // 5/10 sensitivity
}

void OptionsEngine::display_options(N5110 &lcd){
    lcd.printString("Options menu", 6, 0);      //text for each line of display
    lcd.printString("Brightness", 12, 2);
    lcd.printString("Ball speed", 12, 3);
    lcd.printString("High scores", 9, 4);
}

Option OptionsEngine::option_selection(Gamepad &gamepad, N5110 &lcd){
    OptionSelection fsm[3] = {      //finite state machine to power the menu
        {2,{HIGH_SCORES, BALL_SPEED, BRIGHTNESS}},    //output (e.g. 2,3,4) is line to print arrows on for user interface
        {3,{BRIGHTNESS, HIGH_SCORES, BALL_SPEED}},      //next_state[0] is the option above current one
        {4,{BALL_SPEED, BRIGHTNESS, HIGH_SCORES}}       //next_state[1] is the option below current one
    };                                                  //next_state[2] is current state
    if(gamepad.get_direction() == N){ _next_state = 0; }        //move arrows up
    else if(gamepad.get_direction() == S){ _next_state = 1; }   //move arrows down
    else{ _next_state = 2; }        //keep arrows the same (default)
    _state = fsm[_state].next_state[_next_state];       //calculate next state
    lcd.printChar('>', 0, fsm[_state].output);
    lcd.printChar('<', 78, fsm[_state].output);         //draw arrows
    return _state;
}

void OptionsEngine::change_brightness(Gamepad &gamepad, N5110 &lcd){
    while(!(gamepad.check_event(gamepad.A_PRESSED))){       //check for user selection
        lcd.clear();
        lcd.printString("Brightness", 12, 0);       //text to display on each line
        lcd.printString("Use L and R to", 0, 3);
        lcd.printString("change", 24, 4);
        lcd.printString("A = confirm", 9, 5);
        lcd.drawRect(10, 12, 63, 8, FILL_TRANSPARENT);      //outside of slider to display the brightness controller
        read_brightness_input(gamepad);
        for(int i = 0; i < _brightness*10; i ++){
            lcd.drawRect(12+6*i, 14, 5, 4, FILL_BLACK);     //draw small rectangles to represent the current brightness
        }
        lcd.setBrightness(_brightness);     //set brightness with N5110 class
        lcd.refresh();
        wait(0.2);      //avoid button bounce with large time between frames
    }
}

void OptionsEngine::read_brightness_input(Gamepad &gamepad){
    if(gamepad.check_event(gamepad.L_PRESSED)){ _brightness -= 0.1f; }      //Use of f to explicitly convert to a float (to fit declaration type in header file).
    if(gamepad.check_event(gamepad.R_PRESSED)){ _brightness += 0.1f; }      //Otherwise 0.1 is implicitly converted to a double (giving warning messages).
    if(_brightness < 0){ _brightness = 0; }     //keep within range of 0 - 1
    if(_brightness > 1){ _brightness = 1; }
}

void OptionsEngine::change_ball_speed(Gamepad &gamepad, N5110 &lcd, Ball &ball){
    while(!(gamepad.check_event(gamepad.A_PRESSED))){       //check for user selection
        lcd.clear();
        lcd.printString("Ball Speed", 12, 0);           //text for each line of display
        lcd.printString("Use L and R to", 0, 3);
        lcd.printString("change", 24, 4);
        lcd.printString("A = confirm", 9, 5);
        lcd.drawRect(10, 12, 63, 8, FILL_TRANSPARENT);
        read_ball_speed_input(gamepad);                 //get values for sensitivity/ball speed
        for(int i = 0; i < _ball_speed; i ++){
            lcd.drawRect(12+6*i, 14, 5, 4, FILL_BLACK); //draw small rectangles in the same manner as brightness control
        }
        ball.set_ball_speed(_ball_speed);
        lcd.refresh();
        wait(0.2);      //avoid button bounce
    }
}

void OptionsEngine::read_ball_speed_input(Gamepad &gamepad){
    if(gamepad.check_event(gamepad.L_PRESSED)){ _ball_speed -= 1; }     //increment on R press, decrement on L press
    if(gamepad.check_event(gamepad.R_PRESSED)){ _ball_speed += 1; }
    if(_ball_speed < 0){ _ball_speed = 0; }     //keep within range 0 - 10
    if(_ball_speed > 10){ _ball_speed = 10; }
}

void OptionsEngine::view_high_scores(Gamepad &gamepad, N5110 &lcd){
    _leaderboard = CLASSIC_MODE;        //initialise to look at classic leaderboard first
    while(!(gamepad.check_event(gamepad.A_PRESSED))){       //check for user selection
        if(gamepad.check_event(gamepad.R_PRESSED)){ _leaderboard = BRICKBREAKER_MODE; }     //use L and R to flick between the two leaderboards
        if(gamepad.check_event(gamepad.L_PRESSED)){ _leaderboard = CLASSIC_MODE; }
        lcd.clear();
        print_high_scores(lcd);
        lcd.refresh();
        wait(0.2);
    }
}

void OptionsEngine::read_classic_high_scores(){
    FILE *fp;       //open file stream
    fp = fopen("/sd/classichighscores.txt", "r");       //classic high scores file in read mode
    
    if(fp == NULL){
        printf("Error: Could not open file");       //check open
    } else {
        int i = 0;
        rewind(fp);         //reset to start of file to start reading
        
        while(fscanf(fp, "%d,%f", &_classic_index[i], &_classic_values[i]) != EOF){     //read each index and value into arrays till end of file
            i++;        //increment counter
        }
        fclose(fp);         //close file stream
    }
}

void OptionsEngine::read_bb_high_scores(){
    FILE *fp;       //open file stream
    fp = fopen("/sd/bbhighscores.txt", "r");        //brickbreaker high scores in read mode
    
    if(fp == NULL){
        printf("Error: Could not open file");       //check open
    } else {
        int i = 0;
        rewind(fp);     //reset to start of file
        
        while(fscanf(fp, "%d,%f", &_bb_index[i], &_bb_values[i]) != EOF){       //read each index and value into array till end of file
            i++;        //increment counter
        }
        fclose(fp);     //close file stream
    }
}

void OptionsEngine::print_high_scores(N5110 &lcd){
    if(_leaderboard == CLASSIC_MODE){       //check which leaderboard to display
        read_classic_high_scores();
        lcd.printString("Classic", 21, 0);      //text for user interface with leaderboard
        lcd.printString("R>", 72, 3);
        char buffer[14];
        for(int i = 0; i < 5; i++){             //iterate for each of 5 high scores
            sprintf(buffer, "%d. %.0fs", _classic_index[i], _classic_values[i]);        //update the buffer for each line of high scores
            lcd.printString(buffer, 0, i + 1);                                          //print the respective high score to screen
        }   
    }
    if(_leaderboard == BRICKBREAKER_MODE){      //check which leaderboard to display
        read_bb_high_scores();
        lcd.printString("Brickbreak", 12, 0);       //text for user interface
        lcd.printString("<L", 72, 3);
        char buffer[14];
        for(int i = 0; i < 5; i++){             //iterate for each of 5 high scores
            sprintf(buffer, "%d. %.0f", _bb_index[i], _bb_values[i]);       //update buffer for each line of high scores
            lcd.printString(buffer, 0, i + 1);                              //print respective high score to screen
        }
    }
}