#ifndef CLASSICENGINE_H
#define CLASSICENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Ball.h"
#include "Map.h"
#include "Pause.h"

/** ClassicEngine Class
@brief Library to power the Classic game mode
@brief Includes feature to check for and update high scores

@author James Cummins

@code

#include "mbed.h"
#include "ClassicEngine.h"

ClassicEngine engine;
Ball ball;
Map map;
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
FXOS8700CQ accelerometer(I2C_SDA,I2C_SCL);
Gamepad gamepad;

int main(){
    int fps = 30;               //set the game mode frames per sec
    classic.init(ball, map);    //initialise the engine before use
    bool collision = false;     //create bool to check whether to continue
    while(!(collision)){
        
    //classic update contains methods for reading and updating map and ball
        classic.classic_update(ball, accelerometer, map);
    //use clear, classic draw and refresh together to re-render the screen each frame
        lcd.clear();
        classic.classic_draw(lcd, map, ball);
        lcd.refresh();
        wait(1/fps);
    //Check for successful completion of game and display appropriate message
        if(classic.finished()){         //check if the game has been completed
    //mode_complete includes feature to update high scores
            classic.mode_complete(lcd, gamepad, fps);
            break;
        }
    //Check for game failure and display appropriate message with option to
    //restart or go back
        if(map.check_wall_collision(gamepad, ball)){    //end the game if collision with wall
            collision = classic.mode_failed(lcd, gamepad, ball, map);   //gives the option to restart the game (and stay in while loop) or quit
    //reset the game mode to the beginning if user wants to play again
            if(!(collision)){ classic.init(ball, map); }
        }
    }
}
@endcode
*/


class ClassicEngine {

public:
    /** 
    * @brief Create an engine object for the Classic game mode
    */
    ClassicEngine();
    /** 
    * @brief Delete a Classic Engine object to free up memory
    */
    ~ClassicEngine();
    /** 
    * @brief Initialise the engine object with the start position of the game map
    * and fix the ball to the centre of the screen. Initialise frames counted to 0.
    * @param ball - Ball object to set the ball's position
    * @param map - Map object to set the map's initial position
    */
    void init(Ball &ball, Map &map);
    /** 
    * @brief Read the accelerometer input and update the map's motion using it.
    * Update the absolute position of the ball in relation to the map.
    * @param ball - Ball object to access ball's position
    * @param accelerometer - FXOS8700CQ object to read user's input motion
    * @param map - Map object to update map
    */
    void classic_update(Ball &ball, FXOS8700CQ &accelerometer, Map &map);
    /** 
    * @brief render the game mode contents on the LCD screen
    * @param lcd - N5110 object to access screen
    * @param map - Map object to draw Map in current position
    * @param ball - Ball object to draw ball in fixed position
    */
    void classic_draw(N5110 &lcd, Map &map, Ball &ball);
    /** 
    * @brief Check if the player has successfully reached the end of the game mode
    * @returns 
    *       true - ball has reached finish line
    *       false - ball has not reached finish line
    */
    bool finished();
    /** 
    * @brief Display the end of game mode screen with the player's completion time
    * @param lcd - N5110 object to draw display on LCD screen
    * @param gamepad - Gamepad object to check for user pushing button
    * @param fps - integer value of game's frames per second
    */
    void mode_complete(N5110 &lcd, Gamepad &gamepad, int fps);
    /** 
    * @brief Display game over screen and Check whether player wants to play again
    * returns
    *       true - player wants to return to start menu
    *       false - player wants to play again
    */
    bool mode_failed(N5110 &lcd, Gamepad &gamepad, Ball &ball, Map &map);
    
private:
    //private member functions
    void write_high_scores(int time_taken);
    void read_high_scores();
    void check_high_score(int time_taken);
    
    //private member variables
    int _index_array[6];
    float _array_of_values[6];
    int _frames;
    Vector2D _ball_coord;   //this gives the current co-ord of the ball in relation to the screen
    Vector2D _map_coord;
    Vector2D _abs_ball_pos;     //this gives the co-ord of ball on map overall ( x between 0 and 500, y between 0 and 200)

};
    
#endif