#ifndef BRICKBREAKERENGINE_H
#define BRICKBREAKERENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "FXOS8700CQ.h"
#include "Ball.h"
#include "Pause.h"
#include "SDFileSystem.h"

#include <cmath>

/** BrickBreakerEngine Class
@brief Library to power the BrickBreaker game mode
@brief Includes feature to check for and write high scores

@author James Cummins

@code

#include "mbed.h"
#include "BrickBreakerEngine.h"
#define RADIUS 3

BrickBreakerEngine engine;   //Constructor to create engine object
AnalogIn randnoise(PTB0);    //Get a random noise signal to seed the random square generator
Gamepad gamepad;
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Ball ball;
FXOS8700CQ accelerometer(I2C_SDA,I2C_SCL);

int main(){
    //Initialise the engine object with the ball's radius and the ball object
    engine.init(RADIUS, ball)
    int total_frames = 30000;
    for (int i = 0; i < total_frames; i++){
        //Not strictly needed as included in the initialiser
        engine.set_score(0);
        lcd.clear();
        ball.read_input(accelerometer);
        ball.update()
        
        //Check for a collision between a square and the ball object
        engine.check_square_collision(randnoise, ball);     //increments score if there's a collision
        
        engine.draw(lcd, ball);
        lcd.refresh();
        
        time_warning();
    }
    //End screen concludes game mode
    engine.end(gamepad, lcd);
    engine.write_high_scores;   //only needs computing once so perform after while loop
}
@endcode
*/

class BrickBreakerEngine {
    
public:
    /** 
    * @brief Create an engine object for the BrickBreaker game mode
    */
    BrickBreakerEngine();
    /** 
    * @brief Delete a brickbreaker engine object to free up memory
    */
    ~BrickBreakerEngine();
    /** 
    * @brief Initialise the engine object
    * Initialises the ball object used and sets the position of the first square
    * to hit with a random number. Gets the ball radius from ball object
    * @param radius - integer for the ball's radius
    * @param ball - object of Ball.h class
    */
    void init(int radius, Ball &ball);
    /** 
    * @brief draw the features of the brickbreaker game mode on the LCD screen
    * @param lcd - N5110 object to interact with the LCD screen
    * @param ball - Ball object to access ball drawing function
    */
    void brickbreaker_draw(N5110 &lcd, Ball &ball);
    /** 
    * @brief set the stored score for the brickbreaker mode to a specific value
    * @param score - integer with desired score value
    */
    void set_score(int score);
    /** 
    * @brief check whether the ball has collided with the square and, if so,
    * randomly generate a new square based on random noise on a disconnected mbed port
    * @param randnoise - AnalogIn reading from a disconnected port
    * @param ball - Ball object to get the ball's position 
    */
    void check_square_collision(AnalogIn &randnoise, Ball &ball);
    /** 
    * @brief Check how long left in the game mode, and display LED sequence to
    * show how long's left
    * @param gamepad - Gamepad object to turn on and off LEDs
    * @param frame - integer number of frames that have passed
    * @param fps - integer number of frames per second the game runs at
    */
    void time_warning(Gamepad &gamepad, int frame, int fps);
    /** 
    * @brief read the current high scores, check if the current score is higher 
    * than any and update the list accordingly
    */
    void write_high_scores();
    /** 
    * @brief display the end of game screen with the player's final score
    * @param gamepad - Gamepad object for use of buttons
    * @param lcd - N5110 object to display end message on screen
    */
    void end(Gamepad &gamepad, N5110 &lcd);

private:
//private functions
    void generate_rand_square(AnalogIn &randnoise); 
    void print_score(N5110 &lcd);
    void read_high_scores();
    void check_high_score();
    
//private variables
    int _index_array[6];
    float _array_of_values[6];  //holds the 5 previous high scores and the score from previous play
    int _ball_radius;
    Vector2D _square_coord;
    int _score;
};
#endif