#ifndef BALL_H
#define BALL_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "FXOS8700CQ.h"

/**Ball Class
@brief Library for creating a ball and moving it round a pre-defined area.
@brief The library also implements drawing the ball on an LCD display

@author James Cummins

@code

#include "mbed.h"
#include "ball.h"
#define RADIUS 3

Ball ball;
FXOS8700CQ accelerometer(I2C_SDA,I2C_SCL);
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad gamepad;

int main(){
    ball.init(RADIUS);  //first must initialise the ball with its radius
    
    //can check what radius we've initialised with...
    int radius = ball.get_radius();
    
    //...and change it if preferred
    radius = 5;
    ball.set_radius(radius);
    
    //Check the current sensitivity of the ball to input motion
    int sensitivity = ball.get_ball_speed();
    
    //and can set it to a different value to cause greater movement for the
    //same degree of tilt in the gamepad
    sensitivity = 3;
    ball.set_ball_speed(sensitivity);
    
    //read_input, update and draw combine to monitor an input and display the
    //subsequent output changes on an LCD display 
    ball.read_input(accelerometer);
    ball.update();
    
    //Can read the ball's position and velocity into Vector2D structs
    Vector2D coords = ball.get_position();
    Vector2D vel = ball.get_velocity();
    
    if(coords.x > 80){ coords.x = 80; }
    if(vel.y > 5){ vel.y = 5; }
    
    //Passing a Vector2D into set_position or set_velocity 
    //moves the ball to a new desired location or at a new speed
    ball.set_position(coords);
    ball.set_velocity(vel);
    
    ball.draw(lcd);
}

@endcode
*/
    

class Ball {

public:
    /** 
    * @brief Create a Ball object
    */
    Ball();
    /** 
    * @brief Delete a ball member to free up memory
    */
    ~Ball();
    
    /** Initialise ball object
    *
    *   Sets the initial position of the ball to the centre of the LCD display.
    *   Sets the initial velocity of the ball to 0 and the sensitivity of the 
    *   ball to 5/10.
    *   @param radius - integer for radius of the ball
    */
    void init(int radius);
    /** 
    * @brief read the input from the accelerometer and convert it to an instantaneous velocity 
    * for the ball.
    * @param accelerometer - FXOS8700CQ object to retrieve acceleromter readings
    */
    void read_input(FXOS8700CQ &accelerometer);
    /** 
    * @brief update the coordinates of the ball within the range of the screen
    */
    void update();
    /** 
    * @brief render the ball onto the LCD screen
    * @param lcd - N5110 object to interface with the LCD
    */
    void draw(N5110 &lcd);
    /** 
    * @brief get the instantaneous velocity of the ball
    * @returns x and y velocities of the ball in a 2D vector
    */
    Vector2D get_velocity();
    /** 
    * @brief get the instantaneous position of the ball
    * @returns x and y positions in a 2D vector. 0 < x < 83 and 0 < y < 47.
    */
    Vector2D get_position();
    /** 
    * @brief get the radius of the ball
    * @returns radius of the ball as an integer
    */
    int get_radius();
    /** 
    * @brief get the speed multiplying factor of the ball. Essentially the sensitivity
    * to the accelerometer input
    * @returns ball speed multiplier as an integer in the range 1-10
    */
    int get_ball_speed();
    /** 
    * @brief Set the sensitivity of the ball's motion to the accelerometer input
    * @param ball_speed - integer in the range 1-10
    */
    void set_ball_speed(int ball_speed);
    /** 
    * @brief set the instantaneous velocities in the x and y directions
    * @param vel - a 2D vector (using the gamepad class) of the desired x and y velocities
    */
    void set_velocity(Vector2D vel);
    /** 
    * @brief set the instantaneous x and y positions of the ball
    * @param pos - a 2D vector (using the gamepad class) of the desired x and y coordinates
    */
    void set_position(Vector2D pos);
    /** 
    * @brief set the radius of the ball
    * @param radius - integer value
    */
    void set_radius(int radius);
    
private:
    int _radius;
    int _x;
    int _y;
    Vector2D _velocity;
    int _ball_speed;
};
#endif