/**
@file main.h
@brief Header file containng function prototypes, defines and global variables.
@brief Ultra-sonic Distance Sensor
@author Jakobi Blackburn
@date 13 March 2015
*/
#ifndef MAIN_H
#define MAIN_H

#define PI 3.14159265359
#define USR_POWERDOWN    (0x104)

#include "mbed.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"
#include "Speaker.h"
#include "N5110.h"
#include "SRF02/SRF02.h"

// --- Inputs ---
/**
@namespace UnitTog
@brief An Interrupt button that toggles the Unit
*/
InterruptIn UnitTog(p14);

/**
@namespace LogTog
@brief An Interrupt button that toggles the Logging Capabilities
*/
InterruptIn LogTog(p16);
/**
@namespace VisTog
@brief An Interrupt button that changes the visual representation of the data
*/
InterruptIn VisTog(p17);
/**
@namespace BuzVol
@brief An Analogue input (Potentiometer) that is used to change the volume of the buzzer
*/
AnalogIn BuzVol(p19);
/**
@namespace BLEDLevel
@brief An Analogue input (Potentiometer) that is used to change the backlight of the LCD screen
*/
AnalogIn BLEDLevel(p20);
/**
@namespace sensor
@brief Ultra-Sonic distance sensor requires SDA and SCL lines
*/
SRF02 sensor(p28,p27);//SDA SCL

// --- Outputs ---
/**
@namespace N5110 LCD
@brief An LCD screen that requires
@brief Pinout VCC,SCE,RST,D/C,MOSI,SCLK,LED
*/     
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);
/**
@namespace Buzzer
@brief A buzzer that can produce different notes dependant on the frquency of 
@brief the signal inputted to it. Used as an auditory alert
*/  
Speaker Buzzer(p18);
/**
@namespace BLED
@brief The backlight of the LCD screen. Brightness can be changed via Pulse Width Modulation
*/  
PwmOut BLED(p26);
/**
@namespace WLED
@brief A LED used as visual warning
*/ 
DigitalOut WLED(p21);
/**
@namespace LLED
@brief A LED used to show Logging is off/on
*/ 
DigitalOut LLED(p29);
/**
@namespace pc
@brief Serial output used mainly for debugging.
@brief Pinout TX, RX
*/ 
Serial pc(USBTX,USBRX);// the tx and rx respectively
/**
@namespace leds
@brief leds used to display error messages and when the file is being written too
*/ 
BusOut leds(LED4,LED3,LED2,LED1);


// ---- Variables ----

float volume=1; /*!< float used to vary the volume */
int distance; /*!< integer that stores the distance */
float unitX=1; /*!< Float that stores the unit multiple if 1 unit is cm, 0.01 = m, 10 = mm */
const char *units ="cm"; /*!< string of characters that stores the unit */
int visual =0;/*!< integer that stores the visual display mode */

// ---- Flags ----
int timerFlag=0; /*!< flag used to trigger the timer ISR   */
int timer2Flag=0; /*!< flag used to trigger the 2nd timer ISR   */
int timer3Flag=0; /*!< flag used to trigger the 3nd timer ISR   */
int setTimeFlag = 0; /*!< flag for ISR for setting time  */
int logButtonFlag = 0; /*!<flag for button toggle  */

// ---- MISC ----
char Dbuffer [20];/*!<buffer used to store distance string */
char buffer[30]; /*!<buffer used to store time and date string */
char rxString[16]; /*!< buffer to store received string   */
LocalFileSystem local("local"); /*!< create local filesystem   */
float array[84];/*!< creates an array that stores data   */
Ticker timer; /*!< timer for first speed  */
Ticker timer2;/*!< timer for second speed */
Ticker timer3;/*!< timer for third speed  */
float PrevArray[84];/*!< creates an array that stores Previous graph data   */
char Gbuffer[30]; /*!<buffer used to store time string */
//  ---- functions ----
/**
Contains the main code
*/
void theMain();
/**
deals with the visual and audiable alerts.
*/
void warnings();  
/**
takes 10 distances and averages them.
@returns The distance between the Sensor and the nearest object infront of it
*/
float getDistance(); 
/**
changes the units cm/m/mm
*/
void unitToggle();
/**
changes the visual display
*/
void visToggle();    
/**
ISR used for the timer
*/
void timerExpired(); 
/**
ISR used for the 2nd timer
*/
void timer2Expired(); 
/**
ISR used for the 3rd timer
*/
void timer3Expired(); 

/**
Displays an error message on the Mbed LEDs
@brief The error message show is up to 14. IF 15 shown it is a ack bit error for the SRF02
@param E - Error Code
@returns flashing Mbed LEDs signifying the error
*/
void error(int E);
/**
ISR for the LOGTOG button Toggling
@brief logging Capabilities
*/
void logToggle();
/**
Takes the time and temperature value
@brief and inserts them into a CSV file
@param char* data
@param float data1
*/
void logging(char* data,float data1);
/**
ISR for the serial port
*/
void serialISR();
/**
function to set the UNIX time
*/
void setTime(); 
/**
Plays a set of frequencies through the buzzer creating a tune
*/
void introTune();
/**
sets the parameters for the first visual mode
*/
void LCDVis0();
/**
sets the parameters for the second visual mode
*/
void LCDVis1();
/**
sets the parameters for the third visual mode
*/
void LCDVis2();
/**
sets the parameters for the fourth visual mode
*/
void LCDVis3();
/**
sets the parameters for the fifth visual mode
*/
void LCDVis4();
/**
sets an image of a car in pixels
*/
void car();
/**
Used for power saving
*/
int semihost_powerdown();

//----Structs----
typedef const struct State STyp; /*!< defines the struct type */
/**
Sets the structure for the units FSM
*/
struct State { 
 float unitMultiple; /*!< float that stores the unit mulitple */
 const char *Unit; /*!<stores the sting value of the unit */
 int nextState[2]; /*!< array of next states */
};

/**
@brief sets the Finite State Machine for the Unit toggle.
*/
STyp fsm[3] = {
 {1,"cm",{0,1}}, /*!< State 0: stays in the current state unless input is recieved */ 
 {0.01,"m",{1,2}},  /*!< State 1: stays in the current state unless input is recieved  */ 
 {10,"mm",{2,0}} /*!< State 2: stays in the current state unless input is recieved  */ 
}; 
int state=0;/*!< sets the initial state */

typedef const struct VState VSTyp; /*!< defines the struct type */
/**
Sets the structure for the visual change FSM
*/
struct VState { 
 int visual; /*!< integer that stores the visual mode */
 int nextState[2]; /*!< array of next states */
};

/**
@brief sets the Finite State Machine for the Visual toggle.
*/
VSTyp Vfsm[5] = {
 {0,{0,1}}, /*!< State 0: stays in the current state unless input is recieved */ 
 {1,{1,2}},  /*!< State 1: stays in the current state unless input is recieved  */ 
 {2,{2,3}}, /*!< State 2: stays in the current state unless input is recieved  */
 {3,{3,4}}, /*!< State 3: stays in the current state unless input is recieved  */
 {4,{4,0}} /*!< State 3: stays in the current state unless input is recieved  */    
}; 
int Vstate=0;/*!< sets the initial state */

#endif


