/*
 * Copyright (C) 2013 Seeed Technology Inc.
 * Copyright (C) 2012 Paulo Marques (pjp.marques@gmail.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of 
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all 
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/* Information about the P9813 protocol obtained from:
 * http://www.seeedstudio.com/wiki/index.php?title=Twig_-_Chainable_RGB_LED
 *
 * HSB to RGB routine adapted from:
 * http://mjijackson.com/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript
 *
 * This library is ported from Arduino to mbed
 */


// --------------------------------------------------------------------------------------

#include "RGBLED.h"

// Forward declaration
float hue2rgb(float p, float q, float t);
PwmOut          PWMR(P0_11);
PwmOut          PWMG(P0_9);
PwmOut          PWMB(P0_10);
// --------------------------------------------------------------------------------------

ChainableLED::ChainableLED(PinName r_pin, PinName g_pin, PinName b_pin,unsigned int number_of_leds) :
    _r_pin(r_pin), _g_pin(g_pin),_b_pin(b_pin)
{
    _num_leds = number_of_leds;
    

    for (uint8_t i=0; i<_num_leds; i++)
        setColorRGB(i, 0, 0, 0);
}

ChainableLED::~ChainableLED()
{
}

// --------------------------------------------------------------------------------------


 
void ChainableLED::sendColor(uint8_t red, uint8_t green, uint8_t blue)
{
    //control the RGB LED by PWM
    PWMR = (float)red/255;
    PWMG = (float)green/255;
    PWMB = (float)blue/255;
    
}

void ChainableLED::setColorRGB(unsigned int led, float red, float green, float blue)
{
    sendColor(0xff-red, 0xff-green, 0xff-blue );    
}

void ChainableLED::setColorHSB(unsigned int led, float hue, float saturation, float brightness)
{
    float r, g, b;
    
    if(saturation == 0.0)
    {
        r = g = b = brightness;
    }
    else
    {
        float q = brightness < 0.5 ? 
            brightness * (1.0 + saturation) : brightness + saturation - brightness * saturation;
        float p = 2.0 * brightness - q;
        r = hue2rgb(p, q, hue + 1.0/3.0);
        g = hue2rgb(p, q, hue);
        b = hue2rgb(p, q, hue - 1.0/3.0);
    }

    setColorRGB(led, (uint8_t)(255.0*r), (uint8_t)(255.0*g), (uint8_t)(255.0*b));
}

// --------------------------------------------------------------------------------------

float hue2rgb(float p, float q, float t)
{
    if (t < 0.0) 
        t += 1.0;
    if(t > 1.0) 
        t -= 1.0;
    if(t < 1.0/6.0) 
        return p + (q - p) * 6.0 * t;
    if(t < 1.0/2.0) 
        return q;
    if(t < 2.0/3.0) 
        return p + (q - p) * (2.0/3.0 - t) * 6.0;

    return p;
}
