/*

Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without restriction, 
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE 
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "mbed.h"
#include "ble/BLE.h"
#include "ChainableLED.h"

/******************************************************
 *                      Macros
 ******************************************************/
#define BLE_UUID_TXRX_SERVICE            0x0000 /**< The UUID of the Nordic UART Service. */
#define BLE_UUID_TX_CHARACTERISTIC       0x0002 /**< The UUID of the TX Characteristic. */
#define BLE_UUIDS_RX_CHARACTERISTIC      0x0003 /**< The UUID of the RX Characteristic. */

#define TXRX_BUF_LEN                     5

#define LOCAL_NAME                       "RGB-L"
#define BUTTON_PIN                       P0_8//D3
#define DEVICE_ID                        0xFF
#define LIGHT_PIN                        P0_5//A3
#define VOL_PIN                          P0_4//A4

/******************************************************
 *               Variable Definitions
 ******************************************************/

InterruptIn button(BUTTON_PIN);

BLEDevice       ble;

ChainableLED    leds(P0_11, P0_9, P0_10,1);

AnalogIn       light_sen(LIGHT_PIN);
AnalogIn       vin(VOL_PIN);

DigitalOut myled(P0_19);

static uint8_t rgb_data[5]={0xA5,0xFF,0xFF, 0xFF, 0xFF};
static unsigned char rgb_status=0;

static uint8_t light_flag = 0;

// The Nordic UART Service
static const uint8_t uart_base_uuid[]       = {0x5A, 0x2D, 0x3B, 0xF8, 0xF0, 0xBC, 0x11, 0xE5, 0x9C, 0xE9, 0x5E, 0x55, 0x17, 0x50, 0x7E, 0x66}; 
static const uint8_t uart_tx_uuid[]         = {0x5A, 0x2D, 0x40, 0xEE, 0xF0, 0xBC, 0x11, 0xE5, 0x9C, 0xE9, 0x5E, 0x55, 0x17, 0x50, 0x7E, 0x66};
static const uint8_t uart_rx_uuid[]         = {0x5A, 0x2D, 0x42, 0x9C, 0xF0, 0xBC, 0x11, 0xE5, 0x9C, 0xE9, 0x5E, 0x55, 0x17, 0x50, 0x7E, 0x66};
static const uint8_t uart_light_uuid[]      = {0x5A, 0x2D, 0x43, 0x78, 0xF0, 0xBC, 0x11, 0xE5, 0x9C, 0xE9, 0x5E, 0x55, 0x17, 0x50, 0x7E, 0x66};
static const uint8_t uart_base_uuid_rev[]   = {0x66, 0x7E, 0x50, 0x17, 0x55, 0x5E, 0xE9, 0x9C, 0xE5, 0x11, 0xBC, 0xF0, 0xF8, 0x3B, 0x2D, 0x5A};


uint8_t txPayload[TXRX_BUF_LEN] = {0,0,0,0,0};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,0,0,0,0};
uint8_t lightPayload[TXRX_BUF_LEN] = {0,0,0,0,0};
uint8_t vinPayload[TXRX_BUF_LEN] = {0,0,0,0,0};


GattCharacteristic  txCharacteristic (uart_tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE );

GattCharacteristic  rxCharacteristic (uart_rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

GattCharacteristic  lightCharacteristic (uart_light_uuid, lightPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY| GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

GattCharacteristic  batCharacteristic (0x2A19, vinPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY|GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

GattCharacteristic *uartChars[] = {&txCharacteristic, &rxCharacteristic,&lightCharacteristic,&batCharacteristic};

GattService         uartService(uart_base_uuid, uartChars, sizeof(uartChars) / sizeof(GattCharacteristic *));



 /******************************************************
 *               Function Definitions
 ******************************************************/

static void connectCallBack(const Gap::ConnectionCallbackParams_t *params)
{
    leds.setColorRGB(0, 0XFF, 0XFF, 0XFF);  
}


void disconnectionCallBack(const Gap::DisconnectionCallbackParams_t *params)
{
    leds.setColorRGB(0, 0, 0, 0);  
    ble.gap().startAdvertising();
}

void timeoutCallBack()
{
    
}



// GATT call back handle
void writtenHandle(const GattWriteCallbackParams *Handler)
{
    uint8_t buf[TXRX_BUF_LEN];
    uint16_t bytesRead;

    //pc.printf("Write Handle : \n");
 
    if (Handler->handle == txCharacteristic.getValueAttribute().getHandle())
    {
        ble.readCharacteristicValue(txCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(txPayload, 0, TXRX_BUF_LEN);
        memcpy(txPayload, buf, TXRX_BUF_LEN);  
        
        if( (buf[2] != 0x00) || (buf[3] != 0x00) || (buf[4] != 0x00) )  
        {
            rgb_status = 1;
            leds.setColorRGB(0, buf[2], buf[3], buf[4]);  
            rgb_data[0] = 0xA5;
            rgb_data[1] = 0xff;  
            rgb_data[2] = buf[2];
            rgb_data[3] = buf[3];
            rgb_data[4] = buf[4];
            ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 5, true);   
        }
        else
        {
            rgb_status = 0;
            leds.setColorRGB(0, 0, 0, 0);   
            ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 5, true);      
        }
    }
}

//switch on/off the rgb led and notify the value
void button_handle()
{
    uint8_t buf[5];
    buf[0] = 0xA5;
    buf[1] = 0xff;
    if(rgb_status == 0)  
    {
        rgb_status = 1;
        leds.setColorRGB(0, rgb_data[2], rgb_data[3], rgb_data[4]);       
        buf[2] = rgb_data[2];
        buf[3] = rgb_data[3];
        buf[4] = rgb_data[4];
    }
    else 
    {
        rgb_status = 0;
        leds.setColorRGB(0, 0, 0, 0);
        buf[2] = 0x00;
        buf[3] = 0x00;
        buf[4] = 0x00;
    }
    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 5);
    
}

//check the light value and battery value 
void m_status_check_handle(void)
{
    uint16_t light_value, vin_val=0;
    uint8_t light_buf[2] = {0,0};
    
    uint8_t vin_buf[2] = {0,0};
    
    light_value = light_sen.read_u16();
    
    light_buf[0] = (light_value&0xFF00)>>8;
    light_buf[1] = light_value&0x00FF;
    
    vin_val = vin.read_u16();
    
    vin_buf[0] = (vin_val&0xFF00)>>8;
    vin_buf[1] = vin_val&0x00FF;
    
    
    ble.updateCharacteristicValue(batCharacteristic.getValueAttribute().getHandle(),vin_buf , 2,true); 
    if(vin_val<0x230)
    {
        ble.updateCharacteristicValue(batCharacteristic.getValueAttribute().getHandle(),vin_buf , 2); 
    }
    //trigger value is 0x100,you can change it base your environment
    if((light_value<0x100)&&(light_flag==0))
    {
        ble.updateCharacteristicValue(lightCharacteristic.getValueAttribute().getHandle(),light_buf , 2);   
        light_flag = 1;
    }
    if((light_value>0x100)&&(light_flag==1))
    {
        ble.updateCharacteristicValue(lightCharacteristic.getValueAttribute().getHandle(),light_buf , 2);   
        light_flag = 0;
    }
    
}

int main() {
    
      
    uint8_t buf[5]    = {0xA5,0xFF,0x00, 0x00, 0x00};
    
    rgb_status = 0;
    leds.setColorRGB(0, 0, 0, 0);  
        
    button.mode(PullUp);
    button.fall(&button_handle);
    Ticker ticker;
    ticker.attach_us(m_status_check_handle, 200000);
       
    ble.init();

    ble.onConnection(connectCallBack);
    ble.onDisconnection(disconnectionCallBack);
    ble.onDataWritten(writtenHandle);
      

    // setup adv_data and srp_data
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)LOCAL_NAME, sizeof(LOCAL_NAME) - 1);   
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)uart_base_uuid_rev, sizeof(uart_base_uuid_rev));
                                                            
    // add service
    ble.addService(uartService);
    // set device name
    ble.setDeviceName((const uint8_t *)LOCAL_NAME);
    // set appearance
    //ble.setAppearance(BLE_APPEARANCE_UNKNOWN);
    // set tx power,valid values are -40, -20, -16, -12, -8, -4, 0, 4
    ble.setTxPower(4);
    // set adv_type
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    // set adv_interval, 100ms in multiples of 0.625ms.
    ble.setAdvertisingInterval(160);
    // set adv_timeout, in seconds
    ble.setAdvertisingTimeout(0);
    // ger BLE stack version
    //pc.printf( ble.getVersion() );
    // set RGB default status
    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 5, true);   
    // start advertising
    ble.startAdvertising();

    memset(rgb_data, 0xFF, 5);
    myled = 1;   
    
    while(1)
    {
     
        ble.waitForEvent(); 

    }

    
}































