#ifndef TFT_22_ILI9225_h
#define TFT_22_ILI9225_h
#include "mbed.h"

#define FONT_LENGTH          0
#define FONT_HORZ            1
#define FONT_VERT            2
#define FONT_BYTES_VERT      3

#define ILI9225_CHARS_PER_LINE    36   

/* ILI9225 screen size */
#define ILI9225_LCD_WIDTH   176
#define ILI9225_LCD_HEIGHT  220

#define ILI9225_PORTRAIT_L  0   // SD card at the left side
#define ILI9225_LANDSCAPE_B 1   // SD card at the bottom side
#define ILI9225_PORTRAIT_R  2   // SD card at the right side
#define ILI9225_LANDSCAPE_T 3   // SD card at the top side

/* ILI9225 LCD Registers */
#define ILI9225_DRIVER_OUTPUT_CTRL      (0x01u)  // Driver Output Control
#define ILI9225_LCD_AC_DRIVING_CTRL     (0x02u)  // LCD AC Driving Control
#define ILI9225_ENTRY_MODE              (0x03u)  // Entry Mode
#define ILI9225_DISP_CTRL1              (0x07u)  // Display Control 1
#define ILI9225_BLANK_PERIOD_CTRL1      (0x08u)  // Blank Period Control
#define ILI9225_FRAME_CYCLE_CTRL        (0x0Bu)  // Frame Cycle Control
#define ILI9225_INTERFACE_CTRL          (0x0Cu)  // Interface Control
#define ILI9225_OSC_CTRL                (0x0Fu)  // Osc Control
#define ILI9225_POWER_CTRL1             (0x10u)  // Power Control 1
#define ILI9225_POWER_CTRL2             (0x11u)  // Power Control 2
#define ILI9225_POWER_CTRL3             (0x12u)  // Power Control 3
#define ILI9225_POWER_CTRL4             (0x13u)  // Power Control 4
#define ILI9225_POWER_CTRL5             (0x14u)  // Power Control 5
#define ILI9225_VCI_RECYCLING           (0x15u)  // VCI Recycling
#define ILI9225_RAM_ADDR_SET1           (0x20u)  // Horizontal GRAM Address Set
#define ILI9225_RAM_ADDR_SET2           (0x21u)  // Vertical GRAM Address Set
#define ILI9225_GRAM_DATA_REG           (0x22u)  // GRAM Data Register
#define ILI9225_GATE_SCAN_CTRL          (0x30u)  // Gate Scan Control Register
#define ILI9225_VERTICAL_SCROLL_CTRL1   (0x31u)  // Vertical Scroll Control 1 Register
#define ILI9225_VERTICAL_SCROLL_CTRL2   (0x32u)  // Vertical Scroll Control 2 Register
#define ILI9225_VERTICAL_SCROLL_CTRL3   (0x33u)  // Vertical Scroll Control 3 Register
#define ILI9225_PARTIAL_DRIVING_POS1    (0x34u)  // Partial Driving Position 1 Register
#define ILI9225_PARTIAL_DRIVING_POS2    (0x35u)  // Partial Driving Position 2 Register
#define ILI9225_HORIZONTAL_WINDOW_ADDR1 (0x36u)  // Horizontal Address Start Position
#define ILI9225_HORIZONTAL_WINDOW_ADDR2 (0x37u)  // Horizontal Address End Position
#define ILI9225_VERTICAL_WINDOW_ADDR1   (0x38u)  // Vertical Address Start Position
#define ILI9225_VERTICAL_WINDOW_ADDR2   (0x39u)  // Vertical Address End Position
#define ILI9225_GAMMA_CTRL1             (0x50u)  // Gamma Control 1
#define ILI9225_GAMMA_CTRL2             (0x51u)  // Gamma Control 2
#define ILI9225_GAMMA_CTRL3             (0x52u)  // Gamma Control 3
#define ILI9225_GAMMA_CTRL4             (0x53u)  // Gamma Control 4
#define ILI9225_GAMMA_CTRL5             (0x54u)  // Gamma Control 5
#define ILI9225_GAMMA_CTRL6             (0x55u)  // Gamma Control 6
#define ILI9225_GAMMA_CTRL7             (0x56u)  // Gamma Control 7
#define ILI9225_GAMMA_CTRL8             (0x57u)  // Gamma Control 8
#define ILI9225_GAMMA_CTRL9             (0x58u)  // Gamma Control 9
#define ILI9225_GAMMA_CTRL10            (0x59u)  // Gamma Control 10

#define ILI9225C_INVOFF  0x20
#define ILI9225C_INVON   0x21


/* RGB 24-bits color table definition (RGB888). */
#define RGB888_RGB565(color) ((((color) >> 19) & 0x1f) << 11) | ((((color) >> 10) & 0x3f) << 5) | (((color) >> 3) & 0x1f)

#define COLOR_BLACK          RGB888_RGB565(0x000000u)
#define COLOR_WHITE          RGB888_RGB565(0xFFFFFFu)
#define COLOR_BLUE           RGB888_RGB565(0x0000FFu)
#define COLOR_GREEN          RGB888_RGB565(0x00FF00u)
#define COLOR_RED            RGB888_RGB565(0xFF0000u)
#define COLOR_NAVY           RGB888_RGB565(0x000080u)
#define COLOR_DARKBLUE       RGB888_RGB565(0x00008Bu)
#define COLOR_DARKGREEN      RGB888_RGB565(0x006400u)
#define COLOR_DARKCYAN       RGB888_RGB565(0x008B8Bu)
#define COLOR_CYAN           RGB888_RGB565(0x00FFFFu)
#define COLOR_TURQUOISE      RGB888_RGB565(0x40E0D0u)
#define COLOR_INDIGO         RGB888_RGB565(0x4B0082u)
#define COLOR_DARKRED        RGB888_RGB565(0x800000u)
#define COLOR_OLIVE          RGB888_RGB565(0x808000u)
#define COLOR_GRAY           RGB888_RGB565(0x808080u)
#define COLOR_SKYBLUE        RGB888_RGB565(0x87CEEBu)
#define COLOR_BLUEVIOLET     RGB888_RGB565(0x8A2BE2u)
#define COLOR_LIGHTGREEN     RGB888_RGB565(0x90EE90u)
#define COLOR_DARKVIOLET     RGB888_RGB565(0x9400D3u)
#define COLOR_YELLOWGREEN    RGB888_RGB565(0x9ACD32u)
#define COLOR_BROWN          RGB888_RGB565(0xA52A2Au)
#define COLOR_DARKGRAY       RGB888_RGB565(0xA9A9A9u)
#define COLOR_SIENNA         RGB888_RGB565(0xA0522Du)
#define COLOR_LIGHTBLUE      RGB888_RGB565(0xADD8E6u)
#define COLOR_GREENYELLOW    RGB888_RGB565(0xADFF2Fu)
#define COLOR_SILVER         RGB888_RGB565(0xC0C0C0u)
#define COLOR_LIGHTGREY      RGB888_RGB565(0xD3D3D3u)
#define COLOR_LIGHTCYAN      RGB888_RGB565(0xE0FFFFu)
#define COLOR_VIOLET         RGB888_RGB565(0xEE82EEu)
#define COLOR_AZUR           RGB888_RGB565(0xF0FFFFu)
#define COLOR_BEIGE          RGB888_RGB565(0xF5F5DCu)
#define COLOR_MAGENTA        RGB888_RGB565(0xFF00FFu)
#define COLOR_TOMATO         RGB888_RGB565(0xFF6347u)
#define COLOR_GOLD           RGB888_RGB565(0xFFD700u)
#define COLOR_ORANGE         RGB888_RGB565(0xFFA500u)
#define COLOR_SNOW           RGB888_RGB565(0xFFFAFAu)
#define COLOR_YELLOW         RGB888_RGB565(0xFFFF00u)

/// Main and core class
class TFT_22_ILI9225 : public Stream
{

public:

    /** Constructor
     *
     */
    TFT_22_ILI9225(PinName sdi, PinName clk, PinName cs, PinName rs, PinName rst, PinName led, const char *name);

    /** Initialization
     *
     */
    void init(void);

    /** Clear the screen
     *
     */
    void cls(void);

    /** Fill the screen with color
     *
     * @param  color
     *
     */
    void fill(uint16_t color);

    /** Invert screen
     *
     * @param  flag true to invert, false for normal screen
     *
     */
    void invert(bool flag);

    /** Set backlight brightness
     *
     * @param brightness
     *
     */
    void setBacklight(double brightness);
    
    /** Switch backlight off
     *
     */
    void setBacklightOff(void);
    
    /** Switch backlight on
     *
     */
    void setBacklightOn(void);

    /** Switch display on or off
     *
     * @param  flag true=on, false=off
     *
     */
    void setDisplay(bool flag);

    /** Set orientation
     *
     * @param  orientation orientation:
     * 0=ILI9225_PORTRAIT_L  : SD card at the left side
     * 1=ILI9225_LANDSCAPE_B : SD card at the bottom side
     * 2=ILI9225_PORTRAIT_R  : SD card at the right side
     * 3=ILI9225_LANDSCAPE_T : SD card at the top side
     *
     */
    void setOrientation(uint8_t orientation);

    /** Get orientation
     *
     * @return orientation orientation, 0=portrait, 1=right rotated landscape, 2=reverse portrait, 3=left rotated landscape
     *
     */
    uint8_t getOrientation(void);


    // Graphics functions

    /** Draw pixel
     *
     * @param  x1 point coordinate, x-axis
     * @param  y1 point coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void pixel(uint16_t x1, uint16_t y1, uint16_t color);

    /** Draw line, rectangle coordinates
     *
     * @param  x1 top left coordinate, x-axis
     * @param  y1 top left coordinate, y-axis
     * @param  x2 bottom right coordinate, x-axis
     * @param  y2 bottom right coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void line(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);

    /** Draw hline, rectangle coordinates
     *
     * @param  x1 top left coordinate, x-axis
     * @param  x2 bottom right coordinate, x-axis
     * @param  y bottom right coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void hline(uint16_t x1, uint16_t x2, uint16_t y, uint16_t color);

    /** Draw vline, rectangle coordinates
     *
     * @param  x top left coordinate, x-axis
     * @param  y1 top left coordinate, y-axis
     * @param  y2 bottom right coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void vline(uint16_t x, uint16_t y1, uint16_t y2, uint16_t color);

    /** Draw rectangle, rectangle coordinates
     *
     * @param  x1 top left coordinate, x-axis
     * @param  y1 top left coordinate, y-axis
     * @param  x2 bottom right coordinate, x-axis
     * @param  y2 bottom right coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void rect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);

    /** Draw solid rectangle, rectangle coordinates
     *
     * @param  x1 top left coordinate, x-axis
     * @param  y1 top left coordinate, y-axis
     * @param  x2 bottom right coordinate, x-axis
     * @param  y2 bottom right coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void fillrect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);

    /** Draw circle
     *
     * @param  x0 center, point coordinate, x-axis
     * @param  y0 center, point coordinate, y-axis
     * @param  radius radius
     * @param  color 16-bit color
     *
     */
    void circle(uint16_t x0, uint16_t y0, uint16_t radius, uint16_t color);

    /** Draw solid circle
     *
     * @param  x0 center, point coordinate, x-axis
     * @param  y0 center, point coordinate, y-axis
     * @param  radius radius
     * @param  color 16-bit color
     *
     */
    void fillcircle(uint8_t x0, uint8_t y0, uint8_t radius, uint16_t color);

    /** Draw triangle, triangle coordinates
     *
     * @param  x1 corner 1 coordinate, x-axis
     * @param  y1 corner 1 coordinate, y-axis
     * @param  x2 corner 2 coordinate, x-axis
     * @param  y2 corner 2 coordinate, y-axis
     * @param  x3 corner 3 coordinate, x-axis
     * @param  y3 corner 3 coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void triangle(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t x3, uint16_t y3, uint16_t color);

    /** Draw solid triangle, triangle coordinates
     *
     * @param  x1 corner 1 coordinate, x-axis
     * @param  y1 corner 1 coordinate, y-axis
     * @param  x2 corner 2 coordinate, x-axis
     * @param  y2 corner 2 coordinate, y-axis
     * @param  x3 corner 3 coordinate, x-axis
     * @param  y3 corner 3 coordinate, y-axis
     * @param  color 16-bit color
     *
     */
    void filltriangle(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t x3, uint16_t y3, uint16_t color);

    /** Draw line/filled rectangle, rectangle coordinates
     *
     * @param  x1 top left coordinate, x-axis
     * @param  y1 top left coordinate, y-axis
     * @param  x2 bottom right coordinate, x-axis
     * @param  y2 bottom right coordinate, y-axis
     * @param  rad defines the redius of the circle
     * @param  fill fill yes or no
     * @param  color 16-bit color
     *
     */
    void roundrect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t rad, bool fill, uint16_t color);

    /** Screen size, x-axis
     *
     * @return horizontal size of the screen, in pixels
     * @note   219 means 0..219 coordinates (decimal)
     *
     */
    uint16_t maxX(void);

    /** Screen size, y-axis
     *
     * @return vertical size of the screen, in pixels
     * @note   175 means 0..175 coordinates (decimal)
     *
     */
    uint16_t maxY(void);

    /** Screen size, x-axis
     *
     * @return horizontal size of the screen, in pixels
     * @note   220 means 220 pixels and thus 0..219 coordinates (decimal)
     *
     */
    uint16_t width(void);

    /** Screen size, y-axis
     *
     * @return vertical size of the screen, in pixels
     * @note   176 means 176 pixels and thus 0..175 coordinates (decimal)
     *
     */
    uint16_t height(void);

    /** Calculate 16-bit color from 8-bit Red-Green-Blue components
     *
     * @param  red red component, 0x00..0xff
     * @param  green green component, 0x00..0xff
     * @param  blue blue component, 0x00..0xff
     * @return 16-bit color
     *
     */
    uint16_t setColor(uint8_t red, uint8_t green, uint8_t blue);

    /** Calculate 8-bit Red-Green-Blue components from 16-bit color
     *
     * @param  rgb 16-bit color
     * @param  red red component, 0x00..0xff
     * @param  green green component, 0x00..0xff
     * @param  blue blue component, 0x00..0xff
     *
     */
    void splitColor(uint16_t rgb, uint8_t &red, uint8_t &green, uint8_t &blue);


    // Text functions

    /** Set current font
     *
     * @param  font Font name
     *
     */
    void setFont(uint8_t* font);

    /** Font size, x-axis
     *
     * @return horizontal size of current font, in pixels
     *
     */
    uint8_t fontX(void);

    /** Font size, y-axis
     *
     * @return vertical size of current font, in pixels
     *
     */
    uint8_t fontY(void);

    /** put a char on the screen
     *
     * @param value char to print
     * @returns printed char
     *
     */
    virtual int putc(int value);

    /** draw a character on given position out of the active font to the TFT
     *
     * @param x x-position of char (top left) 
     * @param y y-position
     * @param c char to print
     *
     */    
    void character(int x, int y, int c);

    /** get string width
     *
     * @param string 
     * @return string width in pixels
     *
     */    
    uint16_t getStringWidth(char * s);

    /** Set text foreground color
     *
     * @param color
     *
     */   
    void foreground(uint16_t color);

    /** Set text background color
     *
     * @param color
     *
     */   
    void background(uint16_t color);

    /** Set grapical cursor position
     *
     * @param x
     * @param y
     *
     */   
    void locate(int x, int y);
    
    /** Set text cursor position
     *
     * @param x
     * @param y
     *
     */   
    void gotoxy(int x, int y);
    
    /** Set text cursor at 0,0
     *
     */   
    void home(void);

    /** Set spacing in pixels between text lines
     *
     * @param line_spacing spacing in pixels between text lines
     *
     */   
    void linespacing(int line_spacing);
    
    /** calculate the max number of char in a line
     *
     * @returns max columns
     * depends on actual font size
     *
     */    
    virtual int columns(void);
    
    /** calculate the max number of columns
     *
     * @returns max column
     * depends on actual font size
     *
     */   
    virtual int rows(void);

    /** Make an ascii string from an unicode string
     *
     * @param uni_str 
     * @param ascii_str 
     *
     */   
    void unicode2ascii(char *uni_str, char *ascii_str); 

    /** Claim standard output
     *
     */
    virtual bool claim (FILE *stream);


private:

    SPI        _spi;
    DigitalOut _rst;
    DigitalOut _rs;
    DigitalOut _cs;
    PwmOut     _led;

    float _brightness;
    uint16_t _entryMode;
    uint16_t _maxX, _maxY;
    uint8_t _orientation;

    // for claim
    uint16_t _column;
    uint16_t _row;
    char *_path;

    unsigned char* font;
    unsigned int char_x;
    unsigned int char_y;
    unsigned int char_line_spacing;

    // colors
    uint16_t _foreground;
    uint16_t _background;


    // Private functions

    /** Swap two values
     *
     * @param  a b
     * @param  a b
     *
     */
    void _swap(uint16_t &a, uint16_t &b);

    /** Set window
     *
     * @param  x0
     * @param  y0
     * @param  x1
     * @param  y1
     *
     */
    void _setWindow(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1);

    /** Maximize window to entire screen
     *
     */
    void _setWindowMax(void);

    /** Set coordinates
     *
     * @param  x
     * @param  y
     *
     */
    void _orientCoordinates(uint16_t &x, uint16_t &y);

    /** Write register
     *
     * @param  uint16_t register
     * @param  uint16_t data
     *
     */
    void _writeRegister(uint16_t reg, uint16_t data);

    /** Write a command
     *
     * @param  HI
     * @param  LO
     *
     */
    void _writeCommand(uint8_t HI, uint8_t LO);

    /** Start writing data
     *
     */
    void _startData(void);

    /** Write data
     *
     * @param  uint16_t data
     *
     */
    void _writeData(uint16_t data);

    /** End writing data
     *
     */
    void _endData(void);


protected:

    //used by printf - supply a new _putc virtual function for the new device
    virtual int _putc(int c) {
        putc(c); //your new LCD put to print an ASCII character on LCD
        return 0;
    };
    //assuming no reads from LCD
    virtual int _getc() {
        return -1;
    }

};

#endif
