/*
Ported from arduino to mbed platform by Arman Safikhani 31-08-2015
Jack: Many thanks Arman!

Changed a lot of things by Jack Berkhout 23-11-2016

Used for 2.2" Serial SPI TFT Color LCD Module for Arduino
176*220 resolution / 11-pin / ILI9225 IC driver
https://www.fasttech.com/product/3842400-2-2-serial-spi-tft-color-lcd-module-for-arduino

- Some code removed, some added, some changed
- Now uses the same fonts as the SPI_TFT_ILI9341 Library (I have many, and a html/php font editor for that)
- Text line interspacing can be set
- The led backlight is using a PWM output, so it can be adjusted in brightness (from 0.0f = off to 1.0f = full brightness)
- Added formatted printf
- Added claim as standard output on stream
- Orientation definitions are now related to the SD card slot location (front view)
- Fast line functions hline and vline added, also used by outline reactangles, with drastical performance improvement (15 times)
- Performance boost for filled shapes using new vline and hline functions
- Get getStringWidth added
*/

#include "SPI_TFT_ILI9225.h"

// Constructor when using software SPI.  All output pins are configurable.
TFT_22_ILI9225::TFT_22_ILI9225(PinName sdi, PinName clk, PinName cs, PinName rs, PinName rst, PinName led, const char *name) :
    _spi(sdi, NC, clk), _cs(cs), _rs(rs), _rst(rst), _led(led), Stream(name)
{
    _spi.frequency(24000000); 
    _spi.format(8, 0);

    // PWM output to control backlight
    _led.period_ms(10);

    // --- claim ---
    _row = 0;
    _column = 0;
    if (name == NULL) {
        _path = NULL;
    } else {
        _path = new char[strlen(name) + 2];
        sprintf(_path, "/%s", name);
    }
    // --- /claim ---
    
    init();
}

void TFT_22_ILI9225::init()
{
    // Turn off backlight
    setBacklight(0.0f);

    // Initialization Code
    _rst = 0;// Pull the reset pin low to reset ILI9225
    wait_ms(10);
    _rst = 1;// Pull the reset pin high to release the ILI9225C from the reset status
    wait_ms(50);

    /* Start Initial Sequence */
    /* Set SS bit and direction output from S528 to S1 */
    _writeRegister(ILI9225_POWER_CTRL1, 0x0000); // Set SAP,DSTB,STB
    _writeRegister(ILI9225_POWER_CTRL2, 0x0000); // Set APON,PON,AON,VCI1EN,VC
    _writeRegister(ILI9225_POWER_CTRL3, 0x0000); // Set BT,DC1,DC2,DC3
    _writeRegister(ILI9225_POWER_CTRL4, 0x0000); // Set GVDD
    _writeRegister(ILI9225_POWER_CTRL5, 0x0000); // Set VCOMH/VCOML voltage
    wait_ms(40);

    // Power-on sequence
    _writeRegister(ILI9225_POWER_CTRL2, 0x0018); // Set APON,PON,AON,VCI1EN,VC
    _writeRegister(ILI9225_POWER_CTRL3, 0x6121); // Set BT,DC1,DC2,DC3
    _writeRegister(ILI9225_POWER_CTRL4, 0x006F); // Set GVDD   /*007F 0088 */
    _writeRegister(ILI9225_POWER_CTRL5, 0x495F); // Set VCOMH/VCOML voltage
    _writeRegister(ILI9225_POWER_CTRL1, 0x0800); // Set SAP,DSTB,STB
    wait_ms(10);
    _writeRegister(ILI9225_POWER_CTRL2, 0x103B); // Set APON,PON,AON,VCI1EN,VC
    wait_ms(50);

    _writeRegister(ILI9225_DRIVER_OUTPUT_CTRL, 0x011C); // set the display line number and display direction
    _writeRegister(ILI9225_LCD_AC_DRIVING_CTRL, 0x0100); // set 1 line inversion
    _writeRegister(ILI9225_ENTRY_MODE, 0x1030); // set GRAM write direction and BGR=1.
    _writeRegister(ILI9225_DISP_CTRL1, 0x0000); // Display off
    _writeRegister(ILI9225_BLANK_PERIOD_CTRL1, 0x0808); // set the back porch and front porch
    _writeRegister(ILI9225_FRAME_CYCLE_CTRL, 0x1100); // set the clocks number per line
    _writeRegister(ILI9225_INTERFACE_CTRL, 0x0000); // CPU interface
    _writeRegister(ILI9225_OSC_CTRL, 0x0D01); // Set Osc  /*0e01*/
    _writeRegister(ILI9225_VCI_RECYCLING, 0x0020); // Set VCI recycling
    _writeRegister(ILI9225_RAM_ADDR_SET1, 0x0000); // RAM Address
    _writeRegister(ILI9225_RAM_ADDR_SET2, 0x0000); // RAM Address

                                                   /* Set GRAM area */
    _writeRegister(ILI9225_GATE_SCAN_CTRL, 0x0000);
    _writeRegister(ILI9225_VERTICAL_SCROLL_CTRL1, 0x00DB);
    _writeRegister(ILI9225_VERTICAL_SCROLL_CTRL2, 0x0000);
    _writeRegister(ILI9225_VERTICAL_SCROLL_CTRL3, 0x0000);
    _writeRegister(ILI9225_PARTIAL_DRIVING_POS1, 0x00DB);
    _writeRegister(ILI9225_PARTIAL_DRIVING_POS2, 0x0000);
    _writeRegister(ILI9225_HORIZONTAL_WINDOW_ADDR1, 0x00AF);
    _writeRegister(ILI9225_HORIZONTAL_WINDOW_ADDR2, 0x0000);
    _writeRegister(ILI9225_VERTICAL_WINDOW_ADDR1, 0x00DB);
    _writeRegister(ILI9225_VERTICAL_WINDOW_ADDR2, 0x0000);

    /* Set GAMMA curve */
    _writeRegister(ILI9225_GAMMA_CTRL1, 0x0000);
    _writeRegister(ILI9225_GAMMA_CTRL2, 0x0808);
    _writeRegister(ILI9225_GAMMA_CTRL3, 0x080A);
    _writeRegister(ILI9225_GAMMA_CTRL4, 0x000A);
    _writeRegister(ILI9225_GAMMA_CTRL5, 0x0A08);
    _writeRegister(ILI9225_GAMMA_CTRL6, 0x0808);
    _writeRegister(ILI9225_GAMMA_CTRL7, 0x0000);
    _writeRegister(ILI9225_GAMMA_CTRL8, 0x0A00);
    _writeRegister(ILI9225_GAMMA_CTRL9, 0x0710);
    _writeRegister(ILI9225_GAMMA_CTRL10, 0x0710);

    _writeRegister(ILI9225_DISP_CTRL1, 0x0012);
    wait_ms(50);
    _writeRegister(ILI9225_DISP_CTRL1, 0x1017);

    cls();
    linespacing(0);
    setBacklight(1.0f);
    setOrientation(ILI9225_LANDSCAPE_B);
    foreground(COLOR_WHITE);
    background(COLOR_BLACK);
}

void TFT_22_ILI9225::cls()
{
    setBacklightOff();
    _setWindowMax();
    _rs = 1;
    _cs = 0;
    for (uint16_t i = width() * height(); i > 0; i--) {
        _spi.write(0x00);
        _spi.write(0x00);
    }
    _cs = 1;
    setBacklightOn();
    gotoxy(0, 0);
}

void TFT_22_ILI9225::fill(uint16_t color)
{
    fillrect(0, 0, maxX(), maxY(), color);
}

void TFT_22_ILI9225::invert(bool flag)
{
    _writeCommand(0x00, flag ? ILI9225C_INVON : ILI9225C_INVOFF);
}

void TFT_22_ILI9225::setBacklight(double brightness)
{
    // PWM output to control backlight
    _brightness = brightness;
    _led.write(pow(brightness, 2)); // power(x, 2): For the eye better brightness response
}

void TFT_22_ILI9225::setBacklightOff(void)
{
    _led.write(0.0f);
}

void TFT_22_ILI9225::setBacklightOn(void)
{
    setBacklight(_brightness);
}

void TFT_22_ILI9225::setDisplay(bool flag)
{
    if (flag) {
        _writeRegister(0x00ff, 0x0000);
        _writeRegister(ILI9225_POWER_CTRL1, 0x0000);
        wait_ms(50);
        _writeRegister(ILI9225_DISP_CTRL1, 0x1017);
        wait_ms(200);
    }
    else {
        _writeRegister(0x00ff, 0x0000);
        _writeRegister(ILI9225_DISP_CTRL1, 0x0000);
        wait_ms(50);
        _writeRegister(ILI9225_POWER_CTRL1, 0x0003);
        wait_ms(200);
    }
}

void TFT_22_ILI9225::setOrientation(uint8_t orientation)
{
    _orientation = orientation % 4;
    // Entry Mode (R03h)
    // 0x1000 AM = 0: Horizontal, I/D[1:0] = 00: Horizontal: decrement, Vertical: decrement - ILI9225_PORTRAIT_R
    // 0x1010 AM = 0: Horizontal, I/D[1:0] = 01: Horizontal: increment, Vertical: decrement
    // 0x1020 AM = 0: Horizontal, I/D[1:0] = 10: Horizontal: decrement, Vertical: increment
    // 0x1030 AM = 0: Horizontal, I/D[1:0] = 11: Horizontal: increment, Vertical: increment - ILI9225_PORTRAIT_L
    // 0x1008 AM = 1: Vertical,   I/D[1:0] = 00: Horizontal: decrement, Vertical: decrement
    // 0x1018 AM = 1: Vertical,   I/D[1:0] = 01: Horizontal: increment, Vertical: decrement - ILI9225_LANDSCAPE_T
    // 0x1028 AM = 1: Vertical,   I/D[1:0] = 10: Horizontal: decrement, Vertical: increment - ILI9225_LANDSCAPE_B
    // 0x1038 AM = 1: Vertical,   I/D[1:0] = 11: Horizontal: increment, Vertical: increment

    switch (_orientation) {
        case ILI9225_PORTRAIT_L:
            // 0x1030 AM = 0: Horizontal, I/D[1:0] = 11: Horizontal: increment, Vertical: increment
            _entryMode = 0x1030;
            _maxX = ILI9225_LCD_WIDTH;
            _maxY = ILI9225_LCD_HEIGHT;
            break;
        case ILI9225_LANDSCAPE_B:
            // 0x1028 AM = 1: Vertical,   I/D[1:0] = 10: Horizontal: decrement, Vertical: increment
            _entryMode = 0x1028;
            _maxX = ILI9225_LCD_HEIGHT;
            _maxY = ILI9225_LCD_WIDTH;
            break;
        case ILI9225_PORTRAIT_R:
            // 0x1000 AM = 0: Horizontal, I/D[1:0] = 00: Horizontal: decrement, Vertical: decrement
            _entryMode = 0x1000;
            _maxX = ILI9225_LCD_WIDTH;
            _maxY = ILI9225_LCD_HEIGHT;
            break;
        case ILI9225_LANDSCAPE_T:
            // 0x1018 AM = 1: Vertical,   I/D[1:0] = 01: Horizontal: increment, Vertical: decrement
            _entryMode = 0x1018;
            _maxX = ILI9225_LCD_HEIGHT;
            _maxY = ILI9225_LCD_WIDTH;
            break;
    }
    _writeRegister(ILI9225_ENTRY_MODE, _entryMode); // set GRAM write direction and BGR=1.
}

uint8_t TFT_22_ILI9225::getOrientation()
{
    return _orientation;
}


// Graphics functions

void TFT_22_ILI9225::pixel(uint16_t x1, uint16_t y1, uint16_t color)
{
    if ((x1 >= _maxX) || (y1 >= _maxY)) return;

    _setWindow(x1, y1, x1 + 1, y1 + 1);
    _orientCoordinates(x1, y1);
    _rs = 1;
    _cs = 0;
    _spi.write(color >> 8);
    _spi.write(color & 0xff);
    _cs = 1;
}

void TFT_22_ILI9225::line(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color)
{
    // Classic Bresenham algorithm
    int16_t steep = abs(y2 - y1) > abs(x2 - x1);
    int16_t dx, dy;

    if (steep) {
        _swap(x1, y1);
        _swap(x2, y2);
    }

    if (x1 > x2) {
        _swap(x1, x2);
        _swap(y1, y2);
    }

    dx = x2 - x1;
    dy = abs(y2 - y1);

    int16_t err = dx / 2;
    int16_t ystep;

    if (y1 < y2) ystep = 1;
    else ystep = -1;


    for (; x1 <= x2; x1++) {
        if (steep) pixel(y1, x1, color);
        else       pixel(x1, y1, color);

        err -= dy;
        if (err < 0) {
            y1 += ystep;
            err += dx;
        }
    }
}

void TFT_22_ILI9225::hline(uint16_t x1, uint16_t x2, uint16_t y, uint16_t color)
{
    _writeRegister(ILI9225_ENTRY_MODE, 0x1028); // set GRAM write direction and BGR=1.
    _setWindow(x1, y, x2 + 1, y + 1);
    if (x2 < x1) {
        _swap(x1, x2);
    }
    _rs = 1;
    _cs = 0;
    for (uint16_t x = x1; x <= x2; x++) {
        _spi.write(color >> 8);
        _spi.write(color & 0xff);
    } 
    _cs = 1;
    _writeRegister(ILI9225_ENTRY_MODE, _entryMode); // set GRAM write direction and BGR=1.
}

void TFT_22_ILI9225::vline(uint16_t x, uint16_t y1, uint16_t y2, uint16_t color)
{
    _writeRegister(ILI9225_ENTRY_MODE, 0x1010); // set GRAM write direction and BGR=1.
    _setWindow(x, y1, x + 1, y2 + 1);
    if (y2 < y1) {
        _swap(y1, y2);
    }
    _rs = 1;
    _cs = 0;
    for (uint16_t y = y1; y <= y2; y++) {
        _spi.write(color >> 8);
        _spi.write(color & 0xff);
    } 
    _cs = 1;
    _writeRegister(ILI9225_ENTRY_MODE, _entryMode); // set GRAM write direction and BGR=1.
}

void TFT_22_ILI9225::rect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color) {
    
    vline(x1, y1, y2, color);
    hline(x1, x2, y1, color);
    hline(x1, x2, y2, color);
    vline(x2, y1, y2, color);
}

void TFT_22_ILI9225::fillrect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color)
{
    _setWindow(x1, y1, x2, y2);
    _startData();
    for (uint16_t t = (y2 - y1 + 1) * (x2 - x1 + 1); t > 0; t--) {
        _writeData(color);
    }
    _endData();
}

void TFT_22_ILI9225::circle(uint16_t x0, uint16_t y0, uint16_t r, uint16_t color)
{
    int16_t f = 1 - r;
    int16_t ddF_x = 1;
    int16_t ddF_y = -2 * r;
    int16_t x = 0;
    int16_t y = r;

    pixel(x0, y0 + r, color);
    pixel(x0, y0 - r, color);
    pixel(x0 + r, y0, color);
    pixel(x0 - r, y0, color);

    while (x < y) {
        if (f >= 0) {
            y--;
            ddF_y += 2;
            f += ddF_y;
        }
        x++;
        ddF_x += 2;
        f += ddF_x;

        pixel(x0 + x, y0 + y, color);
        pixel(x0 - x, y0 + y, color);
        pixel(x0 + x, y0 - y, color);
        pixel(x0 - x, y0 - y, color);
        pixel(x0 + y, y0 + x, color);
        pixel(x0 - y, y0 + x, color);
        pixel(x0 + y, y0 - x, color);
        pixel(x0 - y, y0 - x, color);
    }
}

void TFT_22_ILI9225::fillcircle(uint8_t x0, uint8_t y0, uint8_t radius, uint16_t color)
{
    int16_t f = 1 - radius;
    int16_t ddF_x = 1;
    int16_t ddF_y = -2 * radius;
    int16_t x = 0;
    int16_t y = radius;

    while (x < y) {
        if (f >= 0) {
            y--;
            ddF_y += 2;
            f += ddF_y;
        }
        x++;
        ddF_x += 2;
        f += ddF_x;

        hline(x0 + x, x0 - x, y0 + y, color); // bottom
        hline(x0 + x, x0 - x, y0 - y, color); // top
        vline(x0 + y, y0 - x, y0 + x, color); // right
        vline(x0 - y, y0 - x, y0 + x, color); // left
    }
    fillrect(x0 - x, y0 - y, x0 + x, y0 + y, color);
}

void TFT_22_ILI9225::triangle(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t x3, uint16_t y3, uint16_t color)
{
    line(x1, y1, x2, y2, color);
    line(x2, y2, x3, y3, color);
    line(x3, y3, x1, y1, color);
}

void TFT_22_ILI9225::filltriangle(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t x3, uint16_t y3, uint16_t color)
{
    uint16_t a, b, y, last;

    // Sort coordinates by Y order (y3 >= y2 >= y1)
    if (y1 > y2) {
        _swap(y1, y2); _swap(x1, x2);
    }
    if (y2 > y3) {
        _swap(y3, y2); _swap(x3, x2);
    }
    if (y1 > y2) {
        _swap(y1, y2); _swap(x1, x2);
    }

    if (y1 == y3) { // Handle awkward all-on-same-line case as its own thing
        a = b = x1;
        if (x2 < a)      a = x2;
        else if (x2 > b) b = x2;
        if (x3 < a)      a = x3;
        else if (x3 > b) b = x3;
        hline(a, b, y1, color);
        return;
    }

    uint16_t    dx11 = x2 - x1,
        dy11 = y2 - y1,
        dx12 = x3 - x1,
        dy12 = y3 - y1,
        dx22 = x3 - x2,
        dy22 = y3 - y2,
        sa = 0,
        sb = 0;

    // For upper part of triangle, find scanline crossings for segments
    // 0-1 and 0-2.  If y2=y3 (flat-bottomed triangle), the scanline y2
    // is included here (and second loop will be skipped, avoiding a /0
    // error there), otherwise scanline y2 is skipped here and handled
    // in the second loop...which also avoids a /0 error here if y1=y2
    // (flat-topped triangle).
    if (y2 == y3) last = y2;   // Include y2 scanline
    else          last = y2 - 1; // Skip it

    for (y = y1; y <= last; y++) {
        a = x1 + sa / dy11;
        b = x1 + sb / dy12;
        sa += dx11;
        sb += dx12;
        /* longhand:
        a = x1 + (x2 - x1) * (y - y1) / (y2 - y1);
        b = x1 + (x3 - x1) * (y - y1) / (y3 - y1);
        */
        if (a > b) _swap(a, b);
        hline(a, b, y, color);
    }

    // For lower part of triangle, find scanline crossings for segments
    // 0-2 and 1-2.  This loop is skipped if y2=y3.
    sa = dx22 * (y - y2);
    sb = dx12 * (y - y1);
    for (; y <= y3; y++) {
        a = x2 + sa / dy22;
        b = x1 + sb / dy12;
        sa += dx22;
        sb += dx12;
        /* longhand:
        a = x2 + (x3 - x2) * (y - y2) / (y3 - y2);
        b = x1 + (x3 - x1) * (y - y1) / (y3 - y1);
        */
        if (a > b) _swap(a, b);
        hline(a, b, y, color);
    }
}

void TFT_22_ILI9225::roundrect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t rad, bool fill, uint16_t color)
{
    signed int a, b, P;

    a = 0;                      // increment by 1
    b = rad;                // decrement by 1 using P
    P = 1 - rad;

    if (fill)
    {
        fillrect(x1, y1 + rad, x2, y2 - rad, color);

        do
        {
            fillrect(x1 - a + rad, y1 - b + rad, a + x2 - rad, y1 - b + rad, color);   // 8 --> 1
            fillrect(x1 - b + rad, y1 - a + rad, b + x2 - rad, y1 - a + rad, color);   // 7 --> 2
            fillrect(x1 - b + rad, a + y2 - rad, b + x2 - rad, a + y2 - rad, color);   // 6 --> 3
            fillrect(x1 - a + rad, b + y2 - rad, a + x2 - rad, b + y2 - rad, color);   // 5 --> 4

            if (P < 0)
                P += 3 + 2 * a++;
            else
                P += 5 + 2 * (a++ - b--);

        } while (a <= b);
    }
    else
    {
        hline(x1 + rad, x2 - rad, y1, color);   // top
        hline(x1 + rad, x2 - rad, y2, color);   // bottom
        vline(x1, y1 + rad, y2 - rad, color);   // left
        vline(x2, y1 + rad, y2 - rad, color);   // right

        do
        {
            pixel(a + x2 - rad, y1 - b + rad, color);   // `````` Segment 1
            pixel(b + x2 - rad, y1 - a + rad, color);   // `````` Segment 2

            pixel(b + x2 - rad, a + y2 - rad, color);   // `````` Segment 3
            pixel(a + x2 - rad, b + y2 - rad, color);   // `````` Segment 4

            pixel(x1 - a + rad, b + y2 - rad, color);   // `````` Segment 5
            pixel(x1 - b + rad, a + y2 - rad, color);   // `````` Segment 6

            pixel(x1 - b + rad, y1 - a + rad, color);   // `````` Segment 7
            pixel(x1 - a + rad, y1 - b + rad, color);   // `````` Segment 8

            if (P < 0)
                P += 3 + 2 * a++;
            else
                P += 5 + 2 * (a++ - b--);
        } while (a <= b);
    }
}

uint16_t TFT_22_ILI9225::maxX(void)
{
    return _maxX - 1;
}

uint16_t TFT_22_ILI9225::maxY(void)
{
    return _maxY - 1;
}

uint16_t TFT_22_ILI9225::width(void)
{
    return _maxX;
}

uint16_t TFT_22_ILI9225::height(void)
{
    return _maxY;
}

uint16_t TFT_22_ILI9225::setColor(uint8_t red8, uint8_t green8, uint8_t blue8)
{
    // rgb16 = red5 green6 blue5
    return (red8 >> 3) << 11 | (green8 >> 2) << 5 | (blue8 >> 3);
}

void TFT_22_ILI9225::splitColor(uint16_t rgb, uint8_t &red, uint8_t &green, uint8_t &blue)
{
    // rgb16 = red5 green6 blue5
    
    red = (rgb & 0xF800) >> 11 << 3;
    green = (rgb & 0x7E0) >> 5 << 2;
    blue = (rgb & 0x1F) << 3;
}


// Text functions

void TFT_22_ILI9225::setFont(unsigned char* f)
{
    font = f;
}

uint8_t TFT_22_ILI9225::fontX(void)
{
    return font[FONT_HORZ];
}

uint8_t TFT_22_ILI9225::fontY(void)
{
    return font[FONT_VERT];
}

int TFT_22_ILI9225::putc(int value)
{
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + (font[FONT_VERT] + char_line_spacing);
        if (char_y >= height() - (font[FONT_VERT] + char_line_spacing)) {
            char_y = 0;
        }
        return value;
    }
    if ((value < 32) || (value > 127)) {
        return value;
    }
    character(char_x, char_y, value);
    return value;
}

void TFT_22_ILI9225::character(int x, int y, int c)
{
    unsigned int hor, vert, offset, bpl, i, j, b;
    unsigned char* char_ptr;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[FONT_LENGTH];         // bytes / char
    hor    = font[FONT_HORZ];           // get hor size of font
    vert   = font[FONT_VERT];           // get vert size of font
    bpl    = font[FONT_BYTES_VERT];     // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert + char_line_spacing;
        if (char_y >= height() - (vert + char_line_spacing)) {
            char_y = 0;
        }
    }
    _setWindow(char_x, char_y, (char_x+hor)-1, (char_y+vert)-1);

    char_ptr = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = char_ptr[0];                          // width of actual char
    _startData();

    // Portrait
    for (j = 0; j < vert; j++) {  //  vert line
        for (i = 0; i < hor; i++) {   //  horz line
            z =  char_ptr[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                _spi.write(_background >> 8);
                _spi.write(_background & 0xff);
            } else {
                _spi.write(_foreground >> 8);
                _spi.write(_foreground & 0xff);
            }
        }
    }
    
    _endData();
    _setWindowMax();
    if ((w + 2) < hor) {                   // x offset to next char
        char_x += w + 2;
    } else {
        char_x += hor;
    }
}

uint16_t TFT_22_ILI9225::getStringWidth(char * s)
{
    unsigned char* char_ptr;
    uint16_t width  = 0;
    uint16_t offset = font[FONT_LENGTH];         // bytes / char
    uint16_t hor    = font[FONT_HORZ];           // get hor size of font

    uint16_t len = strlen(s);
    for (uint8_t i = 0; i < len; i++) {
        if ((s[i] < 31) || (s[i] > 127)) {
            continue;               // test char range
        }
        char_ptr = &font[((s[i] -32) * offset) + 4]; // start of char bitmap
        unsigned char w = char_ptr[0];
        if ((w + 2) < hor) {                   // x offset to next char
            width += w + 2;
        } else {
            width += hor;
        }
    }
    return width - 1;
}

void TFT_22_ILI9225::foreground(uint16_t color)
{
    _foreground = color;
}

void TFT_22_ILI9225::background(uint16_t color)
{
    _background = color;
}

void TFT_22_ILI9225::locate(int x, int y)
{
    char_x = x;
    char_y = y;
}

void TFT_22_ILI9225::gotoxy(int x, int y)
{
    char_x = x * font[FONT_HORZ];
    char_y = y * (font[FONT_VERT] + char_line_spacing);
}

void TFT_22_ILI9225::home(void)
{
    gotoxy(0, 0);
}

void TFT_22_ILI9225::linespacing(int line_spacing)
{
    char_line_spacing = line_spacing;
}

int TFT_22_ILI9225::columns()
{
    return width() / font[FONT_HORZ];
}

int TFT_22_ILI9225::rows()
{
    return height() / font[FONT_VERT];
}

void TFT_22_ILI9225::unicode2ascii(char *uni_str, char *ascii_str)
{
    int counter = 0;
    int Uch = 0;
    char chl, chh;


    while (*uni_str)
    {
        chl = *uni_str++;
        chh = *uni_str++;

        Uch = 0;
        Uch = ((Uch | chh) << 8) | chl;

        if (Uch > 1574 && Uch < 1591)
            *(ascii_str + counter) = (char)(Uch - 1376);
        else if (Uch > 1590 && Uch < 1595)
            *(ascii_str + counter) = (char)(Uch - 1375);
        else if (Uch > 1600 && Uch < 1603)
            *(ascii_str + counter) = (char)(Uch - 1380);
        else if (Uch == 1705)
            *(ascii_str + counter) = (char)(Uch - 1482);
        else if (Uch == 1604)
            *(ascii_str + counter) = (char)(Uch - 1379);
        else if (Uch > 1604 && Uch < 1609)
            *(ascii_str + counter) = (char)(Uch - 1378);
        else if (Uch == 1740)
            *(ascii_str + counter) = (char)(Uch - 1503);
        else if (Uch == 1574)
            *(ascii_str + counter) = (char)(Uch - 1381);
        else if (Uch == 1662)
            *(ascii_str + counter) = (char)(Uch - 1533);
        else if (Uch == 1670)
            *(ascii_str + counter) = (char)(Uch - 1529);
        else if (Uch == 1688)
            *(ascii_str + counter) = (char)(Uch - 1546);
        else if (Uch == 1711)
            *(ascii_str + counter) = (char)(Uch - 1567);
        else if (Uch == 1570)
            *(ascii_str + counter) = (char)(Uch - 1376);
        else if (Uch > 1631 && Uch < 1642)
            *(ascii_str + counter) = (char)(Uch - 1584);
        else if (Uch == 65536)
            *(ascii_str + counter) = NULL;
        else
            *(ascii_str + counter) = (char)Uch;


        counter++;

    }
    *(ascii_str + counter) = NULL;
}

bool TFT_22_ILI9225::claim (FILE *stream)
{
    if ( _path == NULL) {
        fprintf(stderr, "claim requires a name to be given in the instantioator of the TextDisplay instance!\r\n");
        return false;
    }
    if (freopen(_path, "w", stream) == NULL) {
        // Failed, should not happen
        return false;
    }
    // make sure we use line buffering
    setvbuf(stdout, NULL, _IOLBF, ILI9225_CHARS_PER_LINE);
    return true;
} 

// Private functions

void TFT_22_ILI9225::_swap(uint16_t &a, uint16_t &b)
{
    uint16_t w = a;
    a = b;
    b = w;
}

void TFT_22_ILI9225::_setWindow(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1)
{
    _orientCoordinates(x0, y0);
    _orientCoordinates(x1, y1);

    if (x1 < x0) _swap(x0, x1);
    if (y1 < y0) _swap(y0, y1);

    _writeRegister(ILI9225_HORIZONTAL_WINDOW_ADDR1, x1);
    _writeRegister(ILI9225_HORIZONTAL_WINDOW_ADDR2, x0);

    _writeRegister(ILI9225_VERTICAL_WINDOW_ADDR1, y1);
    _writeRegister(ILI9225_VERTICAL_WINDOW_ADDR2, y0);

    _writeRegister(ILI9225_RAM_ADDR_SET1, x0);
    _writeRegister(ILI9225_RAM_ADDR_SET2, y0);

    _writeCommand(0x00, 0x22);
}

void TFT_22_ILI9225::_setWindowMax(void)
{
    _setWindow(0, 0, maxX(), maxY());
}

void TFT_22_ILI9225::_orientCoordinates(uint16_t &x, uint16_t &y)
{
    switch (_orientation) {
        case 0:  // ok
            break;
        case 1: // ok
            y = _maxY - y - 1;
            _swap(x, y);
            break;
        case 2: // ok
            x = _maxX - x - 1;
            y = _maxY - y - 1;
            break;
        case 3: // ok
            x = _maxX - x - 1;
            _swap(x, y);
            break;
    }
}

void TFT_22_ILI9225::_writeRegister(uint16_t reg, uint16_t data)
{
    _writeCommand(reg >> 8, reg & 0xff);
    _startData();
    _writeData(data);
    _endData();
}

void TFT_22_ILI9225::_writeCommand(uint8_t HI, uint8_t LO)
{
    _rs = 0;
    _cs = 0;
    _spi.write(HI);
    _spi.write(LO);
    _cs = 1;
}

void TFT_22_ILI9225::_startData(void)
{
    _rs = 1;
    _cs = 0;
}

void TFT_22_ILI9225::_writeData(uint16_t data)
{
    _spi.write(data >> 8);
    _spi.write(data & 0xff);
}

void TFT_22_ILI9225::_endData(void)
{
    _cs = 1;
}
