/* mbed HSV Library
 * Copyright (c) 2013-2013 JackB, cstyles
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
  
#ifndef MBED_HSV_H
#define MBED_HSV_H
 
#include "mbed.h"
#include "math.h"

/** HSV calculation class
 *
 * Example:
 * @code
 * // Continuously sweep the hsv through it's full range
 * #include "mbed.h"
 * #include "HSV.h"
 * 
 * HSV myhsv(LED1,LED2,LED3);
 *
 * int main() {
 *     myhsv.SetS(1.0);
 *     myhsv.SetV(1.0);
 *     while(1) {
 *         for(int i=0; i<100; i++) {
 *             myhsv.SetH((float)i/100.0);
 *             myhsv.HSV_to_RGB();
 *             wait(0.01);
 *         }
 *     }
 * }
 * @endcode
 */

#define MAXHUE 4095 // 49152/6=8192 (65536/6=10922.7)
#define MAXVAL 4095
 
class HSV {
 
public:
    struct RgbStruct
    {
        float R, G, B, W;
        float Rcf, Gcf, Bcf, Wcf;
        float Out;
    };
    struct RgbStruct Rgb;
    struct HsvStruct
    {
        float H, S, V, P;
    };
    struct HsvStruct Hsv;
    struct HsvStruct Hsv2;
  
  /** Create a hsv object
   *
   * @param void 
   */
  HSV();

    void SetPower(bool Power);
    
    int UseWeb(char* String);
    int UseRGB(int R, int G, int B);
    int UseHSV(float H, float S, float V);

    void SetRcf(float Rcf);
    void SetGcf(float Gcf);
    void SetBcf(float Bcf);
    void SetWcf(float Wcf);

  /** Set the hue value, normalised to it's full range
   *
   * @param Hue normalised number 0.0-1.0 to represent the full range.
   */
  void SetH(float Hue);

  /** Set the saturation value, normalised to it's full range
   *
   * @param Saturation normalised number 0.0-1.0 to represent the full range.
   */
  void SetS(float Saturation);

  /** Set the brightness value, normalised to it's full range
   *
   * @param Value normalised number 0.0-1.0 to represent the full range.
   */
  void SetV(float Value);
  void SetVV(float Value);
  void SetP(float Value);

  /** Set the white value, normalised to it's full range
   *
   * @param Value normalised number 0.0-1.0 to represent the full range.
   */
  void SetW(float Value);

    uint16_t GetRout(void);
    uint16_t GetGout(void);
    uint16_t GetBout(void);
    uint16_t GetWout(void);

  /** Get the red value, normalised to it's full range
   *
   * @param void
   */
  float GetR(void);

  /** Get the green value, normalised to it's full range
   *
   * @param void
   */
  float GetG(void);

  /** Get the blue value, normalised to it's full range
   *
   * @param void
   */
  float GetB(void);

  /** Get the white value, normalised to it's full range
   *
   * @param void
   */
  float GetW(void);

    float GetH(void);
    float GetS(void);
    float GetV(void);
    float GetH2(void);
    float GetS2(void);
    float GetV2(void);

    /** Calculate the red, green, and blue value
     *
     * @param void
     */
    void HSV_to_RGB(void);

    void RGB_to_HSV(void);

private:
    bool _Power;
//    PwmOut _pin1;
//    PwmOut _pin2;
//    PwmOut _pin3;
//    PwmOut _pin4;
//protected:
//  PwmOut _pwm;
//  float _range;
//  float _degrees;
//  float _p;
};
 
#endif

