/* FRAM_SPI - Ramtron FRAM FM24VXX series driver
* Copyright (c) 2012 Oleg Evsegneev
* Released under the MIT License: http://mbed.org/license/mit
*
* Datasheet http://www.ramtron.com/files/datasheets/FM25V02_ds.pdf
*/
 
#ifndef  FRAMSPI_
#define  FRAMSPI_
 
#include "mbed.h"
 
// commands
#define READ            0x03
#define WRITE           0x02
#define READ_STATUS     0x05 // RDSR
#define WRITE_ENABLED   0x06 // WREN
#define WRITE_STATUS    0x01 // WRSR
#define READ_ID         0x9F // RDID
#define READ_SN         0xC3 // RSN
 
/** An interface for the Ramtron FM25VXX FRAM over SPI
*
* 
*
* @code
* #include "mbed.h"
* #include "FRAMSPI.h"
* 
* SPI spi(p5,p6,p7);
* FRAMSPI fram(spi,p8);
* Serial pc(USBTX, USBRX);
* 
* int main() {
*     char wdata[] = "Hello world!";
*     char rdata[14];
* 
*     fram.write(0, wdata, 13); // 12 symbols + zero terminator
*     fram.read(0, rdata, 13);
* 
*     pc.printf("data: %s", rdata);
* }
* @endcode
* connections:
* chip pin 1 to any GPIO. Chip select (!S)
* chip pin 2 SO to MISO. Write (!W)
* chip pin 3 to Vout or N/C
* chip pin 4 to Gnd
* chip pin 5 SI to MOSI
* chip pin 6 SCK to sck 
* chip pin 7 to Vout. !Hold
* chip pin 8 to Vout
*/
 
class FRAMSPI {
 
public:
    /** Create an interface
    *
    * @param spi    An SPI object
    * @param ncs    Chip select pin
    */
    FRAMSPI(SPI& spi, PinName ncs);
 
    /** read chip 8 byte device ID
    */
    void readID(char *buff);
 
    /** read chip 2 byte serial number
    */
    void readSN(char *buff);
 
    /** read a byte from FRAM
    * @param addr    The address to read from
    * @return the byte at that address
    */
    char read(int addr);
 
    /** read multiple bytes from FRAM into a buffer
    * @param addr    The FRAM address to read from
    * @param buff     The buffer to read into (must be big enough!)
    * @param cnt      The number of bytes to read
    */
    void read(int addr, char *buff, int cnt);
 
    /** write a byte to FRAM
    * @param addr    The address SFRAM to write to
    * @param data       The byte to write there
    */
    void write(int addr, char data);
 
    /** write multiple bytes to FRAM from a buffer
    * @param addr    The FRAM address write to
    * @param buff     The buffer to write from
    * @param cnt      The number of bytes to write
    */
    void write(int addr, char *buff, int cnt);
 
private:
    SPI& _spi;
    DigitalOut _ncs;
    void prepareCommand(char cmd, int addr);
    void select();
    void deselect();
};
 
#endif
