/** mbded library for driving the PMAXIM ARM_RTC Real Time Clock
* datasheet link : http://datasheets.maximintegrated.com/en/ds/ARM_RTC.pdf
* breakout       : MACETECH ChronoDot V2.1 High Precision RTC
* remi cormier 2012
* WARNING : sda and sdl should be pulled up with 2.2k resistor
*/

/** Example code
* @code
// ARM_RTC Library test program
// remi cormier 2012

#include "mbed.h"
#include "ARM_RTC.h"

Serial pc(USBTX, USBRX);

int hour;
int minute;
int second;

int dayOfWeek;
int date;
int month;
int year;  
   
ARM_RTC RTC(p28,p27);


int main()
    {printf("\r\n\nARM_RTC Library test program\r\nremi cormier 2012\r\n\n");
    
     RTC.setI2Cfrequency(400000);
    
     //RTC.writeRegister(ARM_RTC_Aging_Offset,0); // uncomment to set Aging Offset 1LSB = approx. 0.1 ppm according from datasheet = 0.05 ppm @ 21 °C from my measurments
     
     RTC.convertTemperature();
      
     int reg=RTC.readRegister(ARM_RTC_Aging_Offset);
     if (reg>127)
        {reg=reg-256;}
     pc.printf("Aging offset : %i\r\n",reg);
         
     pc.printf("OSF flag : %i",RTC.OSF());
     pc.printf("\r\n");
     
     RTC.readDate(&date,&month,&year);
     pc.printf("date : %02i-%02i-%02i",date,month,year);
     pc.printf("\r\n");
     
     //RTC.setTime(19,48,45); // uncomment to set time
     
     RTC.readTime(&hour,&minute,&second);
     pc.printf("time : %02i:%02i:%02i",hour,minute,second);
     pc.printf("\r\n");
     
     //RTC.setDate(6,22,12,2012); // uncomment to set date
     
     RTC.readDateTime(&dayOfWeek,&date,&month,&year,&hour,&minute,&second);
     pc.printf("date time : %i / %02i-%02i-%02i %02i:%02i:%02i",dayOfWeek,date,month,year,hour,minute,second);
     pc.printf("\r\n");
     
     pc.printf("temperature :%6.2f",RTC.readTemp());
     pc.printf("\r\n");
    }
* @endcode
*/

/*
http://www.cplusplus.com/reference/ctime/strftime/
%a   Abbreviated weekday name *  Thu
%A   Full weekday name * Thursday
%b   Abbreviated month name *    Aug
%B   Full month name *   August
%d   Day of the month, zero-padded (01-31)   23
%e   Day of the month, space-padded ( 1-31)  23
%F   Short YYYY-MM-DD date, equivalent to %Y-%m-%d   2001-08-23
%H   Hour in 24h format (00-23)  14
%j   Day of the year (001-366)   235
%m   Month as a decimal number (01-12)   08
%M   Minute (00-59)  55
%R   24-hour HH:MM time, equivalent to %H:%M 14:55
%S   Second (00-61)  02
%T   ISO 8601 time format (HH:MM:SS), equivalent to %H:%M:%S 14:55:02
%u   ISO 8601 weekday as number with Monday as 1 (1-7)   4
%V   ISO 8601 week number (00-53)    34
%w   Weekday as a decimal number with Sunday as 0 (0-6)  4
%W   Week number with the first Monday as the first day of week one (00-53)  34
%X   Time representation *   14:55:02
%y   Year, last two digits (00-99)   01
%Y   Year    2001

http://www.cplusplus.com/reference/ctime/tm/
Member   Type    Meaning                        Range
tm_sec   int     seconds after the minute       0-61*
tm_min   int     minutes after the hour         0-59
tm_hour  int     hours since midnight           0-23
tm_mday  int     day of the month               1-31
tm_mon   int     months since January           0-11
tm_year  int     years since 1900    
tm_wday  int     days since Sunday              0-6     (0 = Sunday)
tm_yday  int     days since January 1           0-365
tm_isdst         int Daylight Saving Time flag   
The Daylight Saving Time flag (tm_isdst) is greater than zero if Daylight Saving Time is in effect,
zero if Daylight Saving Time is not in effect, and less than zero if the information is not available.
* tm_sec is generally 0-59. The extra range is to accommodate for leap seconds in certain systems.

Member  Type    Meaning Range
tm_sec  int seconds after the minute    0-61*
tm_min  int minutes after the hour  0-59
tm_hour int hours since midnight    0-23
tm_mday int day of the month    1-31
tm_mon  int months since January    0-11
tm_year int years since 1900    
tm_wday int days since Sunday   0-6
tm_yday int days since January 1    0-365
tm_isdst    int Daylight Saving Time flag   

http://www.epochconverter.com/programming/c
Convert from epoch to human readable date
    time_t     now;
    struct tm  ts;
    char       buf[80];
    // Get current time
    time(&now);
    // Format time, "ddd yyyy-mm-dd hh:mm:ss zzz"
    ts = *localtime(&now);
    strftime(buf, sizeof(buf), "%a %Y-%m-%d %H:%M:%S %Z", &ts);
    printf("%s\n", buf);

Convert from human readable date to epoch
    struct tm t;
    time_t t_of_day;
    t.tm_year = 2011-1900;
    t.tm_mon = 7;           // Month, 0 - jan
    t.tm_mday = 8;          // Day of the month
    t.tm_hour = 16;
    t.tm_min = 11;
    t.tm_sec = 42;
    t.tm_isdst = -1;        // Is DST on? 1 = yes, 0 = no, -1 = unknown
    t_of_day = mktime(&t);
    printf("seconds since the Epoch: %ld\n", (long) t_of_day)

Convert to struct tm
    time_t rawtime;
    struct tm * t;
    time(&rawtime);
    t = localtime(&rawtime);
    printf("%d\n", t->tm_sec);


https://github.com/raburton/esp8266/blob/master/drivers/ARM_RTC.c
https://github.com/raburton/esp8266/blob/master/drivers/ARM_RTC.h
    
*/
#include <mbed.h>
#include "TableSummerTime.h"
#include "TableDayLight.h"

#ifndef MBED_ARM_RTC_H
#define MBED_ARM_RTC_H

//ARM_RTC 8 bit adress
#define ARM_RTC_ADDR                         0x68    // 0x68 << 1 = 0xD0

#define ARM_RTC_STAT_OSCILLATOR              0x80
#define ARM_RTC_STAT_32KHZ                   0x08
#define ARM_RTC_STAT_BUSY                    0x04
#define ARM_RTC_STAT_ALARM_2                 0x02
#define ARM_RTC_STAT_ALARM_1                 0x01

//ARM_RTC registers
#define ARM_RTC_Seconds                      0x00
#define ARM_RTC_Minutes                      0x01
#define ARM_RTC_Hours                        0x02
// ARM_RTC Hours bits
#define ARM_RTC_bit_AM_PM                    0x20
#define ARM_RTC_bit_12_24                    0x40

#define ARM_RTC_Day                          0x03
#define ARM_RTC_Date                         0x04
#define ARM_RTC_Month_Century                0x05
#define ARM_RTC_Year                         0x06

#define ARM_RTC_Alarm1_Seconds               0x07
#define ARM_RTC_Alarm1_Minutes               0x08
#define ARM_RTC_Alarm1_Hours                 0x09
#define ARM_RTC_Alarm1_Day_Date              0x0A

#define ARM_RTC_Alarm2_Minutes               0x0B
#define ARM_RTC_Alarm2_Hours                 0x0C
#define ARM_RTC_Alarm_2_Day_Date             0x0D

#define ARM_RTC_Control                      0x0E

// ARM_RTC Control bits
#define ARM_RTC_bit_A1IE                     0x01
#define ARM_RTC_bit_A2IE                     0x02
#define ARM_RTC_bit_INTCN                    0x04
#define ARM_RTC_bit_SQW_1Hz                  0x00
#define ARM_RTC_bit_SQW_1024Hz               0x08
#define ARM_RTC_bit_SQW_4096Hz               0x10
#define ARM_RTC_bit_SQW_8192Hz               0x18
#define ARM_RTC_bit_CONV                     0x20
#define ARM_RTC_bit_BBSQW                    0x40
#define ARM_RTC_bit_EOSCb                    0x80

#define ARM_RTC_CTRL_ALARM1_INT              0x01
#define ARM_RTC_CTRL_ALARM2_INT              0x02
#define ARM_RTC_CTRL_ALARM_INTS              0x04
#define ARM_RTC_CTRL_SQWAVE_1HZ              0x00
#define ARM_RTC_CTRL_SQWAVE_1024HZ           0x08
#define ARM_RTC_CTRL_SQWAVE_4096HZ           0x10
#define ARM_RTC_CTRL_SQWAVE_8192HZ           0x18
#define ARM_RTC_CTRL_TEMPCONV                0x20
#define ARM_RTC_CTRL_SQUAREWAVE_BB           0x40
#define ARM_RTC_CTRL_OSCILLATOR              0x80

#define ARM_RTC_ALARM_NONE                   0
#define ARM_RTC_ALARM_1                      1
#define ARM_RTC_ALARM_2                      2
#define ARM_RTC_ALARM_BOTH                   3

#define ARM_RTC_ALARM1_EVERY_SECOND          0
#define ARM_RTC_ALARM1_MATCH_SEC             1
#define ARM_RTC_ALARM1_MATCH_SECMIN          2
#define ARM_RTC_ALARM1_MATCH_SECMINHOUR      3
#define ARM_RTC_ALARM1_MATCH_SECMINHOURDAY   4
#define ARM_RTC_ALARM1_MATCH_SECMINHOURDATE  5

#define ARM_RTC_ALARM2_EVERY_MIN             0
#define ARM_RTC_ALARM2_MATCH_MIN             1
#define ARM_RTC_ALARM2_MATCH_MINHOUR         2
#define ARM_RTC_ALARM2_MATCH_MINHOURDAY      3
#define ARM_RTC_ALARM2_MATCH_MINHOURDATE     4

#define ARM_RTC_ALARM_WDAY                   0x40
#define ARM_RTC_ALARM_NOTSET                 0x80

#define ARM_RTC_ADDR_TIME                    0x00
#define ARM_RTC_ADDR_ALARM1                  0x07
#define ARM_RTC_ADDR_ALARM2                  0x0b
#define ARM_RTC_ADDR_CONTROL                 0x0e
#define ARM_RTC_ADDR_STATUS                  0x0f
#define ARM_RTC_ADDR_AGING                   0x10
#define ARM_RTC_ADDR_TEMP                    0x11

#define ARM_RTC_SET                          0
#define ARM_RTC_CLEAR                        1
#define ARM_RTC_REPLACE                      2

#define ARM_RTC_12HOUR_FLAG                  0x40
#define ARM_RTC_12HOUR_MASK                  0x1f
#define ARM_RTC_PM_FLAG                      0x20
#define ARM_RTC_MONTH_MASK                   0x1f

#define ARM_RTC_Control_Status               0x0F

// ARM_RTC Control/Status bits
#define ARM_RTC_bit_BSY                      0x04
#define ARM_RTC_bit_EN32kHz                  0x08
#define ARM_RTC_bit_OSF                      0x80

#define ARM_RTC_Aging_Offset                 0x10
#define ARM_RTC_MSB_Temp                     0x11
#define ARM_RTC_LSB_Temp                     0x12

#define NTP_OFFSET                          2208988800ULL

/* Interface to MAXIM ARM_RTC RTC */
class ARM_RTC
{
public :
    /** Create an instance of the ARM_RTC connected to specfied I2C pins
    *
    * @param sda The I2C data pin
    * @param scl The I2C clock pin
    */
    ARM_RTC();

    bool checkTimeLost(void);

    /** Set the time registers
    * @param seconds since 1900
    */
    void setDateTimeSecsSince1900(uint32_t secsSince1900);

    /** Set the time registers
    * @param seconds since 1970
    */
    void setDateTimeSecsSince1970(uint32_t secsSince1970);

    /** Set the time registers
    * @return seconds since 1970
    */
    uint32_t getDateTimeSecsSince1970(void);
    time_t getDateTimeSecsSince1970TZ(void);
    
    /** Set the time zone
    * @param hours offset
    * @return void
    */
    void setTimeZone(double TZ);
    
    /** Get the time zone
    * @param void
    * @return hours offset
    */
    double getTimeZone(void);

    /** Set the date registers
    * // @param dayOfWeek : day of week
    * @param date
    * @param month
    * @param year
    */
    void setDate(int date, int month, int year);

    /** Set the time registers
    * @param hours
    * @param minutes
    * @param seconds
    */
    void setTime(int hours, int minutes, int seconds);

    /** Read the date and time and set system clock
    */
    void setSystemClock(void);

    void readDateTime(void);

    /** Set the time
    * @param hours
    * @param minutes
    * @param seconds
    * @param day
    * @param month
    * @param year
    */
    void setDateTime(int day, int month, int year, int hours, int minutes, int seconds);

    /** Set the time registers
    * @param struct tm *time
    */
    void setDateTime(struct tm *time);

    /** Get the time registers
    * @return struct tm *time
    */
    void getDateTime(struct tm *time);
    struct tm *getDateTime(void);

    int getDayOfWeek(int date, int month, int year);
    
    char * getDayOfWeekName(void);

    char * getFormatedDateTime(char *format);

    bool error;

    bool getSummerTime(void);

    int dayOfYearC(void);
    char * getSunRise(void);
    char * getSunSet(void);
    char * getDayLength(void);    
    int getSunRiseMinute(void);
    int getSunSetMinute(void);
    bool checkSunRise(void);

    void substr(char *s, char *d, int pos, int len);
    char * substr(char *s, int pos, int len);

    int bcd2dec(int k); // bcd to decimal conversion
    int dec2bcd(int k); // decimal to bcd conversion
     
private :
//    I2C i2c;
//    char ARM_RTC_Address;
//    void decodeTime(int regHours, int regMinutes, int regSeconds,int *Hours, int *Minutes, int *Seconds);
//    void decodeDate(int regDate,int regMonth, int regYear, int *date, int *month, int *year);

    char buffer[40];

    uint32_t timeZoneOffset;
    char     timeZoneName[32];
    bool     dayLightSaving;

    struct tm t;
//    t.tm_sec = -1;   
//    t.tm_min = -1;    
//    t.tm_hour = -1;   
//    t.tm_mday = -1;
//    t.tm_wday = -1;
//    t.tm_yday = -1;   
//    t.tm_mon = -1;    
//    t.tm_year = -1; 

    time_t secondsEpoch;

    struct DateTime{
       int year;
       int mon;
       int mday;
       int wday;
       int yday;
       int hour;
       int min;
       int sec;
    };
 

};

#endif