#include "mbed.h"
#include "SHA256.h"
#include "rtos.h"
#include <string>


//------------------------ Define Inputs -------------------------------------//

// Photointerrupter input pins
#define I1pin D2
#define I2pin D11
#define I3pin D12

// Incremental encoder input pins
#define CHA   D7
#define CHB   D8

// Motor Drive output pins  //Mask in output byte
#define L1Lpin D4           //0x01
#define L1Hpin D5           //0x02
#define L2Lpin D3           //0x04
#define L2Hpin D6           //0x08
#define L3Lpin D9           //0x10
#define L3Hpin D10          //0x20

// Mapping from sequential drive states to motor phase outputs
/*
State   L1  L2  L3
0       H   -   L
1       -   H   L
2       L   H   -
3       L   -   H
4       -   L   H
5       H   L   -
6       -   -   -
7       -   -   -
*/


//---------------------------- Global Variables ------------------------------//

// For Motor
// Drive state to output table
const int8_t driveTable[] = { 0x12, 0x18, 0x09, 0x21, 0x24, 0x06, 0x00, 0x00 };

// Mapping from interrupter inputs to sequential rotor states.
// 0x00 and 0x07 are not valid
const int8_t stateMap[] = {0x07,0x05,0x03,0x04,0x01,0x00,0x02,0x07};

// Alternative if phase order of input or drive is reversed
//const int8_t stateMap[] = {0x07,0x01,0x03,0x02,0x05,0x00,0x04,0x07};

// Phase lead to make motor spin
const int8_t lead = -2;  //2 for forwards, -2 for backwards
//enum messageName ={"Hash_Rate","Nonce","Key"};

//Status LED
DigitalOut led1(LED1);

//Photointerrupter inputs
InterruptIn I1(I1pin);
InterruptIn I2(I2pin);
InterruptIn I3(I3pin);

//Motor Drive outputs
DigitalOut L1L(L1Lpin);
DigitalOut L1H(L1Hpin);
DigitalOut L2L(L2Lpin);
DigitalOut L2H(L2Hpin);
DigitalOut L3L(L3Lpin);
DigitalOut L3H(L3Hpin);

int8_t orState = 0;    //Rotot offset at motor state 0

//Initialise the serial port
RawSerial pc(SERIAL_TX, SERIAL_RX);
int8_t intState = 0;
int8_t intStateOld = 0;

// Variables for Hash
uint8_t sequence[] = {0x45, 0x6D, 0x62, 0x65, 0x64, 0x64, 0x65, 0x64,
                      0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x73,
                      0x20, 0x61, 0x72, 0x65, 0x20, 0x66, 0x75, 0x6E,
                      0x20, 0x61, 0x6E, 0x64, 0x20, 0x64, 0x6F, 0x20,
                      0x61, 0x77, 0x65, 0x73, 0x6F, 0x6D, 0x65, 0x20,
                      0x74, 0x68, 0x69, 0x6E, 0x67, 0x73, 0x21, 0x20,
                      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

uint64_t* key = (uint64_t*)((int)sequence + 48);
uint64_t* nonce = (uint64_t*)((int)sequence + 56);
uint8_t hash[32];

// For Thread

// Timer
Timer t;

// Threading
Thread commOutT;
Thread commOutT2;

typedef struct{
    uint8_t code;
    uint32_t data;
    } message_t;

// Mail
Mail<message_t, 16> outMessages;

// Queue
char command[19];

Queue <void,8> inCharQ;

uint32_t newKey;//to pass new key
uint32_t keyLala;
uint64_t combinedKey;

Mutex key_mutex;

//------------------------- Motor Functions ----------------------------------//

// Set a given sdrive state
void motorOut(int8_t driveState) {
    //Lookup the output byte from the drive state.
    int8_t driveOut = driveTable[driveState & 0x07];

    //Turn off first
    if (~driveOut & 0x01) L1L = 0;
    if (~driveOut & 0x02) L1H = 1;
    if (~driveOut & 0x04) L2L = 0;
    if (~driveOut & 0x08) L2H = 1;
    if (~driveOut & 0x10) L3L = 0;
    if (~driveOut & 0x20) L3H = 1;

    //Then turn on
    if (driveOut & 0x01) L1L = 1;
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) L2L = 1;
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) L3L = 1;
    if (driveOut & 0x20) L3H = 0;
}

// Convert photointerrupter inputs to a rotor state
inline int8_t readRotorState() {
    return stateMap[I1 + 2 * I2 + 4 * I3];
}

// Basic synchronisation routine
int8_t motorHome() {
    //Put the motor in drive state 0 and wait for it to stabilise
    motorOut(0);
    wait(1.0);

    //Get the rotor state
    return readRotorState();
}

void read() {
    intState = readRotorState();
    if (intState != intStateOld) {
        intStateOld = intState;
        motorOut((intState - orState + lead + 6) % 6);
        //+6 to make sure the remainder is positive
    }
}


// -------------------------- Thread Functions -------------------------------//

void putMessage(uint8_t code, uint64_t data) {
    message_t *pMessage = outMessages.alloc();
    pMessage->code = code;
    pMessage->data = data;
    outMessages.put(pMessage);
}

void commOutFn() {
    while(1){
        osEvent newEvent = outMessages.get();
        message_t *pMessage = (message_t*)newEvent.value.p;
        //pc.printf("%s = 0x%016x \r \n",messageName[pMessage->code], pMessage->data);
        if(pMessage->code == 1) {
            pc.printf("Nonce = 0x%016x \r \n", pMessage->data);
        } else if(pMessage->code == 0){
            pc.printf("Hash Rate = %d \r \n", pMessage->data);
        } else if(pMessage->code == 2){
            pc.printf("Key = 0x%016x \r \n", pMessage->data);
        } 
     

        outMessages.free(pMessage);
    }
}

void serialISR() {
    uint8_t newChar = pc.getc();
    inCharQ.put((void*)newChar);
}

void keyBuffer() {
    pc.attach(&serialISR);
    int i = 0;
    while(1){
        osEvent newEvent =inCharQ.get();
        uint8_t newchar = (uint8_t)newEvent.value.p;
        command[i] = newchar;
        if (newchar =='r' && command[i-1] == '\\' ) { // detect end of line
            command[i]='0';
            if (command[0] == 'K') {
                key_mutex.lock();
                sscanf(command, "%*c %8x %8x", key, (uint64_t*)((int)key+4) );
                 pc.printf("Key: %8x%8x \r\n", *key, *((uint64_t*)(int)key+4));
                key_mutex.unlock();
            }
            i = -1;
        }
        i++;
        if(i>=sizeof(command)-1){
            i=0;
            putMessage(2,0);
            }
    }
}


//-------------------------------- Main --------------------------------------//

int main() {
    newKey=*key;

    commOutT.start(commOutFn);
    commOutT2.start(keyBuffer);

    // Run the motor synchronisation
    orState = motorHome();
    // orState is subtracted from future rotor state inputs to align rotor and motor states

    // Interrupt, if the motor is moved it will enter the interrupt and keep rotating
    I1.rise(&read);
    I2.rise(&read);
    I3.rise(&read);
    I1.fall(&read);
    I2.fall(&read);
    I3.fall(&read);

    // Poll the rotor state and set the motor outputs accordingly to spin the motor

    // Create an old ref to nonce
    uint64_t oldNonce = *nonce;
    uint32_t hashRate;

    while (1) {

        // Start the timer
        t.start();

        // Compute hash key after nonce is added by 1 every time
        SHA256::computeHash(hash, (uint8_t*)((int)sequence), sizeof(sequence));

        if(t.read() > 1) {
            t.stop();
            hashRate = ((int)*nonce - (int)oldNonce) / t.read();
            oldNonce = *nonce;
            putMessage(0, hashRate);
            t.reset();
        }

        if(hash[0] == 0 && hash[1] == 0) {
            putMessage(1, (int)*nonce);
        }
        *nonce= (int)*nonce+1;
    }
}

