/**
 * @file LsrModule.h
 * @author LS Research LLC
 * @version 1.0
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * UART communication API for LS Research's ProFLEX01 and SiFLEX02 ModFLEX Shields.
 */

/**
 * Modified by Mark Harris at SAIT.ca
 * Improved functionality for mbed
 */

#ifndef LsrModule_h
#define LsrModule_h

#include <inttypes.h>
#include "mbed.h"

#ifndef NULL
    #define NULL    0
#endif

#define SIFLEX02  ///< Use with a SiFLEX02 ModFLEX Shield (900MHz)
//#define PROFLEX01  ///< Use with a ProFLEX01 ModFLEX Shield (2.4GHz)

/**
* Message Index Defintions.
*
*/
#define LSR_MODULE_MSG_START_BYTE_INDEX                                 0
#define LSR_MODULE_MSG_LENGTH_BYTE_INDEX                                1
#define LSR_MODULE_MSG_TYPE_BYTE_INDEX                                  2

/**
* Message Byte Types.
*
*/
#define LSR_MODULE_SERIAL_MSG_START_BYTE                                0x01
#define LSR_MODULE_SERIAL_MSG_END_BYTE                                  0x04

/**
* Rx Message Type Limits.
*
*/
#define LSR_MODULE_MIN_SERIAL_RX_MSG_TYPE                               0x81
#define LSR_MODULE_MAX_SERIAL_RX_MSG_TYPE                               0xC5

/**
* LSR Module Host TX Message Definitions.
*
*/
#define LSR_MODULE_QUERY_VERSION_MSG_TYPE                               0x01
#define LSR_MODULE_SET_SECURITY_TX_FRAME_COUNTER_MSG_TYPE               0x0C
#define LSR_MODULE_QUERY_SECURITY_TX_FRAME_COUNTER_MSG_TYPE             0x0D
#define LSR_MODULE_SET_BASIC_RF_SETTINGS_MSG_TYPE                       0x10
#define LSR_MODULE_QUERY_BASIC_RF_SETTINGS_MSG_TYPE                     0x11
#define LSR_MODULE_SAVE_SETTINGS_TO_NV_MEMORY_MSG_TYPE                  0x12
#define LSR_MODULE_RESET_REQUEST_MSG_TYPE                               0x13
#define LSR_MODULE_QUERY_SUPPLY_VOLTAGE_MSG_TYPE                        0x14
#define LSR_MODULE_QUERY_STATISTICS_MSG_TYPE                            0x15
#define LSR_MODULE_CLEAR_STATISTICS_MSG_TYPE                            0x16
#define LSR_MODULE_SET_HOST_DATA_RATE_MSG_TYPE                          0x18
#define LSR_MODULE_SET_RF_DATA_RATE_MSG_TYPE                            0x19
#define LSR_MODULE_SEND_SIMPLE_SHORT_RF_DATA_PACKET_MSG_TYPE            0x20
#define LSR_MODULE_SEND_ADVANCED_SHORT_RF_DATA_PACKET_MSG_TYPE          0x22
#define LSR_MODULE_SEND_SIMPLE_LONG_RF_DATA_PACKET_MSG_TYPE             0x24
#define LSR_MODULE_SEND_ADVANCED_LONG_RF_DATA_PACKET_MSG_TYPE           0x26
#define LSR_MODULE_SEND_SIMPLE_REPEATED_RF_DATA_PACKET_MSG_TYPE         0x2A
#define LSR_MODULE_CHANNEL_ENERGY_SCAN_MSG_TYPE                         0x44
#define LSR_MODULE_QUERY_HOST_INTERFACE_CONFIGURATION                   0x51


/**
* LSR Module Host RX Message Definitions.
*
*/
#define LSR_MODULE_RESPONSE_HOST_INTERFACE_CONFIGURATION                0xD1


/**
* LSR Module Settings Definitions.
*
*/
#define LSR_MODULE_TX_POWER_LEVEL_MIN                                   0
#define LSR_MODULE_SECURITY_OVERHEAD                                    14
#define LSR_MODULE_MIN_SERIAL_RX_MSG_LENGTH                             5
#define LSR_MODULE_MAX_SERIAL_RX_MSG_LENGTH                             131

#ifdef PROFLEX01
    #define LSR_MODULE_RF_CHANNEL_MIN                                   11
    #define LSR_MODULE_RF_CHANNEL_MAX                                   25
    #define LSR_MODULE_TX_POWER_LEVEL_MAX                               19
    #define LSR_MODULE_SIMPLE_SHORT_RF_DATA_LENGTH                      98
    #define LSR_MODULE_ADVANCED_SHORT_RF_DATA_LENGTH                    98
    #define LSR_MODULE_SIMPLE_LONG_RF_DATA_LENGTH                       98
    #define LSR_MODULE_ADVANCED_LONG_RF_DATA_LENGTH                     98
    #define LSR_MODULE_MAX_SERIAL_TX_MSG_LENGTH                         115
#elif defined SIFLEX02
    #define LSR_MODULE_RF_CHANNEL_MIN                                   1
    #define LSR_MODULE_RF_CHANNEL_MAX                                   10
    #define LSR_MODULE_TX_POWER_LEVEL_MAX                               21
    #define LSR_MODULE_SIMPLE_SHORT_RF_DATA_LENGTH                      112
    #define LSR_MODULE_ADVANCED_SHORT_RF_DATA_LENGTH                    110
    #define LSR_MODULE_SIMPLE_LONG_RF_DATA_LENGTH                       100
    #define LSR_MODULE_ADVANCED_LONG_RF_DATA_LENGTH                     98
    #define LSR_MODULE_MAX_SERIAL_TX_MSG_LENGTH                         121
#else
    #error "Incorrect module type"
#endif

/**
* Receiver ConfigurationBitmask Definitions.
*
*/
#define LSR_MODULE_RX_CONFIG_ALLOW_BROADCAST_ADDRESS_BITMASK            0x01
#define LSR_MODULE_RX_CONFIG_ALLOW_BROADCAST_PAN_BITMASK                0x02
#define LSR_MODULE_RX_CONFIG_PROMISCUOUS_MODE_BITMASK                   0x04
#define LSR_MODULE_RX_CONFIG_ALLOW_SECURED_PACKETS_MODE_BITMASK         0x08
#define LSR_MODULE_RX_CONFIG_MAX                                        (LSR_MODULE_RX_CONFIG_ALLOW_BROADCAST_ADDRESS_BITMASK + LSR_MODULE_RX_CONFIG_ALLOW_BROADCAST_PAN_BITMASK + LSR_MODULE_RX_CONFIG_PROMISCUOUS_MODE_BITMASK + LSR_MODULE_RX_CONFIG_ALLOW_SECURED_PACKETS_MODE_BITMASK)

/**
* Transmit Options Bitmask Definitions.
*
*/
#define LSR_MODULE_TX_OPTIONS_RETRIES_ACKS_BITMASK                      0x01
#define LSR_MODULE_TX_OPTIONS_USE_SECURITY_BITMASK                      0x02
#define LSR_MODULE_TX_OPTIONS_MAX                                       (LSR_MODULE_TX_OPTIONS_RETRIES_ACKS_BITMASK + LSR_MODULE_TX_OPTIONS_USE_SECURITY_BITMASK)

/**
* Host UART Data Rate Definitions.
*
*/
#define LSR_MODULE_HOST_DATA_RATE_1200                                  0
#define LSR_MODULE_HOST_DATA_RATE_2400                                  1
#define LSR_MODULE_HOST_DATA_RATE_4800                                  2
#define LSR_MODULE_HOST_DATA_RATE_9600                                  3
#define LSR_MODULE_HOST_DATA_RATE_19200                                 4
#define LSR_MODULE_HOST_DATA_RATE_38400                                 5
#define LSR_MODULE_HOST_DATA_RATE_57600                                 6
#define LSR_MODULE_HOST_DATA_RATE_115200                                7
#define LSR_MODULE_HOST_DATA_RATE_230400                                8
#define LSR_MODULE_HOST_DATA_RATE_460800                                9
#define LSR_MODULE_HOST_DATA_RATE_921600                                10
#define LSR_MODULE_HOST_DATA_RATE_MAX                                   LSR_MODULE_HOST_DATA_RATE_921600
#define LSR_MODULE_10MS_UART_TIMEOUT                                    239

/**
* RF Data Rate Definitions.
*
*/
#define LSR_MODULE_RF_DATA_RATE_BPSK_40KBPS             0
#define LSR_MODULE_RF_DATA_RATE_OQPSK_SIN_250KBPS       1
#define LSR_MODULE_RF_DATA_RATE_OQPSK_SIN_1MBPS         2
#define LSR_MODULE_RF_DATA_RATE_MAX                     LSR_MODULE_RF_DATA_RATE_OQPSK_SIN_1MBPS

/**
* Channel Energy Scan Duration Definitions.
*
*/
#define LSR_MODULE_SCAN_DURATION_MAX                                    14

/**
* Type Defintions.
*
*/
typedef unsigned int    word_t;

/**
* Word structure.
*
*/
typedef struct _words
{
    uint8_t lb; ///< Low byte.
    uint8_t hb; ///< High byte.
} Words_t;

/**
* Word union.
*
*/
typedef union _wordu
{
    word_t   w; ///< Word.
    uint16_t    u16; ///< Unsigned 16.
    Words_t  ws; ///< Word structure.
} Wordu_t;

/**
* Word structure.
*
*/
typedef struct _dwords
{
    uint8_t lb; ///< Low byte.
    uint8_t mlb; ///< Middle low byte.
    uint8_t mhb; ///< Middle high byte.
    uint8_t hb; ///<  High byte.
} DWords_t;

/**
* Double word union.
*
*/
typedef union _dwordu
{
    uint32_t u32; ///< Unsigned 32.
    DWords_t dws; ///< Double word structure.
} DWordu_t;


class LsrModule : Serial
{
    public:
    uint8_t* pu8RxBuffer; ///< Pointer to private UART receive buffer.
    uint8_t u8RxMsgLength; ///< Length of private UART receive buffer.
    //Functions:
    LsrModule(PinName tx, PinName rx, int baudRate = 19200);
    ~LsrModule();
    bool SubscribeRxMsgCallback(uint8_t u8RxMsgType, void (*callback)(void)); 
    bool UnsubscribeRxMsgCallback(uint8_t u8RxMsgType); 
    void RunHostRxStateMachine(void);
    // Tx Messages:
    // 0x01
    void QueryVersionMsg(void);
    // 0x0C
    void SetSecurityTransmitFrameCounterMsg(uint32_t* pu32FrameCounter);
    // 0x0D
    void QuerySecurityTransmitFrameCounterMsg(void);
    // 0x10
    void SetBasicRfSettingsMsg(uint16_t u16PanId, uint16_t u16AddrShort, uint8_t* pu8AddrLong, uint8_t u8RfChannel, uint8_t u8TxPowerLevel, uint8_t u8ReceiverConfig, uint8_t* pu8SecurityKey);
    // 0x11
    void QueryBasicRfSettingsMsg(void);
    // 0x12
    void SaveSettingsToNVMemoryMsg(void);
    // 0x13
    void ResetRequestMsg(void);
    // 0x14
    void QuerySupplyVoltageMsg(void);
    // 0x15
    void QueryStatisticsMsg(void);
    // 0x16
    void ClearStatisticsMsg(void);
    // 0x18
    void SetHostDataRateMsg(uint8_t u8HostDataRate);
    // 0x18
    void SetRfDataRateMsg(uint8_t u8RfDataRate);
    // 0x20
    void SendSimpleShortAddrRfDataPacketMsg(uint8_t* pu8Data, uint8_t u8DataLength, uint16_t u16DestAddress, uint8_t u8TxOptions, uint8_t u8PacketId);
    // 0x22
    void SendAdvancedShortAddrRfDataPacketMsg(uint8_t* pu8Data, uint8_t u8DataLength, uint16_t u16DestPanId, uint16_t u16DestAddress, uint8_t u8TxOptions, uint8_t u8PacketId);
    // 0x24
    void SendSimpleLongAddrRfDataPacketMsg(uint8_t* pu8Data, uint8_t u8DataLength, uint8_t* pu8DestAddress, uint8_t u8TxOptions, uint8_t u8PacketId);
    // 0x26
    void SendAdvancedLongAddrRfDataPacketMsg(uint8_t* pu8Data, uint8_t u8DataLength, uint16_t u16DestPanId, uint8_t* pu8DestAddress, uint8_t u8TxOptions, uint8_t u8PacketId);
    // 0x44
    void ChannelEnergyScanMsg(uint16_t u16ChannelMask, uint8_t u8ScanDuration);
    // 0x51
    void QueryHostInterfaceConfiguration(void);

    private:
    // Variables
    uint8_t u8RxReadByte;
    uint8_t u8ForLoopCounter;
    uint8_t u8UartRxBufferIndex;
    uint8_t u8UartTxBufferIndex;
    uint8_t u8TxMsgChecksum;///< Transmit message checksum.
    uint8_t au8UartRxBuffer[LSR_MODULE_MAX_SERIAL_RX_MSG_LENGTH];///< Receive buffer for UART communication from module.
    uint8_t au8UartTxBuffer[LSR_MODULE_MAX_SERIAL_TX_MSG_LENGTH];///< Transmit buffer for UART communication to module.
    bool bUartMethodFlag;

    
    // Function Pointers
    void (LsrModule::*ptrHostState)(void);///< Runs uart UART receive state machine.
    void (LsrModule::*ptrHostProcessCallbackState)(uint8_t*);///< Used to process received UART messages.
    
    
    // Functions
    void AddSerialMsgHeader(uint8_t u8MsgType, uint8_t u8MsgLength);
    void AddSerialByteToMsgBuffer(uint8_t u8Data);
    void AddSerialMsgTrailer(void);
    void WriteSerialMsg(void);
    void HostProcessCallbackMsgStart (uint8_t* pu8MsgBuffer, uint8_t u8Length);
    // Rx Message Callbacks
    

    // Rx functions
    bool ValidMsgLengthAndType(uint8_t u8MsgType, uint8_t u8MsgLength);
    bool ValidRxChecksum(uint8_t* pu8MsgBuffer, uint8_t u8MsgLength);
    void SerialFlush(void);
    bool SerialAvailable(void);
    uint8_t SerialRead(void);
    
    
    // Rx State Machine States
    void SerialRxCleanupRestart(void);
    void HostRxWaitForMsgStartByteState(void);
    void HostRxGetMsgLengthState(void);
    void HostRxGetMsgTypeState(void);
    void HostRxWaitToGetRestOfMsgState(void);
    void HostRxValidateMsgState(void);
    void HostRxGoodMsgState(void);
};

#endif