/*  Librairie de gestion d'un DMD RGB 128x32 pixels
*   - 256 Couleurs mode RGB332 et Monochrome -
*   decembre 2017 par Christophe Girardot Aka Aganyte
*/

/* Ouverture de la liaison série */
Serial pc(SERIAL_TX, SERIAL_RX, 9600);

// Mémoire vidéo de 128x32 pixels avec 1 octet par pixel
unsigned char Display_Ram[4096];

// Mémoire vidéo Monochrome de 128x32 pixels avec 1 bit par pixel (Octet)
unsigned char Mono_Display_Ram[512];

// Mémoire vidéo Monochrome de 128x32 pixels avec 1 bit par pixel(booléen)
bool Bool_Display_Ram[4096];

// Mémoire de stockage des Scrolling de la SD
unsigned char Scrolling_Buffer[40960];

// Paramètres réglables depuis la SD (avec les valeurs par défaut au démarrage)
unsigned char Replay=1; 
unsigned char Speed=5;
unsigned char Sleep=5;
unsigned char Color_R=1;
unsigned char Color_G=0;
unsigned char Color_B=0;
unsigned char Up = 1;
unsigned char Left = 1;
unsigned char Brightness_level = 9;
char Command[6];

unsigned int Up_Ligne_adress = 0; // Evite une re-déclaration
unsigned int Bottom_Ligne_adress = 0; // Evite une re-déclaration

/* Masque de décodage */
// En RGB332 (le Bleu possède moins de nuance car l'oeil humain les distinguent moins)
// Seulement 7 niveau par masque car la nuance la plus basse correspond à une led éteinte
// R => 0x11100000
// G => 0x00011100
// B => 0x00000011
unsigned char Red_Mask[7] = {
    0b00100000,  // nuance 0, C
    0b01000000,  // nuance 1, B
    0b01100000,  // nuance 2, B ET C
    0b10000000,  // nuance 3, A
    0b10100000,  // nuance 4, A ET C
    0b11000000,  // nuance 5, A ET B
    0b11100000,  // nuance 6, A ET B ET C
};
unsigned char Green_Mask[7] = {
    0b00000100, // nuance 0, C
    0b00001000, // nuance 1, B
    0b00001100, // nuance 2, B ET C
    0b00010000, // nuance 3, A
    0b00010100, // nuance 4, A ET C
    0b00011000, // nuance 5, A ET B
    0b00011100, // nuance 6, A ET B ET C
};
unsigned char Blue_Mask[7] = { // On compte en double pour créer 7 niveaux (Comme R et G)
    0b00000000, // nuance 0, rien
    0b00000001, // nuance 1, B
    0b00000001, // nuance 2, B
    0b00000010, // naunce 3, A
    0b00000010, // nuance 4, A
    0b00000011, // nuance 5, A ET B
    0b00000011, // nuance 6, A ET B
};

void Image(unsigned int Time); // Prédéclaration de la fonction Image

/* Fonction qui efface la mémoire vidéo RGB
*/
void Clear_Display_Ram()
{
    for(unsigned int count = 0 ; count < 4096; count ++)
        Display_Ram[count] = 0;
}

/* Fonction qui convertit le buffer monochrome en octet vers le buffer monochrome en booléen
*
*/
void Convert_Mono_To_Bool()
{
    for(unsigned int count = 0; count < 512 ; count++) // lecture des 512 octets
    {
        unsigned char buffer = Mono_Display_Ram[count]; // lecture d'un octet
        for(unsigned char bit = 0; bit < 8; bit++) // remplir 8 booléen 
        {
            if( (buffer & 0x80) == 0x80) // vers la gauche
                Bool_Display_Ram[(count*8)+bit] = 1;
            else
                Bool_Display_Ram[(count*8)+bit] = 0;
            buffer <<= 1;
        }
    }
}

/* Fonction de convertion du buffer booléen vers le buffer RGB avec superposition 
*
*/
void Convert_Bool_To_RGB()
{
    Convert_Mono_To_Bool();

    for(unsigned int count = 0; count < 4096; count++) 
    {
        if( (Color_R == 1) & (Bool_Display_Ram[count] == 1) )
            Display_Ram[count] = Display_Ram[count] | Red_Mask[6];

        if( (Color_G == 1) & (Bool_Display_Ram[count] == 1) )
            Display_Ram[count] = Display_Ram[count] | Green_Mask[6];

        if( (Color_B == 1) & (Bool_Display_Ram[count] == 1) )
            Display_Ram[count] = Display_Ram[count] | Blue_Mask[6];
        
        if( (Color_R == 0) & (Color_G == 0) & (Color_B == 0) & (Bool_Display_Ram[count] == 1) )
            Display_Ram[count] = 0;
    }
}

/* sous programme de gestion des lignes (tous les modes) */
void Line(unsigned char number)
{
    D = (number>>3) & 0x01;
    C = (number>>2) & 0x01;
    B = (number>>1) & 0x01;
    A = number & 0x01;
}

/* Envoie de deux octets RGB avec choix de la nuance actuelle (mode RGB332) */
void Transfer(unsigned char Up_RGB, unsigned char Down_RGB, unsigned char shade)
{
    CLK = 0;

    if(Up_RGB!=0)
    {
        // Préparation de R1,G1 et B1
        R1 = (Up_RGB & Red_Mask[shade]) == Red_Mask[shade];
        G1 = (Up_RGB & Green_Mask[shade]) == Green_Mask[shade];
        B1 = (Up_RGB & Blue_Mask[shade]) == Blue_Mask[shade];
    }
    else
    {
        // Préparation de R1,G1 et B1
        R1 = 0;
        G1 = 0;
        B1 = 0;       
    }
    
    if(Down_RGB!=0)
    {
        // Préparation de R2,G2 et B2
        R2 = (Down_RGB & Red_Mask[shade]) == Red_Mask[shade];
        G2 = (Down_RGB & Green_Mask[shade]) == Green_Mask[shade];
        B2 = (Down_RGB & Blue_Mask[shade]) == Blue_Mask[shade];
    }
    else
    {
        // Préparation de R1,G1 et B1
        R2 = 0;
        G2 = 0;
        B2 = 0;       
    }
    CLK = 1;  // Front montant = Envoie de la donnée
}

/* Envoyer et afficher 256 pixels sur le dot (deux lignes) depuis 
*  la mémoire vidéo avec sélection de la nuance (mode RGB332)
*/
void Send_256pixels(unsigned char ligne, unsigned char shade)
{ 
    LATCH = 0 ; // Préparer le front montant pour le Latch
    
    Up_Ligne_adress = ligne * 128;             // calcul du décalage en fonction de la ligne (moitié haute de l'écran)
    Bottom_Ligne_adress = Up_Ligne_adress + 2048; // calcul du décalage en fonction de la ligne (moitié basse de l'écran)
    
    /* Envoie de 256 octets (128 octets par ligne) */
    for(unsigned char Count_SPI2 = 0 ; Count_SPI2<128 ; Count_SPI2++ )
        Transfer(Display_Ram [Count_SPI2 + Up_Ligne_adress], Display_Ram [Count_SPI2 + Bottom_Ligne_adress], shade);

    /* Pour les nuances 4,5,6 on coupe la ligne avec le timer */
    while( Timer_Refresh.read_us() < Time_Shade[shade]); 
    Timer_Refresh.reset(); // Redemarrer le timer pour le prochain tour
    
    EN = 1;                         // Désactiver la ligne
    Line(ligne);                    // Sélectionner la ligne
    LATCH = 1;                      // Transferer la donnée sur les latchs
    EN = 0;                         // Activer la ligne
    if(shade<4) // Pour les nuances 0,1,2 et 3 on coupe la ligne après un wait (trop rapide pour la méthode du timer)
    {
        wait_us(Time_Shade[shade]);     // Gestion du temps d'affichage de la ligne en fonction de sa nuance
        EN = 1;                         // Désactiver la ligne 
    }
}

/* Rafraichir l'écran 1 fois (mode RGB332) */
void refresh()
{
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 0
        Send_256pixels(ligne,6);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 1
        Send_256pixels(ligne,5);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 2
        Send_256pixels(ligne,4);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 3
        Send_256pixels(ligne,3);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 4
        Send_256pixels(ligne,2);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 5
        Send_256pixels(ligne,1);
    for (unsigned char ligne = 0; ligne<16 ; ligne++) // Envoyer 16x2 lignes de la nuance 6
        Send_256pixels(ligne,0);
    EN = 1; // Désactiver la ligne en fin de rafraichissement (sinon, elle brille plus que les autres....c'est moche)
}

/* Rafraichir l'écran avec la mémoire vidéo (mode monochrome) pendant x secondes (Sleep) */
void refresh_mono()
{
    Clear_Display_Ram(); // Effacer la mémoire vidéo RGB (sinon superposition d'image)
    
    Convert_Bool_To_RGB(); // Convertir le buffer mono avec superposition dans le buffer couleur
    
    Image(Sleep); // Afficher l'image
}

/* Fonction affichage d'un scrolling vertical stocké dans le micro-controleur 
*  height => hauteur de l'image en pixels
*  Array => nom de l'image 
*/
void Scrolling(unsigned int height, const unsigned char *Array)
{
    pc.printf("Affichage d'un scrolling vertical de 128x%d pixels depuis le STM32\n",height);
    for(unsigned char decal = 0; decal<(height-32); decal++) 
    {
        // Remplir la mémoire vidéo avec une image
        for(unsigned int count = 0; count<4096; count++) 
            Display_Ram[count] = Array[(decal*128)+count];
        // Affichage de la mémoire vidéo 
        refresh();
        refresh();
        refresh();
    }
}

/* Fonction affichage d'un scrolling horizontal stocké dans le micro-controleur
*  width => largeur de l'image en pixels
*  Array => nom de l'image 
*/
void Horizontal_Scrolling(unsigned int width, const unsigned char *Array)
{
    pc.printf("Affichage d'un scrolling horizontal de %dx32 pixels depuis le STM32\n",width);
    for(unsigned int image = 0 ; image < width-128; image++)
    {
        // Remplir la mémoire vidéo avec une ligne
        for(unsigned char ligne = 0; ligne<32; ligne++) 
        {
           for(unsigned char count = 0 ; count<128 ; count++)
             Display_Ram[count+(ligne*128)] = Array[image+count+(width*ligne)];
        }
        // Affichage de la mémoire vidéo
        refresh();
        refresh();
        refresh();
    }
}

/* Fonction affichage d'une animation stockée dans le micro-controleur 
*  Size => nombre de trame
*  Array => nom de l'animation
*/
void Animation(unsigned int size, const unsigned char *Array)
{
    pc.printf("Affichage d'une animation de %d images depuis le STM32\n",size);
    for(unsigned char decal = 0; decal<size; decal++) 
    {
        // Remplir la mémoire vidéo avec une image
        for(unsigned int count = 0; count<4096; count++) 
            Display_Ram[count] = Array[(decal*4096)+count];
        // Affichage de la mémoire vidéo 
        refresh();
        refresh();
        refresh();
    }
}

/* Fonction affichage d'une image stockée dans buffer vidéo
*  time -> nombre de secondes d'affichage de l'image
*/
void Image(unsigned int time)
{
    timer.reset();
    timer.start();
    while(timer.read_ms() < 1000*time) 
        refresh();
    timer.stop();
}

/* Fonction écriture 2 lignes de 8 caractères en Mononchrome */
void print_2_8(const char *chaine,bool Line_2)
{
  unsigned int FontPosData,select = 0;
  unsigned char Caractere = 0, count = 8;

  if(Line_2 == true) select = 256;
  
  while(count!=0)
  {
    FontPosData=((*chaine++)-32)*32;  // FontPosData détermine la position dans font.h du caractére à afficher.
    for(unsigned char Count=0 ; Count<16; Count++) // Font de 16 lignes de haut
    {
      for(unsigned char cT=0; cT<2; cT++) // Font de 2 octets de large
      {
        Mono_Display_Ram[select+cT+(16*Count)+(2*Caractere)] = Robot[(FontPosData+2*Count)+cT];
      }
    }
    Caractere++; // Passer au caractère suivant
    count--;
  }
}

/* Effacer la mémoire vidéo Monochrome */
void Clear_Mono_Buffer()
{
  for ( unsigned int count = 0; count < 512 ; count++) 
    Mono_Display_Ram[count] = 0x00;
}

/* Rafraichir l'écran avec la mémoire Monochrome pendant X millisecondes (en Blanc si non précisé) */
void Refresh_in_seconds()
{
  timer.reset(); // Remise à zéro du timer
  timer.start(); // Démarrer le timer
  while(timer.read_ms() < (1000*Sleep) ) // Tant que le timer n'a pas atteind la valeur
    refresh_mono(); // Rafraichir l'écran
  timer.stop(); // Arreter le timer
  Clear_Mono_Buffer(); // Effacer la mémoire vidéo
}

/* Inversion de 8 bits pour le mode 2 lignes de 16 caractères (font inversée) */
unsigned char bit_reverse8(unsigned char n)
{
   n = (n << 4) | (n >> 4);
   n = ((n << 2) & 0xCC) | ((n >> 2) & 0x33);
   n = ((n << 1) & 0xAA) | ((n >> 1) & 0x55);
   return n;
}

/* Fonction écriture d'une ligne en RAM par superposition avec choix de la ligne (mode 4 lignes de 16 caractères) */
void print_4_16(const char *chaine, unsigned char line)
{
  unsigned int select = 0;
  if(line > 2)
  {
    select = 256;
    line-=2;
  }
  unsigned int FontPosData;
  unsigned char Count, Buffer, Caractere = 0;
  while(*chaine !='\0')
  {
    FontPosData=((*chaine++)-32)*8; // FontPosData détermine la position dans font.h du caractére à afficher.

    for(Count=0 ; Count<8; Count++) // Font de 8 lignes de haut
    {
      Buffer = bit_reverse8(Font88[FontPosData+Count]);
      if(line == 1) Mono_Display_Ram[select+(16*Count)+Caractere] = Mono_Display_Ram[select+(16*Count)+Caractere] | Buffer;
      else Mono_Display_Ram[select+128+(16*Count)+Caractere] = Mono_Display_Ram[select+128+(16*Count)+Caractere] | Buffer;
    }
    Caractere++; // Passer au caractère suivant
  }
}

/* Fonction affichage de la date et de l'heure */
void Clock()
{
    char buffer[16];
    unsigned char Sleep_Buffer = Sleep;
    unsigned char Run_in_second = Sleep;
    Sleep = 1;
    while(1)
    {
        Clear_Mono_Buffer(); // Effacer la mémoire vidéo
        time(&Time);                        // Mettre à jour l'heure
        print_4_16("                ",1);   // Ligne 1
        strftime(buffer, 16,"    %d/%m/%y", localtime(&Time)); // Date au format JJ/MM/AA      
        print_4_16(buffer,2);               // Ligne 2
        print_4_16("                ",3);   // Ligne 3
        //strftime(buffer, 16, "  %I:%M:%S %p", localtime(&Time)); // heure au format 0-12
        strftime(buffer, 16,"    %H:%M:%S", localtime(&Time)); // heure au format 0-24  
        print_4_16(buffer,4); // Ligne 4
        refresh_mono(); // Rafraichir l'écran
        Run_in_second--;
        if(Run_in_second ==0)
            break;
    }
    Sleep = Sleep_Buffer;
    Clear_Mono_Buffer(); // Effacer la mémoire vidéo
}

/* Fonction affichage de la temperature */
void Temperature()
{      
        char buffer[16];
        print_4_16("  TEMPERATURE   ",1); // Ligne 1
        print_4_16("                ",2); // Ligne 2
        sprintf(buffer,"   %.2f DEGRES", (adc_temp.read()*100));
        print_4_16(buffer,3);             // Ligne 3
        print_4_16("                ",4); // Ligne 4
        Refresh_in_seconds();  // Rafraichir l'écran pendant une seconde (2000 millisecondes)
}

/* Fonction affichage de la tension de la pile de sauvegarde */
void Backup()
{
        char buffer[16];
        print_4_16("     PILE       ",1); // Ligne 1
        print_4_16(" DE SAUVEGARDE  ",2); // Ligne 2
        sprintf(buffer,"   %.2f VOLT", adc_vbat.read());
        print_4_16(buffer,3);             // Ligne 3
        print_4_16("                ",4); // Ligne 4
        Refresh_in_seconds();  // Rafraichir l'écran pendant une seconde (2000 millisecondes)
}

/* Fonction affichage de l'intro */
void Intro()
{
        pc.printf("Affichage de l'intro pendant %d secondes avec R=%d,G=%d,B=%d\n",Sleep,Color_R,Color_G,Color_B);
        print_4_16(" DIY DMD-CLOCK ",1); // Ligne 1
        print_4_16(" 128x32 Pixels ",2); // Ligne 2
        print_4_16("v0.1 256 colors",3); // Ligne 3
        print_4_16("By Aganyte/Iro ",4); // Ligne 4
        Refresh_in_seconds();  // Rafraichir l'écran pendant une seconde (5000 millisecondes)
}

/* Remplir la mémoire vidéo avec une Mire */
void Mire()
{
    for(unsigned char ligne =0; ligne < 8; ligne++)
    {
        unsigned char bit = 0;
        while(bit<16) {
            Display_Ram[(128*ligne)+bit] = 0;
            bit++;
        }
        for(unsigned char shade = 2; shade<9; shade++) {
            while(bit<(16*shade)) {
                Display_Ram[(128*ligne)+bit] = Red_Mask[shade-2];
                bit++;
            }
        }
    }
    for(unsigned char ligne =0; ligne < 8; ligne++)
    {
        unsigned char bit = 0;
        while(bit<16) {
            Display_Ram[(128*ligne)+bit] = 0;
            bit++;
        }
        for(unsigned char shade = 2; shade<9; shade++) {
            while(bit<(16*shade)) {
                Display_Ram[(128*ligne)+bit] = Red_Mask[shade-2];
                bit++;
            }
        }
    }
    for(unsigned char ligne =0; ligne < 8; ligne++)
    {
        unsigned char bit = 0;
        while(bit<16) {
            Display_Ram[1024+(128*ligne)+bit] = 0;
            bit++;
        }
        for(unsigned char shade = 2; shade<9; shade++) {
            while(bit<(16*shade)) {
                Display_Ram[1024+(128*ligne)+bit] = Green_Mask[shade-2];
                bit++;
            }
        }
    }
    for(unsigned char ligne =0; ligne < 8; ligne++)
    {
        unsigned char bit = 0;
        while(bit<16) {
            Display_Ram[2048+(128*ligne)+bit] = 0;
            bit++;
        }
        for(unsigned char shade = 2; shade<9; shade++) {
            while(bit<(16*shade)) {
                Display_Ram[2048+(128*ligne)+bit] = Blue_Mask[shade-2];
                bit++;
            }
        }
    }
    for(unsigned char ligne =0; ligne < 8; ligne++)
    {
        unsigned char bit = 0;
        while(bit<16) {
            Display_Ram[3072+(128*ligne)+bit] = 0;
            bit++;
        }
        for(unsigned char shade = 2; shade<9; shade++) {
            while(bit<(16*shade)) {
                Display_Ram[3072+(128*ligne)+bit] = Red_Mask[shade-2] | Green_Mask[shade-2] | Blue_Mask[shade-2];
                bit++;
            }
        }
    }
}