#include "mbed.h"
#include "BiQuad.h"
#include "math.h"
#include "QEI.h"
# define M_PI           3.14159265358979323846  /* pi */

Serial pc(USBTX, USBRX);                       

//  Input
AnalogIn    emg1(A0);                           // EMG1
AnalogIn    emg2(A1);                           // EMG2
DigitalIn   btn1(PTC6);                         // Button 1 (right from led on FRDM K64)
DigitalIn   btn2(PTA4);                         // Button 2 (left from led on FRDM K64)

//  Output
DigitalOut led1(LED_RED);                       // LED output
DigitalOut led2(LED_BLUE);
DigitalOut led3(LED_GREEN);
PwmOut      pwmpin1(D6);                        // Motor 1 output 
DigitalOut  directionpin1(D7);
PwmOut      pwmpin2(D5);                        // Motor 2 output
DigitalOut  directionpin2(D4);  

//  Objects
Ticker      LoopTimer;                          // Timer for loop function 
QEI         Encoder1(D12,D13,NC,64);            // Note that we use X2 encoding, so there are 4200 counts/rotation, as opposed to 8400 from the documentation.
QEI         Encoder2(D10,D11,NC,64);            // Note that we use X2 encoding, so there are 4200 counts/rotation, as opposed to 8400 from the documentation.

//  ---------- Other variables -----------
//  EMG
double      emg1_array[200]={0.00}, emg1_RMS, emg1_threshold=10.0f, // remember the last 200 emg samples and put them in an array, RMS is the RootMeanSquare of the array, threshold will be determined in emg calibrating state 
            emg2_array[200]={0.00}, emg2_RMS, emg2_threshold=10.0f,
            MaxValue;                                               // Maximal value of emg, used in emg calibrating state.
bool        emg1_state, emg2_state;                                 // when muscle contracts, state=true

//  MOTOR
double      q1, q2,                             // Actual motor angles (measured from encoders)
            r1, r2,                             // reference position (setpoint motor angles)
            u1=0, u2=0,                         // Input for SetMotor function 
            homing_q1, homing_q2,               // Motor angles corresponding to Homing State, these angles are determined during calibrating state 
            error1, error2;                     // error = reference angle (setpoint: r1 and r2) - actual angle (q1 and q2)

// Inverse Kinematics 
double      invj00, invj01, invj10, invj11,     // Inverse Jacobian for inverse kinematic model
            vx, vy;                             // Desred velocity in x and y direction

//  Regarding states and state transfers: 
int         TimesCalled;                        // Functions as a timer. After 1e3 times called, 1 second has passed. 
enum        States{Wait, calibratinghoming, calibrating1, calibrating2, homing, movingX, movingY}; // states for finite state machine
States      current_state;                      // Used for switching between states in finite state machine
bool        state_changed = true;               // Used for doing actions when a state is called for the first time

// Make BiQuadChains
BiQuadChain bqc1; // Biquad Chain 1 >> corresponding to emg 1
BiQuad      bqlp_1(0.20657128726265578, 0.41314257452531156, 0.20657128726265578, -0.36952595241514796, 0.19581110146577102);   // Lowpass filter fc=20 Hz
BiQuad      bqhp_1(0.9355261472147435,-1.871052294429487,0.9355261472147435, -1.8668911626483358, 0.8752134262106381);          // Highpass filter fc=200 Hz
BiQuad      bqn50_1(-1.56097580, 0.64130708, 0.82065354, -1.56097580, 0.82065354);                                              // Notch filter fc=50 Hz
BiQuad      bqn100_1(-1.14292982, 0.41273895, 0.70636948, -1.14292982, 0.70636948);                                             // Notch filter fc=100 Hz
BiQuad      bqn150_1(-0.74774808,0.27214502, 0.63607251, -0.74774808, 0.63607251);                                              // Notch filter fc=150 Hz
BiQuad      bqn200_1(-0.36950494,0.19574310, 0.59787155, -0.36950494,0.59787155);                                               // Notch filter fc=200 Hz

BiQuadChain bqc2;// Biquad Chain 2 >> corresponding to emg 2 
BiQuad      bqlp_2(0.20657128726265578, 0.41314257452531156, 0.20657128726265578, -0.36952595241514796, 0.19581110146577102);   // Lowpass filter fc=20 Hz
BiQuad      bqhp_2(0.9355261472147435,-1.871052294429487,0.9355261472147435, -1.8668911626483358, 0.8752134262106381);          // Highpass filter fc=200 Hz
BiQuad      bqn50_2(-1.56097580, 0.64130708, 0.82065354, -1.56097580, 0.82065354);                                              // Notch filter fc=50 Hz
BiQuad      bqn100_2(-1.14292982, 0.41273895, 0.70636948, -1.14292982, 0.70636948);                                             // Notch filter fc=100 Hz
BiQuad      bqn150_2(-0.74774808,0.27214502, 0.63607251, -0.74774808, 0.63607251);                                              // Notch filter fc=150 Hz
BiQuad      bqn200_2(-0.36950494,0.19574310, 0.59787155, -0.36950494,0.59787155);                                               // Notch filter fc=200 Hz

//              -------- FUNCTIONS ----------
// PID functions 
double PID_controller1(double &error){          // Controller for motor 1
    static double error_integral = 0;
    static double error_prev; // initialization with this value only done once!
    static BiQuad LowPassFilter(0.0640, 0.1279, 0.0640, -1.1683, 0.4241);
    double Ts = 1e-3;
    
    // Proportional part
        double Kp = 0.2;
        double u_k = Kp*error;

    // Integral part
        double Ki = 0;
        error_integral = error_integral + error * Ts;
        double u_i = Ki * error_integral;
    
    // Derivative part
        double Kd = 10.1;
        double error_derivative = (error - error_prev)/Ts;
        double filtered_error_derivative = LowPassFilter.step(error_derivative);
        double u_d = Kd * filtered_error_derivative;
        error_prev = error;
    // Sum all parts and return it
    return u_k + u_i + u_d;
}

double PID_controller2(double error){           // Controller for motor 2
    static double error_integral = 0;
    static double error_prev; // initialization with this value only done once!
    static BiQuad LowPassFilter(0.0640, 0.1279, 0.0640, -1.1683, 0.4241);
    double Ts = 1e-3; 
    
    // Proportional part
        double Kp = 0.2;
        double u_k = Kp*error;

    // Integral part
        double Ki = 0;
        error_integral = error_integral + error * Ts;
        double u_i = Ki * error_integral;
    
    // Derivative part
        double Kd = 10.1;
        double error_derivative = (error - error_prev)/Ts;
        double filtered_error_derivative = LowPassFilter.step(error_derivative);
        double u_d = Kd * filtered_error_derivative;
        error_prev = error;
    // Sum all parts and return it
    return u_k + u_i + u_d;
}

// Motor functions 
void SetMotor1(double motorValue){              // Set directionpin and PWMpin for motor 1
    // Given -1<=motorValue<=1, this sets the PWM and direction
    // bits for motor 1. Positive value makes motor rotating
    // clockwise. motorValues outside range are truncated to
    // within range. Motor value = error from the PID controller.
    if (motorValue >=0)
    {
        directionpin1=1;
        if (fabs(motorValue)>0.2f) pwmpin1 = 0.2f;
        else pwmpin1 = fabs(motorValue);
    }
    
    else
    {
        directionpin1=0;
        if (fabs(motorValue)>0.8f) pwmpin1 = 0.8f;
        else pwmpin1 = fabs(motorValue);
    }
}

void SetMotor2(double motorValue){              // Set directionpin and PWMpin for motor 2
    // Given -1<=motorValue<=1, this sets the PWM and direction
    // bits for motor 2. Positive value makes motor rotating
    // clockwise. motorValues outside range are truncated to
    // within range
    if (motorValue >=0) 
    {
        directionpin2=0;
        if (fabs(motorValue)>0.2f) pwmpin2 = 0.2f;
        else pwmpin2 = fabs(motorValue);
    }
    else 
    {
        directionpin2=1;
        if (fabs(motorValue)>0.8f) pwmpin2 = 0.8f;
        else pwmpin2 = fabs(motorValue);
    }
}

// EMG functions
double emg1_Filt()          // Sample and filter EMG1 using biquad filters of biquadchain 1
{
    double y = bqc1.step(emg1.read());   
    return y;
}

double emg2_Filt()          // Sample and filter EMG2 using biquad filters of biquadchain 2
{
    double y = bqc2.step(emg2.read());   
    return y;
}

void emg_GetArray()         // Get 200 samples of EMG1 and EMG2 and put the samples in array
{ 
    for (int n = 0; n < 199; n++)
    {
        emg1_array[n]= emg1_array[n+1];
        emg2_array[n]= emg2_array[n+1];
    }
    emg1_array[199]=emg1_Filt();
    emg2_array[199]=emg2_Filt();
}

double emg_getRMS(double x[])  // Get RootMeanSquare of EMG arrays  
{ 
    double sum=0.0;
    for(int n = 0; n < 200; n++)
    {
        sum += x[n]*x[n];
    }
    double result=sqrt(sum/200);
    return result;
}

bool emg_GetState(double currentRMS, double threshold) // Get the state of EMG
{
    bool state;
    if(currentRMS>threshold)    state=1;    // Muscle contracts
    else                        state=0;    // Muscle relaxes
    return state;
}

void GetMaxEMG(double x, double &MaxVal)    // Get the maximum value of emg, this function is only used during calibration
{
    if(x > MaxVal){
        MaxVal = x;}
}   

void IK(double q1,double q2)                // Inverse Kinematica
{
    double Q1, Q2, V_Q1, V_Q2, R1, R2;
    Q1 = q1*0.5;                // q1 is motor1 angle, gear ratio (driven/driving) of 2:1, Q1 is the angle of joint1 so q1/2 (in total 3 gears are used, so direction of driver is equal to the direction of driving gear)
    Q2 = -q2*0.5;               // q2 is motor2 angle, gear ratio of 1:2, Q2 is the angle of joint2 so -q2/2 (2 gears are used, so direction of driver is opposite from the direction of driving gear)
    Q2 = Q2 + 90*(M_PI/180);    // In our inverse kinematics, the angle of joint 2 is not the same as the angle of joint 2 in zero position. 

    invj00 = sin(Q1 + Q2)/(0.35*cos(Q1 + Q2)*sin(Q1) - 0.35*sin(Q1 + Q2)*cos(Q1));  // Inverse Jacobian used for transforming desired velocity of endeffector to joint angles
    invj01 = -cos(Q1 + Q2)/(0.35*cos(Q1 + Q2)*sin(Q1) - 0.35*sin(Q1 + Q2)*cos(Q1));
    invj10 = -(0.3*sin(Q1 + Q2) + 0.35*sin(Q1))/(0.35*0.3*cos(Q1 + Q2)*sin(Q1) - 0.35*0.3*sin(Q1 + Q2)*cos(Q1));
    invj11 = (0.3*cos(Q1 + Q2) + 0.35*cos(Q1))/(0.35*0.3*cos(Q1 + Q2)*sin(Q1) - 0.35*0.3*sin(Q1 + Q2)*cos(Q1));
 
    V_Q1 = invj00*vx + invj01*vy;   // Angular velocity joint 1
    V_Q2 = invj10*vx + invj11*vy;   // Angular velocity joint 2
    
    // Numerical Integral to make it position controlled
    R1 = Q1 + V_Q1*1e-3;            // Reference position joint 1 (setpoint)
    R2 = Q2 + V_Q2*1e-3;            // Reference position joint 1 (setpoint)
    
    r1= 2.0f*R1;                    // Reference position motor angle 1 (setpoint)
    r2 = -2.0f*(R2 - 90*(M_PI/180));// Reference position motor angle 1 (setpoint)
}


// -------------- STATES ---------------------------------
// State machine implemented in switch-case conditional statement
void do_state_wait(){                               
//  pc.printf("Wait. \n");                          // Was uncommented when testing if state wait was entered
    led1=0; led2=1; led3 = 0 ;                      // yellow
    if(btn1==0)  current_state=calibratinghoming;   // EXIT: go to calibratinghoming
}

void do_state_calibratinghoming(){                  
  if(state_changed == true)                         // DO ONCE 
    {   
        TimesCalled=0;                              // Set timer to zero
        pc.printf("Calibrate homing...\n");
        state_changed = false;
    }      
           
    TimesCalled++;                                  // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    led1=0; led2=0; led3=0;                         // white   
    homing_q1=q1;                                   
    homing_q2=q2;                            
    
    if(btn1==0 && TimesCalled>2e3){                 // EXIT: go to calibrating1
        pc.printf(" Homing angles1 and 2: %f and %f \n",homing_q1, homing_q2);
        current_state = calibrating1;               
        state_changed=true; 
    }                            
}

void do_state_calibrating1()                        
{
    if(state_changed == true){                      // DO ONCE 
        pc.printf("Calibrate EMG1...\n");
        MaxValue=0.0;
        TimesCalled=0;                              // Set timer to zero
        state_changed = false;
    }             
    
    TimesCalled++;                                  // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    led1=0; led2=1; led3=1;                         // red
    GetMaxEMG(emg1_RMS, MaxValue);
    
    if(emg1_RMS< 0.1* MaxValue && TimesCalled>5e3)  // EXIT: go to calibrating2
    {
        emg1_threshold = 0.2*MaxValue;
        current_state = calibrating2; 
        state_changed=true;        
    }
} 
    
void do_state_calibrating2(){
    if(state_changed == true)                       // DO ONCE
    {   
        pc.printf("Calibrate EMG2...\n"); 
        MaxValue=0.0;
        TimesCalled=0;                              // Set timer to zero
        state_changed = false;
    }      
    
    TimesCalled++;                                  // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    led1=1; led2=0; led3=1;                         // blue
    GetMaxEMG(emg2_RMS,MaxValue);
    
    if(emg2_RMS< 0.1* MaxValue && TimesCalled>5e3)  // EXIT: go to homing
    {   
        emg2_threshold = 0.2*MaxValue;
        current_state = homing; 
        state_changed=true;
    }
}  
   
   // homing: when a button is pressed, it should move to the plate
void do_state_homing()
{
    if(state_changed == true)          // DO ONCE 
    {
        pc.printf("homing\n");
        TimesCalled=0;                 // Set timer to zero
        state_changed = false;
    }  
               
    
    TimesCalled++;                      // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    led1=1; led2=1; led3=0;             // green
    u1 = PID_controller1(homing_q1);
    u2 = PID_controller2(homing_q2);                               
    IK(q1,q2);
    error1=r1-q1;
    error2=r2-q2;
    
    if(TimesCalled>2e3&& error1<0.05 && error2<0.05 ) //EXIT: go to movingX
    { 
        u1=0; u2=0;
        current_state = movingX; 
        state_changed=true;        
    } 
}
    
void do_state_movingX()
{
    // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    // EMG1=true && EMG2=false      move in -x direction
    // EMG2=true && EMG1=false      move in +x direction
    // Both EMG false               desired x velocity = zero
    
    led1=0; led2=0; led3=1;                 // Purple
    
    if(emg1_state==1 && emg2_state==0)      // EMG1 is on -> move in the -X direction.
    {
        // Get the position of motor 1 and get the wanted position, where the
        // difference is the error for our PID controllers.
        led3=0;                             // white
        vy = 0;  vx = -0.05;                // move in neg. x direction
        IK(q1,q2);                          // determine setpoint motor angles r1 and r2
        error1=r1-q1; error2=r2-q2;         // determine error for input into PID-controller
        u1 = 10*PID_controller1(error1); 
        u2 = 10*PID_controller2(error2);
    }
    
    else if(emg1_state==0 && emg2_state==1)
    {
        // Get the position of motor 1 and get the wanted position, where the
        // difference is the error for our PID controllers.
        led3=0;                             // white
        vy = 0; vx = 0.05;                  // move in pos. x direction
        IK(q1,q2);                          // determine setpoint motor angles r1 and r2
        error1=r1-q1; error2=r2-q2;         // determine error for input into PID-controller
        u1 = 10*PID_controller1(error1);
        u2 = 10*PID_controller2(error2);
    }  
    
    else if(btn1==0)                        // EXIT: go to homing
    {
        u1=0; u2=0;                         // set motor input to zero
        current_state = homing; 
        state_changed=true;  
    } 
    
    else if(btn2==0)                        // EXIT: go to movingY
    {
        u1=0; u2=0;                         // set motor input to zero
        state_changed=true;  
        current_state = movingY;  
    }
    
    else                                    // when in this state, and emg1 and emg2 are both false
    {
        // Get the position of motor 1 and get the wanted position, where the
        // difference is the error for our PID controllers.
        vy = 0; vx = 0;                     // do not move
        IK(q1,q2);      
        error1=r1-q1; error2=r2-q2;        
        u1 = PID_controller1(error1);
        u2 = PID_controller2(error2); 
    }         
}
   
void do_state_movingY()
{
    // EACH TIME FUNCTION IS CALLED (at 1000 Hz)
    // EMG1=true && EMG2=false      move in -y direction
    // EMG2=true && EMG1=false      move in +y direction
    // Both EMG false               desired y velocity = zero
    
    led1=1; led2=0; led3=0;                 // lightblue
    
    if(emg1_state==1 && emg2_state==0)     // EMG1 is on -> move in the -Y direction.
    {
        // Get the position of motor 1 and get the wanted position, where the
        // difference is the error for our PID controllers.
        led1=0;                             
        vy = -0.05;  vx = 0;
        IK(q1,q2);
        error1=r1-q1; error2=r2-q2;
        u1 = 10*PID_controller1(error1);
        u2 = 10*PID_controller2(error2);
    }
    else if(emg1_state==0 && emg2_state==1) // EMG2 is on -> move in the +Y direction.
    {
        // Get the position of motor 1 and get the wanted position, where the
        // difference is the error for our PID controllers.
        led1=0;                             // white
        vy = 0.05;  vx = 0;
        IK(q1,q2);
        error1=r1-q1;  error2=r2-q2;
        u1 = 10*PID_controller1(error1);
        u2 = 10*PID_controller2(error2);
    }  
    
    else if(btn1==0)                        // EXIT: go to homing
    {
        u1=0; u2=0;                         // set motor input to zero
        current_state = homing; 
        state_changed=true;
    }
    
    else if(btn2==0)                        // EXIT: go to movingX
    {
        u1=0; u2=0;                         // set motor input to zero
        state_changed=true;  
        current_state = movingX;  
    }
    
    else 
    {
        vy = 0; vx = 0;                     // do not move
        IK(q1,q2);
        u1 = PID_controller1(error1);
        u2 = PID_controller2(error2);  
    }            
}

// ------- Measure all and loop function -------------------
void measure_all()
{
    // Measure motor angles
     q1 = Encoder1.getPulses()/4200*2*3.14159265358979;     // 4200 counts for 1 revolution
     q2 = Encoder2.getPulses()/4200*2*3.14159265358979;
    
    // Measure EMG
    emg_GetArray(); 
    emg1_RMS=emg_getRMS(emg1_array);
    emg2_RMS=emg_getRMS(emg2_array);
    emg1_state=emg_GetState(emg1_RMS,emg1_threshold);
    emg2_state=emg_GetState(emg2_RMS, emg2_threshold);  
}

void output_all()   
{
    SetMotor1(u1);  
    SetMotor2(u2);           
} 
void loop_function(){
    measure_all();                          // Measure EMG signals and actual motor angles from encoders
    switch(current_state) {                 // Finite state machine
        case Wait:          
            do_state_wait();
            break;
        case calibratinghoming:  
           do_state_calibratinghoming();
           break;
        case calibrating1:  
            do_state_calibrating1();
            break;
        case calibrating2:
            do_state_calibrating2();
            break;
        case homing:
            do_state_homing();
            break;
        case movingX:
            do_state_movingX();
            break;
        case movingY:
            do_state_movingY();
            break;
            }
     output_all();                          // Output to motor directionpin and PWMpin
    }
    
// ------- MAIN  -------------------
int main(){
    led1=1; led2=1; led3=1;                 // first all leds are out
    current_state = Wait;                   // Begin in state Wait (do nothing, yellow led)
    bqc1.add( &bqlp_1 ).add( &bqhp_1).add(&bqn50_1).add(&bqn100_1).add(&bqn150_1).add(&bqn200_1);
    bqc2.add( &bqlp_2 ).add( &bqhp_2).add(&bqn50_2).add(&bqn100_2).add(&bqn150_2).add(&bqn200_2);
    LoopTimer.attach_us(&loop_function, 1e3);   // Call loop_function every millisecond 
    while (true) { }
}