#include "mbed.h"
#include "TCS3472_I2C.h"
#include "Colour.h"
/*  TCS3472_I2C
 *  A MBED library specifically made for the TCS3472 RGB Colour Sensor.
 */
TCS3472_I2C rgb_sensor(p28,p27); //p28 =sda, p27=scl

int rgb_readings[100][4];   //A 2D array to store 100 values of each colour.
int rgb_average[4] = {0,0,0,0}; //An array to store the average colour values calculated.
double rMax = 9244; //The maximum possible value of red, from the colour sensor.
double gMax = 3194; //The maximum possible value of red, from the colour sensor.
double bMax = 3590; //The maximum possible value of red, from the colour sensor.

int tubeSize = 20;  //The upper software limit for the number of chips that can be stored in each tube.

/*  Threshold values for each colour are defined below.
 *  Threshold values for none (nothing above the sensor) are also stored,
 *  so that we know when there are no chips in the hopper.
 *  This allows us to stop sorting when there are no chips in the hopper,
 *  and differenciate between nothing there and an invalid token colour above the sensor.
 */
int redLT [3] =     {308,84,162};   //The lower threshold value for red.
int redUT [3]=      {400,144,204};   //The upper threshold value for red.

int greenLT [3] =   {91,180,142};   //The lower threshold value for green.
int greenUT [3]=    {132,220,184};   //The upper threshold value for green.

int blueLT [3]=     {79,95,117};   //The lower threshold value for blue.
int blueUT [3]=     {117,139,157};   //The upper threshold value for blue.

int noneLT [3] = {0,0,0};   //The lower threshold value for none.
int noneUT [3] = {80,80,80};   //The upper threshold value for none.

void initColourSensor() //Intialise the colour sensor.
{
    rgb_sensor.enablePowerAndRGBC();    //Enable the internal oscillator and 2-channel ADC
    rgb_sensor.setIntegrationTime(100); //Set integration time to 100. Longer integration time increases sensitivity for low-light.
}
/*  readColourSensor()
 *  Takes 100 readings of all 4 colour values and takes an average of the 100 values.
 *  The average is used to allow the chip to settle on the colour sensor.
 *  
 */
Colour readColourSensor()
{   //Take 100 Colour readings
    for(int i = 0; i < 100; i++) {
        rgb_sensor.getAllColors(rgb_readings[i]);
        Thread::wait(1);
    }
    
    //Add the 100 values together for each colour
    for(int i = 0; i < 100; i++) {
        for(int j = 0; j < 4; j++) {
            rgb_average[j] += rgb_readings[i][j];
        }
    }
    
    //Divide the four summated values by 100 to get the four averages (one for each colour).
    for(int i = 0; i < 4; i++) {
        rgb_average[i] = rgb_average[i] / 100;
    }
    
    //Scale the red, green and blue values bwetween 0 and 255.
    double redd = (rgb_average[1] /gMax) * 255;
    double greend = (rgb_average[2] /bMax) * 255;
    double blued = (rgb_average[0] /rMax) * 255;
    
    //Convert the values from doubles to integers.
    int red = redd;
    int green = greend;
    int blue = blued;

    //Create boolean arrays to check threshold values.
    bool redWithinThreshold[4] = {0,0,0,0};
    bool greenWithinThreshold[4]= {0,0,0,0};
    bool blueWithinThreshold[4]= {0,0,0,0};
    
    //Set red Thresholds
    redWithinThreshold[0] = (red >= redLT[0]) && (red <= redUT[0]);
    greenWithinThreshold[0] = (green >= redLT[1]) && (green <= redUT[1]);
    blueWithinThreshold[0] = (blue >= redLT[2]) && (blue <= redUT[2]);
    
    //Set green Thresholds
    redWithinThreshold[1] = (red >= greenLT[0]) && (red <= greenUT[0]);
    greenWithinThreshold[1] = (green >= greenLT[1]) && (green <= greenUT[1]);
    blueWithinThreshold[1] = (blue >= greenLT[2]) && (blue <= greenUT[2]);
    
    //Set blue Thresholds
    redWithinThreshold[2] = (red >= blueLT[0]) && (red <= blueUT[0]);
    greenWithinThreshold[2] = (green >= blueLT[1]) && (green <= blueUT[1]);
    blueWithinThreshold[2] = (blue >= blueLT[2]) && (blue <= blueUT[2]);
    
    //Set none Thresholds
    redWithinThreshold[3] = (red >= noneLT[0]) && (red <= noneUT[0]);
    greenWithinThreshold[3] = (green >= noneLT[1]) && (green <= noneUT[1]);
    blueWithinThreshold[3] = (blue >= noneLT[2]) && (blue <= noneUT[2]);
    
    //Check the threshold values to find out what colour is on the sensor.
    if(redWithinThreshold[0] && greenWithinThreshold[0] && blueWithinThreshold[0]) {
        return RED;
    } else if(redWithinThreshold[1] && greenWithinThreshold[1] && blueWithinThreshold[1]) {
        return GREEN;
    } else if(redWithinThreshold[2] && greenWithinThreshold[2] && blueWithinThreshold[2]) {
        return BLUE;
    } else if(redWithinThreshold[3] && greenWithinThreshold[3] && blueWithinThreshold[3]) {
        return NONE;
    } else {
        return BIN;
    }

}



