/*******************************************************************
 *
 *  File: RemoteDataStoreAPI.h
 *
 *  Description: Remote DataStore API definition
 *
 *  Copyright 2015 Innomatix, LLC., All Rights Reserved
 *
 *  THIS DOCUMENT AND ITS CONTENTS ARE INTELLECTUAL PROPERTY
 *  OF INNOMATIX, LLC.  ANY DUPLICATION IN PART OR WHOLE
 *  WITHOUT PRIOR WRITTEN CONSENT IS STRICTLY PROHIBITED.
 *
 *******************************************************************/
#ifndef _REMOTEDATASTOREAPI_H_
#define _REMOTEDATASTOREAPI_H_

typedef enum 
{
    rdsSuccess,
    rdsNoConnection,
    rdsNotAllowed,
    rdsServerFull,
    rdsInvalidBin,
    rdsNoData,
    rdsTooBig,
    rdsBadParameter,
    rdsIncompatibleType,
    rdsInUse,
    rdsReadOnly,
    rdsGeneralError
}RdsResults_e;


// Information about a data bin
typedef struct
{
    unsigned int BinId;
    unsigned char BinDataType;      // Store as uchar instead of DataType so 
                                    // we dont have to include rdsdefinds.h  
    char BinName[ 64 ];  /* NULL terminated */

}BinInfoStruct_t;


// Since storage for the data bin is a short, 
#define BinId_None  ((unsigned int)-1)


/**
* Initialize the Remote DataStore API and establish a connection to the server 
*
* @retval RdsResults_e - 
*   •   rdsSuccess – a connection was established
*   •   rdsNoConnection – the timeout expired without a connection
***************************************************************************************************************************/
RdsResults_e DataStoreInit( const char *addr, unsigned short port );

/**
* Close the host connection and shut down the Remote DataStore API 
**************************************************************************************************************************/
void DataStoreClose();

/**
* Retrieve information from the server about the named data bin  
*
* @param binName - name of bin of interest
* @param BinInfoStruct - structure to be populated with the data bin info
* @retval RdsResults_e - 
*   • rdsSuccess – the BinInfoStruct has been filled with information about the requested data bin
*   • resNoConnection – the connection has been lost, or the timeout expired with no response 
*                   from the server
*   • rdsInvalidDataBin – a data bin with the given name does not exist
*   • rdsGeneralError – an unknown or unspecified error has occurred
*
**************************************************************************************************************************/
RdsResults_e GetDataBinInfo(const char *Name, BinInfoStruct_t *Info);

/**
* Retrieve the current value from a data bin
*
* @param BinId - Bin ID as returned by GetBinInfo
* @param Value - value retrieved from the data bin 
* @param Timestamp - system-relative timestamp that the data bin was last udpated
*                       NOTE: the value may not coorelate to clock-time.  Checking
*                       the value against a previously-saved timestamp indicates
*                       the data bin has been updated since the previous timestamp 
*                       was saved.
* @retval RdsResults_e - 
*   •   rdsSuccess – the Value and Timestamp parameters have been filled
*   •   rdsNoConnection – the connection has been lost, or the timeout expired with no response from the server
*   •   rdsInvalidBin – a data bin with the given ID does not exist
*   •   rdsIncompatibleType – the specified data bin is not of the requested type and the data bin value cannot 
*                           be convereted to the requested type.
*   •   rdsNoData – the specified data bin has not been written to yet
*   •   rdsBadParameter – the value buffer pointer is NULL
*   •   rdsTooBig – the value buffer is not big enough to hold the retrieved data, the value buffer is left unchanged
*   •   rdsInUse – the requested data bin is in use by another application or client and cannot be read 
*   •   rdsGeneralError – an unknown or unspecified error has occurred
***************************************************************************************************************************/
RdsResults_e GetSignedInteger(unsigned int BinId, int *Value, unsigned int *Timestamp);
RdsResults_e GetUnsignedInteger(unsigned int BinId, unsigned int *Value, unsigned int *Timestamp);
RdsResults_e GetDouble(unsigned int BinId, double *Value, unsigned int *Timestamp);
RdsResults_e GetString(unsigned int BinId, char *Str, unsigned short MaxStrLen,unsigned int *Timestamp);
RdsResults_e GetBlob( unsigned int BinId, unsigned char *Blob, unsigned short MaxBlobLen, unsigned short *ActualBlobLen, unsigned int *Timestamp);

/**
* Put a new value into the data bin
*
* @param BinId - Bin ID to update, as retrieved from GetBinInfo()
* @param Value - new value for the data bin 
* @retval RdsResults_e - 
*   •   rdsSuccess – the Value and Timestamp parameters have been filled
*   •   rdsNoConnection – the connection has been lost, or the timeout expired with no response from the server
*   •   rdsInvalidBin – a data bin with the given ID does not exist
*   •   rdsIncompatibleType – the specified data bin is not of the requested type and the data bin value cannot 
*                           be convereted to the requested type.
*   •   rdsNoData – the specified data bin has not been written to yet
*   •   rdsBadParameter – the value buffer pointer is NULL
*   •   rdsTooBig – the value buffer is not big enough to hold the retrieved data, the value buffer is left unchanged
*   •   rdsInUse – the requested data bin is in use by another application or client and cannot be read 
*   •   rdsGeneralError – an unknown or unspecified error has occurred
*
*   example function call:  PutSignedInteger( BinId, SignedValue )
******************************************************************************************************************************/
RdsResults_e PutSignedInteger(unsigned int BinId, int Value);
RdsResults_e PutUnsignedInteger(unsigned int BinId, unsigned int Value);
RdsResults_e PutDouble(unsigned int BinId, double Value);
RdsResults_e PutString(unsigned int BinId, char *Str);
RdsResults_e PutBlob( unsigned int BinId, unsigned char *Blob, unsigned short BlobLen );


/**
* Helper function to send a string to a pre-defined "debug log" data bin
*
* @param String - the message to send
* @retval RdsResults_e - same as PutString()
******************************************************************************************************************************/
RdsResults_e PutDebug( char *Str );


#endif // _REMOTEDATASTOREAPI_H_