/*
 * DMX512 send/recv library
 * Copyright (c) 2013 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief DMX512 send/recv
 */
 
#ifndef DMX_H
#define DMX_H

#include "mbed.h"
#include "RawSerial.h"

//#define DMX_UART_DIRECT

#define DMX_SIZE        512
#define DMX_START_CODE  0

#define DMX_TIME_BREAK  100 // 100us (88us-1s)
#define DMX_TIME_MAB    12 // 12us (8us-1s)
#define DMX_TIME_MBB    200 // 10us (0-1s)

enum DMX_MODE {
    DMX_MODE_BEGIN,
    DMX_MODE_START,
    DMX_MODE_BREAK,
    DMX_MODE_MAB,
    DMX_MODE_DATA,
    DMX_MODE_ERROR,
    DMX_MODE_STOP,
};

/** DMX512 class (sender/client)
 */
class DMX {
public:
    /** init DMX class
     * @param p_tx TX serial port (p9, p13, p28)
     * @param p_rx RX serial port (p10, p14, p27)
     */
    DMX (PinName p_tx, PinName p_rx); 

    /** Send the data
     * @param addr DMX data address (0-511)
     * @param data DMX data (0-255)
     */
    void put (int addr, int data);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void put (unsigned char *buf, int addr = 0, int len = DMX_SIZE);

    /** Send the data
     * @param addr DMX data address (0-511)
     * @return DMX data (0-255)
     */
    int get (int addr);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void get (unsigned char *buf, int addr = 0, int len = DMX_SIZE);

    /** Start DMX send operation
     */
    void start ();
    /** Stop DMX send operation
     */
    void stop ();
    /** Clear DMX data
     */
    void clear ();

    int isReceived ();
    int isSent ();
    unsigned char *getRxBuffer ();
    unsigned char *getTxBuffer ();
    int setTimingParameters (int breaktime, int mab, int mbb);
    
    void attach(void (*function)(void));

protected:

    void int_timer ();
    void int_tx ();
    void int_rx ();
    void on_received();

//    Serial _dmx;
    RawSerial _dmx;
    Timeout timeout01;
    volatile DMX_MODE mode_tx, mode_rx;
    volatile int addr_tx, addr_rx;
    unsigned char data_tx[DMX_SIZE];
    unsigned char data_rx_working[DMX_SIZE];
    unsigned char data_rx[DMX_SIZE];
    int is_received, is_sent;
    int time_break, time_mab, time_mbb;
    Callback<void()> on_rx;

private:
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368) || defined(TARGET_LPC4088)
    LPC_UART_TypeDef *_uart;
#elif defined(TARGET_LPC11UXX)
    LPC_USART_Type *_uart;
#elif defined(TARGET_LPC11XX)
    LPC_UART_TypeDef *_uart;
#elif defined(TARGET_STM)
    USART_TypeDef *_uart;
#else
#error "this CPU not supported."
#endif

};

#endif