/* MCU Gear Library, only for testing MCUGear without any circuit you connected.
 * Copyright (c) 2013, NestEgg Inc., http://www.mcugear.com/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"
#include "TextLCD.h"

#include "MCUGearBase.h"
#include "MCUGear.h"
#include "common.h"

Serial pc(USBTX, USBRX); // tx, rx Set BaudRate = 115200 in main()

 //Make instance. Set I2C pins, Module address and Connector numnber of Modules.

#ifdef LPC1768_mbed
//caution: Do not use I2C(p28 & p27). It is reserved in this system with high speed 1.5MHz I2C.
//If you want to use I2C, you should change I2C clock.
// or use multi-function module.

//Address setting:you can see the detail on common.h file.
//address format : (Type of device)_AD2pin_AD1pin_AD0pin
//N(PCA9674) A(PCA9674A) (VDD = +3V3  VSS = GND)

//Initialize modules-----------------------------------------------------------------------
MCUGear AD12M(p28, p27, N_SCL_SCL_VSS,4);       //AD(MCP3202) 12bit Module : Set number of module pin 
MCUGear AD8ch12M(p28, p27, N_SDA_SDA_SDA,4);    //AD(MCP3208) 12bit 8ch Module
MCUGear DA12M(p28, p27, N_SCL_SCL_SCL,4);       //DA(MCP4901) 12bit Module
MCUGear LCDM(p28, p27, N_VDD_VSS_VSS,6);        //LCD Module
MCUGear SHM(p28, p27, N_VDD_VSS_VDD,8);         //Signal Hold Module
MCUGear Uni2M(p28, p27, N_VDD_VDD_VDD,8);       //2 Connector Universal Module
MCUGear MFM(p28, p27, N_VSS_VSS_VSS,4);         //Multifunction Module
//-----------------------------------------------------------------------------------------

//Sample setting-----------------------------------------------------------------------
//TextLCD lcd(p21, p22, p23, p24, p25, p26, TextLCD::LCD16x2); // rs, e, d4-d7
SPI spi(p5, p6, p7); // mosi, miso, sclk
DigitalOut cs(p8); //for SPI communications
DigitalOut Dout(p15);//for 12bit DA module
BusOut SignalHold (p16,p21,p22,p23,p24,p25,p26);

DigitalOut Dout1(p16);
DigitalOut Dout2(p21);
DigitalOut Dout3(p22);
DigitalOut Dout4(p23);
DigitalOut Dout5(p24);
DigitalIn Din1(p9);
DigitalIn Din2(p10);

//-------------------------------------------------------------------------------------
#endif


#ifdef FS_KL25Z
//Initialize modules-----------------------------------------------------------------------
MCUGear AD12M(PTE0, PTE1, N_SCL_SCL_VSS,4);       //AD(MCP3202) 12bit Module : Set number of module pin 
MCUGear AD8ch12M(PTE0, PTE1, N_SDA_SDA_SDA,4);    //AD(MCP3208) 12bit 8ch Module
MCUGear DA12M(PTE0, PTE1, N_SCL_SCL_SCL,4);       //DA(MCP4901) 12bit Module
MCUGear LCDM(PTE0, PTE1, N_VDD_VSS_VSS,6);        //LCD Module
MCUGear SHM(PTE0, PTE1, N_VDD_VSS_VDD,8);         //Signal Hold Module
MCUGear Uni2M(PTE0, PTE1, N_VDD_VDD_VDD,8);       //2 Connector Universal Module
MCUGear MFM(PTE0, PTE1, N_VSS_VSS_VSS,4);         //Multifunction Module
//-----------------------------------------------------------------------------------------

//Sample setting-----------------------------------------------------------------------
//TextLCD lcd(PTA1, PTA2, PTD4, PTA12, PTA4, PTA5, TextLCD::LCD16x2); // rs, e, d4-d7
SPI spi(PTD2,PTD3,PTD1); // mosi, miso, sclk
DigitalOut cs(PTD0); //for SPI communications
DigitalOut Dout(PTA13);//for 12bit DA module
BusOut SignalHold (PTB0, PTB1, PTD4, PTA12, PTA4, PTA5, PTC8);

DigitalOut Dout1(PTB0);
DigitalOut Dout2(PTB1);
DigitalOut Dout3(PTD4);
DigitalOut Dout4(PTA12);
DigitalOut Dout5(PTA4);
DigitalIn Din1(PTB0);
DigitalIn Din2(PTB1);
#endif

//sample functions------------------------------------------------------------
void IOSimpleSPI(MCUGear *mcugear);             //IO setting for Simple SPI module
void IO12bitDA(MCUGear *mcugear);               //IO setting for DA 12bit module
void IOLCD(MCUGear *mcugear);                   //IO setting for LCD module
void IOSHM(MCUGear *mcugear);                   //IO setting for Signal Hold module
void IOUni2M(MCUGear *mcugear);                 //IO setting for 2 Connector Universal module 

int read12bitAD(MCUGear *mcugear, char ch);     //simple ADC 
void write12bitDA(MCUGear *mcugear, int data);  //simple DA
int read12bit8chAD(MCUGear *mcugear, char ch);  //simple ADC 8ch

#ifdef AD_MODE //for Multifunction Module
int read10bitAD(MCUGear *mcugear, int ch);
#endif

#ifdef PWM_MODE //for Multifunction Module
void initPWM(MCUGear *mcugear, unsigned int Divider, unsigned int friquency, unsigned int duty0, unsigned int duty1, unsigned int duty2);
void PWMfriq(MCUGear *mcugear, unsigned int friquency);
void PWMDuty(MCUGear *mcugear, int ch, unsigned int Duty);
void StopPWM(MCUGear *mcugear);
void StartPWM(MCUGear *mcugear);
#endif
//----------------------------------------------------------------------------


//***************************************************************************************
int main() {

    int SW1,SW2;

//    pc.printf("hello world!!!");
    
    initBase(); //initialize Baseboard
    

   //Set IO --------------------------------------------------------------

    IOSimpleSPI(&AD12M);
    IOSimpleSPI(&AD8ch12M);
    IO12bitDA(&DA12M);
    IOLCD(&LCDM);
    IOSHM(&SHM);
    IOUni2M(&Uni2M);
    IOSimpleSPI(&MFM);    //IO setting for Multifunction module

   //---------------------------------------------------------------------
   wait(1);
   
    LCDM.connectModule();
 #ifdef LPC1768_mbed
    TextLCD lcd(p21, p22, p23, p24, p25, p26, TextLCD::LCD16x2); // rs, e, d4-d7
 #endif
 
 #ifdef FS_KL25Z
    TextLCD lcd(PTB0, PTB1, PTD4, PTA12, PTA4, PTA5, TextLCD::LCD16x2); // rs, e, d4-d7
 #endif
 
    lcd.cls();
    lcd.printf("Hello World!\n");
    wait(1);
    LCDM.disconnectModule();



    int data1[2];
    int data2[8];
    int data3[3];
    uint8_t busData = 0;

#ifdef PWM_MODE

    uint32_t PWM_data = 0x7FF;
    uint32_t PWM_data_old = 0;
    
    MFM.connectModule();   //connect---
    initPWM(&MFM, 24, 40950, 10, 10, 10);// 0-2msec 12bit 20msec for RC servosetting 
    //initPWM(&MFM, 120, 4095, 20, 50, 80);//Simple 10msec PWM
    //initPWM(&MFM, 48, 26, 70, 70, 70);//38kHz 26usec for IR LED
    MFM.disconnectModule();   //disconnect---

#endif    
    
    while(1) {
       //12bit AD Mobdule-------------------------------------------------

       cs = 1; //reset CS pin
       AD12M.connectModule();   //connect---
       data1[0] = read12bitAD(&AD12M, 0);
       data1[1] = read12bitAD(&AD12M, 1);
       AD12M.disconnectModule();   //disconnect---
       
       //12bit 8ch AD Module
       
       cs = 1; //reset CS pin
       AD8ch12M.connectModule();   //connect---
       data2[0] = read12bit8chAD(&AD8ch12M, 0);    //0ch
       data2[1] = read12bit8chAD(&AD8ch12M, 1);    //1ch
       data2[2] = read12bit8chAD(&AD8ch12M, 2);    //2ch
       data2[3] = read12bit8chAD(&AD8ch12M, 3);    //3ch
       data2[4] = read12bit8chAD(&AD8ch12M, 4);    //4ch
       data2[5] = read12bit8chAD(&AD8ch12M, 5);    //5ch
       data2[6] = read12bit8chAD(&AD8ch12M, 6);    //6ch
       data2[7] = read12bit8chAD(&AD8ch12M, 7);    //7ch
       AD8ch12M.disconnectModule();   //disconnect---
       
#ifdef AD_MODE //for Multifunction Module
       cs = 1; //reset CS pin
       MFM.connectModule();   //connect---
       data3[0] = read10bitAD(&MFM, 1); //1ch
       data3[1] = read10bitAD(&MFM, 2); //2ch
       data3[2] = read10bitAD(&MFM, 3); //3ch
       MFM.disconnectModule();   //disconnect---
#endif

#ifdef PWM_MODE //for Multifunction Module

        cs = 1; //reset CS pin
        MFM.connectModule();   //connect---
        
        //RC Servo and Simple PWM
/*        PWMDuty(&MFM, 1, 0x7FF);    //duty 50%
        PWMDuty(&MFM, 2, 0x7FF);
        PWMDuty(&MFM, 3, 0x7FF);
*/
 
        if(PWM_data > 0xFFF){
            PWM_data = 0x7FF;
        }
        
        //RC Servo and Simple PWM
        PWMDuty(&MFM, 1, 0x7FF);    //duty 50%
        
        if(PWM_data_old != PWM_data){
            PWMDuty(&MFM, 2, PWM_data);
            PWM_data_old = PWM_data;
            
        }
        
        //PWMDuty(&MFM, 3, PWM_data);
        PWMDuty(&MFM, 3, data1[0]+0x7FF);
        //mwait01(10000);   //If the system is heavy or not, you should adjust the value.

        PWM_data = PWM_data + 0x0FF;

        //Sample for IR LED 
/*
        StopPWM(&MFM);
        wait_us(10);
        StartPWM(&MFM);
        wait_us(10);        
        StopPWM(&MFM);
        wait_us(10);
        StartPWM(&MFM);
        wait_us(10);        
        StopPWM(&MFM);
*/
        MFM.disconnectModule();   //disconnect---

#endif        
       

       //LCD Module------------------------------------------------------
       LCDM.connectModule();   //connect---
       lcd.printf(" ");    //set IO as a neutral position
       lcd.cls();   //clear LCD
       lcd.printf("A%d B%d\nC%d D%d",data1[0],data1[1],data2[0],data2[6]);   //write LCD data
       wait_ms(10);    //delay for view the LCD
       LCDM.disconnectModule();   //disconnect---

       //12bit 2ch AD
       //pc.printf("0ch%d, 1ch%d,  ",data1[0],data1[1]);
       //12bit 8ch AD
       pc.printf("0ch%d, 1ch%d, 2ch%d, 3ch%d, 4ch%d, 5ch%d, 6ch%d, 7ch%d,  ",data2[0],data2[1],data2[2],data2[3],data2[4],data2[5],data2[6],data2[7]);
#ifdef AD_MODE //for Multifunction Module
       pc.printf("0ch%d, 1ch%d, 2ch%d",data3[0],data3[1],data3[2]);
#endif
       pc.printf("\r\n");


       //12bit SPI DA module---------------------------------------------
       cs = 1; //reset CS pin
       Dout=1;    //reset ratch signal
       DA12M.connectModule();   //connect---
       write12bitDA(&DA12M, data1[0]);
       DA12M.disconnectModule();   //disconnect---


       //Signal Hold module (TC74VHCT540AF)------------------------------
       Dout = 0;    //ratch standby
       SHM.connectModule();   //connect---
       
       SignalHold = busData;   //set 7bit bus out
       Dout = 1;    //ratch
       Dout = 0;    //ratch standby
       SHM.disconnectModule();   //disconnect---
       
       //2 connector universal module ------------------------------
       
       Dout = 1;
       Dout1 = 1;
       Dout2 = 1;
       Dout3 = 1;
       Dout4 = 1;
       Dout5 = 1;
       
       Uni2M.connectModule();   //connect---
       
       SW1 = Din1.read();
       SW2 = Din2.read();
       
       if(SW1 == 1){
            Dout = 1;
            Dout2 = 1;
            Dout3 = 1;
       }else{
            Dout = 0;
            Dout2 = 0;
            Dout3 = 0;
       }
       if(SW2 == 1){
            Dout1 = 1;
            Dout4 = 1;
            Dout5 = 1;
       }else{
            Dout1 = 0;
            Dout4 = 0;
            Dout5 = 0;
       }
       
       Uni2M.disconnectModule();   //disconnect---

       /*++busData;
       if(busData >= 0x80)
            busData = 0x00;
       */
       
       if(busData == 0x00)
            busData = 0x7F;
       
       --busData;
    }    
}
//***************************************************************************************


//Smple functions
//----------------------------------------------------------------------------------
//Simple read 12bit 2ch AD module
//----------------------------------------------------------------------------------
int read12bitAD(MCUGear *mcugear, char ch){

    char sendData;
    int whoami[3];
    
#ifdef LPC1768_mbed
    spi.frequency(1000000);//1MHz
    if(ch == 0){
        sendData = 0x06;
    }else if(ch == 1){
        sendData = 0x07;
    }else{
        return -1;
    }
    //12bit AD Mobdule----------------
//    cs = 1; //reset CS pin
//    mcugear->connectModule();   //connect---
    spi.format(14,0);    //Change SPI format mbed only
    cs = 0;
    spi.write(sendData);
    wait_us(1);
    whoami[0] = spi.write(0x2000)-0x2000;   //get data from AD device
    cs = 1;
//    mcugear->disconnectModule();   //disconnect---
    return whoami[0];
#endif

#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    if(ch == 0){
        sendData = 0x80;
    }else if(ch == 1){
        sendData = 0xC0;
    }else{
        return -1;
    }
    
    //12bit AD Mobdule----------------
 //   cs = 1; //reset CS pin
 //   mcugear->connectModule();   //connect---
    
    cs = 0; 
    //FRDM KL25Z is 8-bit format.
    whoami[2] = spi.write(0x01);//0000 0001
    whoami[2] = spi.write(sendData);
    whoami[0] = whoami[2]<<8;
    whoami[0] = whoami[0] + spi.write(0x00) - 0xE000;//0000 0000
//    whoami[2] = 0;//clear buffer
    cs = 1;
 //   mcugear->disconnectModule();   //disconnect---
    return whoami[0];
#endif
}

//----------------------------------------------------------------------------------
//IO setting for LCD module
//----------------------------------------------------------------------------------
void IOLCD(MCUGear *mcugear){
    uint8_t fio[12];
    mcugear->detect_module(fio);    // detect LCDM

#ifdef LPC1768_mbed
    mcugear->savePinSetting(0, IO_MBED_P21, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_P22, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_P23, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_P24, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_P25, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_P26, IO_REG_OUT_DIR, fio[5]);
#endif

#ifdef FS_KL25Z
    mcugear->savePinSetting(0, IO_MBED_PTB0, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_PTB1, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_PTD4, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_PTA12, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_PTA4, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_PTA5, IO_REG_OUT_DIR, fio[5]);
#endif

#ifdef BOOST_MODE
    mcugear->makeCircuit();
#endif
}


//----------------------------------------------------------------------------------
//IO setting for 12bit DA module
//----------------------------------------------------------------------------------
void IO12bitDA(MCUGear *mcugear){
    uint8_t fio[12];
   //---------------------
    mcugear->detect_module(fio);
    //Dout(p15),mosi(p5),sck(p7),cs(p8)
#ifdef LPC1768_mbed
    mcugear->savePinSetting(0, IO_MBED_P15, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_P5, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_P7, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_P8, IO_REG_OUT_DIR, fio[3]);
#endif

#ifdef FS_KL25Z
    //Dout(PTA13),mosi(PTD2),sck(PTD1),cs(PTD0)
    mcugear->savePinSetting(0, IO_MBED_PTA13, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_PTD2, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_PTD1, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_PTD0, IO_REG_OUT_DIR, fio[3]);
#endif

#ifdef BOOST_MODE
    mcugear->makeCircuit();
#endif
}

//----------------------------------------------------------------------------------
//Simple write for 12bit DA module
//----------------------------------------------------------------------------------
void write12bitDA(MCUGear *mcugear, int data){

       //12bit SPI DA module----------------
       
#ifdef LPC1768_mbed
//       spi.frequency(1000000); //1MHz
       spi.frequency(20000000); //20MHz
       spi.format(16,0);        //Change SPI format
//       mcugear->connectModule();   //connect---
       cs = 0;
       spi.write((0x3000)+(data)); //write 0011 1000 0000 0000
#endif

       //12bit SPI DA module----------------
#ifdef FS_KL25Z
       //spi.frequency(2000000); //1MHz for KL25Z
       spi.frequency(20000000); //10MHz
       //connection
 //      mcugear->connectModule();   //connect---
       cs = 0;
       spi.write((0x30)+((data>>8)&0x0F));//write 0011 0000
       spi.write(data&0xFF);//write 0011 1000 0000 0000
#endif

       wait_us(1);  //over 15nsec
       cs = 1;      //CS high
       wait_us(1);  //over 40nsec
       Dout=0;      //ratch for DA out enable
       wait_us(1);  //over 100nsec
       Dout=1;      //reset ratch signal
//       mcugear->disconnectModule();   //disconnect---
//       wait_us(1);  //wait after disconnect
}

//----------------------------------------------------------------------------------
//Simple write for 12bit 8ch AD module
//----------------------------------------------------------------------------------
int read12bit8chAD(MCUGear *mcugear, char ch){

    char sendData;
    int whoami[3];
    
#ifdef LPC1768_mbed
//    if((ch >= 0 )&&(ch < 8)){
    if(ch < 8){
        sendData = 0x18+ch; //0001 1000
    }else{
        return -1;
    }
    
    //12bit AD Mobdule----------------
    spi.frequency(1000000); //1MHz
 //   cs = 1; //reset CS pin
 //   mcugear->connectModule();   //connect---
    spi.format(14,0);    //Change SPI format mbed only
    cs = 0;
    spi.write(sendData);
    whoami[0] = spi.write(0x0000)-0x2000;   //get data from AD device
    cs = 1;
//    mcugear->disconnectModule();   //disconnect---
    return whoami[0];
#endif

#ifdef FS_KL25Z
    char sendData2;
//0000 0000 0110 00 00 [0000 0000 0000]
//    if((ch >= 0 )&&(ch < 8)){
    if(ch < 8){
        sendData = 0x18+ch; //0001 1000
        sendData2 = sendData >> 2;
        sendData = sendData << 6;
        
    }else{
        return -1;
    }
    
    //12bit AD Mobdule----------------
    spi.frequency(2000000); //1MHz for KL25Z
//    cs = 1; //reset CS pin
//    mcugear->connectModule();   //connect---
    cs = 0;
    spi.write(sendData2);
    whoami[2] = spi.write(sendData);
    whoami[0] = whoami[2]<<8;
    whoami[0] = whoami[0] + spi.write(0x00)-0xE000;   //get data from AD device
    cs = 1;
//    mcugear->disconnectModule();   //disconnect---
    return whoami[0];


#endif
}


//----------------------------------------------------------------------------------
//Simple universal module
//----------------------------------------------------------------------------------
void IOSHM(MCUGear *mcugear){

    uint8_t fio[12];
    // detect SHM
    mcugear->detect_module(fio);
    //CLK for ratch(1), DigialOut(2-8)

#ifdef LPC1768_mbed
    mcugear->savePinSetting(0, IO_MBED_P15, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_P16, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_P21, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_P22, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_P23, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_P24, IO_REG_OUT_DIR, fio[5]);
    mcugear->savePinSetting(6, IO_MBED_P25, IO_REG_OUT_DIR, fio[6]);
    mcugear->savePinSetting(7, IO_MBED_P26, IO_REG_OUT_DIR, fio[7]);
#endif

#ifdef FS_KL25Z
    //PTA1, PTA2, PTD4, PTA12, PTA4, PTA5, PTC8
    mcugear->savePinSetting(0, IO_MBED_PTA13, IO_REG_OUT_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_PTA1, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_PTA2, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_PTD4, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_PTA12, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_PTA4, IO_REG_OUT_DIR, fio[5]);
    mcugear->savePinSetting(6, IO_MBED_PTA5, IO_REG_OUT_DIR, fio[6]);
    mcugear->savePinSetting(7, IO_MBED_PTC8, IO_REG_OUT_DIR, fio[7]);
#endif
 
#ifdef BOOST_MODE
    mcugear->makeCircuit();
#endif

}

//----------------------------------------------------------------------------------
//Simple universal module
//----------------------------------------------------------------------------------
void IOUni2M(MCUGear *mcugear){

    uint8_t fio[12];
    // detect SHM
    mcugear->detect_module(fio);
    //CLK for ratch(1), DigialOut(2-8)

#ifdef LPC1768_mbed
    mcugear->savePinSetting(0, IO_MBED_P9, IO_REG_IN_DIR, fio[0]);  //input
    mcugear->savePinSetting(1, IO_MBED_P10, IO_REG_IN_DIR, fio[1]); //input
    mcugear->savePinSetting(2, IO_MBED_P15, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_P16, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_P21, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_P22, IO_REG_OUT_DIR, fio[5]);
    mcugear->savePinSetting(6, IO_MBED_P23, IO_REG_OUT_DIR, fio[6]);
    mcugear->savePinSetting(7, IO_MBED_P24, IO_REG_OUT_DIR, fio[7]);
 
#endif

#ifdef FS_KL25Z
    //PTA1, PTA2, PTD4, PTA12, PTA4, PTA5, PTC8
    mcugear->savePinSetting(0, IO_MBED_PTB0, IO_REG_IN_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_PTB1, IO_REG_IN_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_PTA13, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_PTA1, IO_REG_OUT_DIR, fio[3]);
    mcugear->savePinSetting(4, IO_MBED_PTA2, IO_REG_OUT_DIR, fio[4]);
    mcugear->savePinSetting(5, IO_MBED_PTD4, IO_REG_OUT_DIR, fio[5]);
    mcugear->savePinSetting(6, IO_MBED_PTA12, IO_REG_OUT_DIR, fio[6]);
    mcugear->savePinSetting(7, IO_MBED_PTA4, IO_REG_OUT_DIR, fio[7]);
 
#endif
 
#ifdef BOOST_MODE
    mcugear->makeCircuit();
#endif

}



//----------------------------------------------------------------------------------
//Read from Multifunction module for 10bit AD mode
//----------------------------------------------------------------------------------
int read10bitAD(MCUGear *mcugear, int ch){
    //Multi-function module
    int setCh = 0;
    int getData;
    
 //   mcugear->connectModule();
    
    switch(ch){
        case 1:
            setCh = 3;
            break;
        case 2:
            setCh = 1;
            break;
        case 3:
            setCh = 2;
            break;
        
        default:
            return -1;
    }
    
#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    spi.write(setCh);
    cs = 1;
    wait_us(1);
    cs = 0;
    getData = spi.write(0);
    cs = 1;
    wait_us(1);
#endif
        
#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    spi.write(0x00);
    spi.write(setCh);
    cs = 1;
    wait_us(1);
    cs = 0;
    getData = spi.write(0);
    getData = (getData<<8)+spi.write(0);
    cs = 1;
    wait_us(1);
#endif


 //   mcugear->disconnectModule();
 //   wait_ms(1);
    return getData;
}

#ifdef PWM_MODE
//----------------------------------------------------------------------------------
// Initialize PWM
//----------------------------------------------------------------------------------
void initPWM(MCUGear *mcugear, unsigned int Divider, unsigned int friquency, unsigned int duty0, unsigned int duty1, unsigned int duty2){

 //   mcugear->connectModule();

#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    //spi.write(4800);//Divider
    spi.write(Divider);//Divider
    cs = 1;
    wait_us(1);

    cs = 0;
    //spi.write(100);//friquency
    spi.write(friquency);//friquency
    cs = 1;
    wait_us(1);

    cs = 0;
    spi.write(duty0);//Duty0
    cs = 1;
    wait_us(1);
        
    cs = 0;
    spi.write(duty1);//Duty1
    cs = 1;
    wait_us(1);
        
    cs = 0;
    spi.write(duty2);//Duty2
    cs = 1;
    wait_us(1);        
#endif


#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    spi.write(Divider >> 8);//Divider
    spi.write(0x00FF & Divider);//Divider
    cs = 1;
    wait_us(1);

    cs = 0;
    spi.write(friquency >> 8);//friquency
    spi.write(0x00FF & friquency);//friquency
    cs = 1;
    wait_us(1);

    cs = 0;
    spi.write(duty0 >> 8);//Duty0
    spi.write(0x00FF & duty0);//Duty0
    cs = 1;
    wait_us(1);
        
    cs = 0;
    spi.write(duty1 >> 8);//Duty1
    spi.write(0x00FF & duty1);//Duty1
    cs = 1;
    wait_us(1);
        
    cs = 0;
    spi.write(duty2 >> 8);//Duty2
    spi.write(0x00FF & duty2);//Duty2
    cs = 1;
    wait_us(1);        
#endif

//    mcugear->disconnectModule();
}

//----------------------------------------------------------------------------------
// changing PWM friquency
//----------------------------------------------------------------------------------
void PWMfriq(MCUGear *mcugear, unsigned int friquency){

 //   mcugear->connectModule();
#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    spi.write(friquency);//friquency
    cs = 1;
    wait_us(1);
#endif

#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    spi.write(friquency >> 8);//Divider
    spi.write(0x00FF & friquency);//Divider
    cs = 1;
    wait_us(1);
#endif

 //   mcugear->disconnectModule();
}

void PWMDuty(MCUGear *mcugear, int ch, unsigned int Duty){

    //data format
    //15-12:setting(Header)
    //  0 :stop PWM out
    //  1 :start PWM out
    //  2 :Frequency setting
    //  3 :Duty0 setting
    //  4 :Duty1 setting
    //  5 :Duty2 setting
    //  other :stop PWM out
    //11-0:Data
    
 //   mcugear->connectModule();
    
#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    switch (ch){
        case 1:
            spi.write(0x3000 + Duty);   //Duty0
        case 2:
            spi.write(0x4000 + Duty);   //Duty0
        case 3:
            spi.write(0x5000 + Duty);   //Duty0
        default:
            break;
    }
    
    cs = 1;    
#endif

#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    switch (ch){
        case 1:
            spi.write((0x3000 + Duty) >> 8);//Divider
            spi.write(0x00FF & (0x3000 + Duty));//Divider
        case 2:
            spi.write((0x4000 + Duty) >> 8);//Divider
            spi.write(0x00FF & (0x4000 + Duty));//Divider
        case 3:
            spi.write((0x5000 + Duty) >> 8);//Divider
            spi.write(0x00FF & (0x5000 + Duty));//Divider
        default:
            break;
    }
    
    cs = 1;    
#endif

 //   mcugear->disconnectModule();

}


//----------------------------------------------------------------------------------
// Stop PWM
//----------------------------------------------------------------------------------
void StopPWM(MCUGear *mcugear){

//    mcugear->connectModule();
    
#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    spi.write(0x1000);//stop
    cs = 1;
#endif

#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    spi.write(0x10);//Duty1
    spi.write(0x00);//Duty1
    cs = 1;
#endif

//    mcugear->disconnectModule();
}

//----------------------------------------------------------------------------------
// Start PWM
//----------------------------------------------------------------------------------
void StartPWM(MCUGear *mcugear){

 //   mcugear->connectModule();
    
#ifdef LPC1768_mbed
    spi.frequency(1000000); //1MHz
    spi.format(16,0);
    cs = 0;
    spi.write(0x6000);//start
    cs = 1;
#endif

#ifdef FS_KL25Z
    spi.frequency(2000000); //1MHz for KL25Z
    cs = 0;
    spi.write(0x60);//start
    spi.write(0x00);//start
    cs = 1;
#endif

//    mcugear->disconnectModule();
}


#endif

//----------------------------------------------------------------------------------
//IO setting for Multifunction module
//----------------------------------------------------------------------------------
void IOSimpleSPI(MCUGear *mcugear){
    uint8_t fio[12];
    mcugear->detect_module(fio);
    
#ifdef LPC1768_mbed
    //miso(p6),mosi(p5),sck(p7),cs(p8)
    mcugear->savePinSetting(0, IO_MBED_P6, IO_REG_IN_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_P5, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_P7, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_P8, IO_REG_OUT_DIR, fio[3]);
#endif

#ifdef FS_KL25Z
    //miso(PTD3),mosi(PTD2),sck(PTD1),cs(PTD0)
    mcugear->savePinSetting(0, IO_MBED_PTD3, IO_REG_IN_DIR, fio[0]);
    mcugear->savePinSetting(1, IO_MBED_PTD2, IO_REG_OUT_DIR, fio[1]);
    mcugear->savePinSetting(2, IO_MBED_PTD1, IO_REG_OUT_DIR, fio[2]);
    mcugear->savePinSetting(3, IO_MBED_PTD0, IO_REG_OUT_DIR, fio[3]);
#endif

#ifdef BOOST_MODE
    mcugear->makeCircuit();
#endif
}

//end of sample functions----------------------------------------------------


