#include "mbed.h"
#include "WiflyInterface.h"
#include "NokiaLCD.h"
#include "picojpeg.h"
#include "DebounceIn.h"
#include "GPS.h"
#include <vector>

#define DEFAULTZOOM 15

/* wifly interface:
*     - p13 and p14 are for the serial communication
*     - p28 is for the reset pin
*     - p27 is for the connection status
*     - 5th field is the ssid of the network
*     - 6th field is the password
*     - WPA is the security
*/

WiflyInterface wifly(p13, p14, p28, p27, "username", "password", WPA);
Serial pc(USBTX,USBRX);
LocalFileSystem local("local");// Create the local filesystem under the name "local"
GPS gps(p9, p10);
DebounceIn ZoomIn(p18);
DebounceIn Refresh(p19);
DebounceIn ZoomOut(p20);
FILE *jpegfile;
int jpeg_filesize = 0;
int jpeg_filepos = 0;
NokiaLCD lcd(p5, p7, p8, p11, NokiaLCD::LCD6610);
int zoom = DEFAULTZOOM;


unsigned char pjpeg_need_bytes_callback(unsigned char* pBuf, unsigned char buf_size, unsigned char *pBytes_actually_read, void *pCallback_data)
{
    unsigned int n = min((unsigned int)(jpeg_filesize - jpeg_filepos), (unsigned int)buf_size);
    if (n && (fread(pBuf, 1, n, jpegfile) != n))
        return PJPG_STREAM_READ_ERROR;
    *pBytes_actually_read = (unsigned char)(n);
    jpeg_filepos += n;
    return 0;
}

void ReadJPEGFromFile(const char *filename, NokiaLCD *lcd)
{
    pjpeg_image_info_t imageInfo;
    jpegfile = fopen(filename,"rb");
    if(!jpegfile) {
        pc.printf("File not Found: %s\n\r",filename);
    }
    fseek(jpegfile, 0, SEEK_END);
    jpeg_filesize = ftell(jpegfile);
    jpeg_filepos = 0;
    fseek(jpegfile, 0, SEEK_SET);
    int status = pjpeg_decode_init(&imageInfo, pjpeg_need_bytes_callback, NULL);
    //const unsigned int row_pitch = imageInfo.m_width * imageInfo.m_comps;
    int mcu_x = 0;
    int mcu_y = 0;

    for ( ; ; ) {
        status = pjpeg_decode_mcu();

        if (status) {
            if (status != PJPG_NO_MORE_BLOCKS) {
                //pc.printf("pjpeg_decode_mcu() failed with status %u\n", status);
                fclose(jpegfile);
                return;
            }

            break;
        }

        if (mcu_y >= imageInfo.m_MCUSPerCol) {
            fclose(jpegfile);
            return;
        }

        // Copy MCU's pixel blocks into the destination bitmap.

        for (int y = 0; y < imageInfo.m_MCUHeight; y += 8) {
            const int by_limit = min(8, imageInfo.m_height - (mcu_y * imageInfo.m_MCUHeight + y));
            for (int x = 0; x < imageInfo.m_MCUWidth; x += 8) {

                unsigned int src_ofs = (x * 8U) + (y * 16U);
                const unsigned char *pSrcR = imageInfo.m_pMCUBufR + src_ofs;
                const unsigned char *pSrcG = imageInfo.m_pMCUBufG + src_ofs;
                const unsigned char *pSrcB = imageInfo.m_pMCUBufB + src_ofs;

                const int bx_limit = min(8, imageInfo.m_width - (mcu_x * imageInfo.m_MCUWidth + x));

                if (imageInfo.m_scanType == PJPG_GRAYSCALE) {
                    for (int by = 0; by < by_limit; by++) {
                        for (int bx = 0; bx < bx_limit; bx++) {
                            unsigned int color = ((*pSrcR++) << 16);
                            (*lcd).pixel(mcu_x*imageInfo.m_MCUWidth+x+bx,mcu_y*imageInfo.m_MCUHeight+y+by,color);
                        }
                        pSrcR += (8 - bx_limit);
                    }
                } else {
                    for (int by = 0; by < by_limit; by++) {
                        for (int bx = 0; bx < bx_limit; bx++) {
                            unsigned int color = ((*pSrcR++) << 16) | ((*pSrcG++) << 8) | (*pSrcB++);
                            (*lcd).pixel((130-imageInfo.m_width)/2+mcu_x*imageInfo.m_MCUWidth+x+bx,(130-imageInfo.m_height)/2+mcu_y*imageInfo.m_MCUHeight+y+by,color);
                        }

                        pSrcR += (8 - bx_limit);
                        pSrcG += (8 - bx_limit);
                        pSrcB += (8 - bx_limit);

                    }
                }
            }
        }

        mcu_x++;
        if (mcu_x == imageInfo.m_MCUSPerRow) {
            mcu_x = 0;
            mcu_y++;
        }
    }

    fclose(jpegfile);
}

void wiflyConnect()
{
    wifly.disconnect();
    wifly.init();
    while (!wifly.connect()) {
        pc.printf("waiting\n\r");
    }// join the network
}

void getNewImage(float latitude, float longitude, int zoom)
{
    TCPSocketConnection sock;
    sock.connect("maps.googleapis.com", 80);
    int ret = 0;
    int total_bytes = 0;
    char http_cmd[300];
    sprintf(http_cmd, "GET /maps/api/staticmap?center=%f%%2C%f&zoom=%d&size=130x130&sensor=true&format=jpg-baseline HTTP/1.0\n\n",latitude,longitude,zoom);
    sock.send_all(http_cmd, sizeof(http_cmd)-1);

    sock.set_blocking(false,10000);
    char buffer[300];
    FILE* pFile;
    pc.printf("%s\n\r",http_cmd);
    pFile = fopen ("/local/image.jpg","w");
    pc.printf("Opening File\n\r");
    bool found_start = false;
    bool found_end = false;
    while (true) {
        ret = sock.receive(buffer, sizeof(buffer)-1);
        if (ret <= 0) {
            pc.printf("Ret < 0. Break. \n\r");
            //wiflyConnect();
            break;
        }
        buffer[ret] = '\0';
        total_bytes += ret;
        pc.printf("Received %d chars from server, total %d: %s\n\r", ret,total_bytes, buffer);
        if(found_start) {
            for(int i=1; i<ret; i++) {
                if(buffer[i-1]==0xFF && buffer[i]==0xD9) { //jpg file end at FFD9
                    pc.printf("found end. \n\r");
                    fwrite(buffer,1,i+1,pFile);
                    found_end = true;
                    break;
                }
            }
            if(!found_end)
                fwrite(buffer,1,ret,pFile);
            else {
                pc.printf("Found Start and End. Break. \n\r");
                break;
            }

        } else {
            for(int i=1; i<ret; i++) {
                if(buffer[i-1]==0xFF && buffer[i]==0xD8) { //jpg file starts from FFD8
                    pc.printf("found start.\n\r");
                    fwrite(&buffer[i-1],1,ret-i+1,pFile);
                    found_start = true;
                    break;
                }
            }
        }
    }

    fclose (pFile);
    pc.printf("Closing File\n\r");
    sock.close();
    lcd.cls();
    ReadJPEGFromFile("/local/IMAGE.JPG", &lcd);
}

int main()
{
    pc.printf("Hello\n\r");
    wifly.init(); //Use DHCP
    while (!wifly.connect()) {
        pc.printf("waiting\n\r");
    }// join the network
    pc.printf("IP Address is %s\n", wifly.getIPAddress());

    float longitude = 0.0;
    float latitude = 0.0;
    while(1) {
        if(gps.sample()) {
           // pc.printf("I'm at %f, %f\n", gps.longitude, gps.latitude);
            longitude = gps.longitude;
            latitude = gps.latitude;
        } else {
            pc.printf("Oh Dear! No lock :(\n");
        }
        if(Refresh)
            getNewImage(latitude,longitude,zoom);
        else if(ZoomIn)
            getNewImage(latitude,longitude,++zoom);
        else if(ZoomOut)
            getNewImage(latitude,longitude,--zoom);
    }
    wifly.disconnect();
}