#ifndef ADNS5020EN_H
#define ADNS5020EN_H

#include "mbed.h"

 /** Interface to control a ADNS-5020-EN Avago mouse 
  * chip using a SPI and 2 DigitalOuts
  */

class ADNS5020EN {
public:

   /** Create the mouse chip control interface. Returns Error if the connection is not established.
    *
    * @param mosi A SPI(mosi) pin, for the spi interface of the mouse chip (SDIO)
    * @param miso A SPI(miso) pin, for the spi interface of the mouse chip (SDIO)
    * @param sclk A SPI(sclk) pin, for the spi interface of the mouse chip (SCLK)
    * @param cs A DigitalOut, set low for chip select
    * @param cr A DigitalOut, set los for chip reset
    * @param freq A float value between 1.0 and 0.1 for the SPI frequency(in MHz)
    */
    ADNS5020EN(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName cr, float freq);
    
    
   /** Read the deltaX and deltaY registers and assign to the input variable the counts for the previous read.
    * 
    * @param a_dx A pointer to an integer, to return the value of the deltaX register in reading counts
    * @param a_dy A pointer to an integer, to return the value of the deltaY register in reading counts
    */       
    void read_deltas(int* a_dx, int* a_dy);
    
   /** Same as the read_deltas() function but the return is the movement in mm according to the working resolution.
    * 
    * @param a_dx A pointer to a float, to return the value of the deltaX register in mm
    * @param a_dy A pointer to a float, to return the value of the deltaY register in mm
    * @see read_deltas()
    * @see changeCPI()    
    */     
    void read_deltas_mm(float* dx, float* dy);
    
 
   /** Chip Power Down.
    */
    void end();
    
    
   /** Set new operating frequency to the mouse chip.
    * 
    * @param freq A float value between 1.0 and 0.1 for the SPI frequency(in MHz)
    */    
    void set_freq(float freq);
    
    
   /** Get the false readings of the mouse chip.
    * 
    * @return The false readings
    */         
    int falser();
  
  
   /** Change the resolution of the mouse chip.
    * 
    * @param cpi A bool value where 0 is 500cpi (default) and 1 is 1000cpi 
    * @return The mouse control register value
    */      
    int changeCPI(bool cpi);

protected:

    void creset();
    int cread(int cregister);
    void cwrite(int caddress, int cdata);
    int MChipMotion(int reading);
    float counts_to_mm(int counts);

    SPI _spi;
    DigitalOut _cs;
    DigitalOut _cr;
    int _false_readings;
    int _cpi;

};

#endif // ADNS5020EN_H
