/*
  Hotboards_SpiLcd.cpp - Library to control and write an lcd with spi interface and ST7032 controller.
  base on Arduino's Liquid Cristal library
  Library ported by diego from Hotboards January 16, 2016. 
  and originally cretaed by
  by David A. Mellis
  library modified 5 Jul 2009
  by Limor Fried (http://www.ladyada.net)
  example added 9 Jul 2009
  by Tom Igoe
  modified 22 Nov 2010
  by Tom Igoe
  Released into the public domain.
*/

#include "Hotboards_SpiLcd.h"

Hotboards_SpiLcd::Hotboards_SpiLcd( SPI &spi, PinName cs, PinName rs, PinName rst ) 
    : _spi(spi), _cs_pin(cs), _rs_pin(rs), _rst_pin(rst) 
{
    _cs_pin = 1; /*chip select deactivated*/
    _rs_pin = 0;
    _rst_pin = 1; /*reset not active*/
}


void Hotboards_SpiLcd::init( void )
{
    _rst_pin = 0;
    wait( 0.002 );
    _rst_pin = 1;
    wait( 0.02 );
    command( 0x30 );/*wakeup*/
    wait( 0.002 );
    command( 0x30 );/*wakeup*/
    command( 0x30 );/*wakeup*/
    
    // configure default display functions, two lines, 5x8 charcters
    _displayfunction = HT_SPILCD_8BITMODE | HT_SPILCD_2LINE | HT_SPILCD_5x8DOTS | HT_SPILCD_EXTINST;
    command( HT_SPILCD_FUNCTIONSET | _displayfunction );
    
    command( 0x14 );/*internaloscfrequency*/
    command( 0x56 );/*powercontroll*/
    command( 0x6d );/*followercontrol*/
    wait( 0.2 );
    command( 0x70 );/*constrast*/
    
    // reenable shift functions disbling extended instructions
    _displayfunction &= ~HT_SPILCD_EXTINST;
    command( HT_SPILCD_FUNCTIONSET | _displayfunction );
    
    // turn the display on with no cursor or blinking default
    _displaycontrol = HT_SPILCD_DISPLAYON | HT_SPILCD_CURSOROFF | HT_SPILCD_BLINKOFF;  
    display( );
    
    // Initialize to default text direction (for romance languages)
    _displaymode = HT_SPILCD_ENTRYLEFT | HT_SPILCD_ENTRYSHIFTDECREMENT;
    // set the entry mode
    command( HT_SPILCD_ENTRYMODESET | _displaymode );
    clear( );/*clearscreen*/
}


void Hotboards_SpiLcd::clear( void )
{
    command( HT_SPILCD_CLEARDISPLAY );  // clear display, set cursor position to zero
    wait( 0.002 );          // this command takes a long time!   
}


void Hotboards_SpiLcd::home( void )
{
    command( HT_SPILCD_RETURNHOME );      // set cursor position to zero
    wait( 0.002 );          // this command takes a long time!   
}


void Hotboards_SpiLcd::noDisplay( void )
{
    _displaycontrol &= ~HT_SPILCD_DISPLAYON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );
}


void Hotboards_SpiLcd::display( void )
{
    _displaycontrol |= HT_SPILCD_DISPLAYON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );  
}


void Hotboards_SpiLcd::noBlink( void )
{
    _displaycontrol &= ~HT_SPILCD_BLINKON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );
}    
    

void Hotboards_SpiLcd::blink( void )
{
    _displaycontrol |= HT_SPILCD_BLINKON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );
}



void Hotboards_SpiLcd::noCursor( void )
{
    _displaycontrol &= ~HT_SPILCD_CURSORON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );
}


void Hotboards_SpiLcd::cursor( void )
{
    _displaycontrol |= HT_SPILCD_CURSORON;
    command( HT_SPILCD_DISPLAYCONTROL | _displaycontrol );
  
}


void Hotboards_SpiLcd::scrollDisplayLeft( void )
{
    command( HT_SPILCD_CURSORSHIFT | HT_SPILCD_DISPLAYMOVE | HT_SPILCD_MOVELEFT );
}


void Hotboards_SpiLcd::scrollDisplayRight( void )
{
    command( HT_SPILCD_CURSORSHIFT | HT_SPILCD_DISPLAYMOVE | HT_SPILCD_MOVERIGHT );  
}


void Hotboards_SpiLcd::leftToRight( void )
{
    _displaymode |= HT_SPILCD_ENTRYLEFT;
    command( HT_SPILCD_ENTRYMODESET | _displaymode );
}


void Hotboards_SpiLcd::rightToLeft( void )
{
    _displaymode &= ~HT_SPILCD_ENTRYLEFT;
    command( HT_SPILCD_ENTRYMODESET | _displaymode );  
}


void Hotboards_SpiLcd::autoscroll( void )
{
    _displaymode |= HT_SPILCD_ENTRYSHIFTINCREMENT;
    command( HT_SPILCD_ENTRYMODESET | _displaymode );
}


void Hotboards_SpiLcd::noAutoscroll( void )
{
    _displaymode &= ~HT_SPILCD_ENTRYSHIFTINCREMENT;
    command( HT_SPILCD_ENTRYMODESET | _displaymode );   
}
    

void Hotboards_SpiLcd::setCursor( uint8_t col, uint8_t row )
{
    const uint8_t Buffer[ 2 ]= { 0x00u, 0x40u };
    uint8_t address;
  
    if( row > 1 )
    {
        row = 1;
    }
    address = Buffer[ row ] + col;
    command( 0x80 | address );
}


/*********** mid level commands, for sending data/cmds */

inline void Hotboards_SpiLcd::command( uint8_t value )
{
    _cs_pin = 0;
    send( value, 0 );
    _cs_pin = 1;
    wait( 0.00003 ); // commands need > 27us to settle
}

/*
 * Write a character to the LCD.
 */
int Hotboards_SpiLcd::_putc(int value)
{
    _cs_pin = 0;
    send( value, 1 );
    _cs_pin = 1;
    wait( 0.00003 ); // commands need > 27us to settle
    return 1;
}

int Hotboards_SpiLcd::_getc(void)
{
    return -1;
}

/************ low level data pushing commands **********/

// write either command or data, with automatic 4/8-bit selection
void Hotboards_SpiLcd::send( uint8_t value, uint8_t mode ) 
{
    _rs_pin = mode;
    _spi.write( value );
}
