/* WiFi Example
 * Copyright (c) 2016 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "http_server.h"
#include "http_response_builder.h"
    
#define BUFFER_SIZE 1024*2

WiFiInterface *wifi;

TCPSocket socket;

char buf[BUFFER_SIZE];
unsigned int bufIndex=0;;
bool RxTriggred=false;

/* Flash Control hoko*/
void _eraseFlash(void)
{
	FLASH_EraseInitTypeDef erase;
	erase.TypeErase 	= FLASH_TYPEERASE_SECTORS;	/* Select sector    */
	erase.Sector 		= FLASH_SECTOR_23;       	/* Set sector 23    */
    erase.Banks 		= FLASH_BANK_2;         	/* Erase bank =2    */
	erase.NbSectors 	= 1;		                /* Erase sector num */
	erase.VoltageRange 	= FLASH_VOLTAGE_RANGE_3;	/* Driver 3.3V      */

	uint32_t pageError = 0;

	HAL_FLASHEx_Erase(&erase, &pageError);	    	/* Erase use by HAL_FLASHEx */
}

void writeFlash(uint32_t address, uint8_t *data, uint32_t size)
{
	HAL_FLASH_Unlock();		                    	/* Unlock FLASH */
	_eraseFlash();			                    	/* Erase sector 23 */
	do {
		HAL_FLASH_Program(FLASH_TYPEPROGRAM_BYTE, address, *data);
	} while (++address, ++data, --size);
	HAL_FLASH_Lock();		                    	/* Lock FLASH */
}

void loadFlash(uint32_t address, uint8_t *data, uint32_t size)
{
	memcpy(data, (uint8_t*)address, size);
}

typedef struct {
	char    wifi_ssid[20]       = "WIFI_SSID" ;
	char	wifi_passwd[20]     = "WIFI_PASSWORD" ;
} NTRIP_Condition;

NTRIP_Condition get_ntrip_condition (){

    NTRIP_Condition ntrip_condition;

    mbed_stats_sys_t stats;
    mbed_stats_sys_get(&stats);

    printf("\n ---- ODIN-W2 Status -----\r\n");
    printf("Mbed OS version %d.%d.%d\r\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
    printf("CPU ID: 0x%x\r\n", stats.cpu_id);
    printf("Compiler ID: %d\r\n", stats.compiler_id);
    printf("Compiler Version: %ld\r\n", stats.compiler_version);

	const uint32_t address = 0x81E0000;

    loadFlash(address, (uint8_t*)&ntrip_condition, sizeof(NTRIP_Condition));
    printf (" -------------------------\r\n");
    printf ("| Current C099 conditions |\r\n");
    printf (" -------------------------\r\n");
    printf ("WiFi SSID      = %s \r\n" , ntrip_condition.wifi_ssid );
    printf ("WiFi Password  = %s \r\n" , ntrip_condition.wifi_passwd );

    DigitalIn switch_in0(SW0);              // PUSH = 0 ;
    switch_in0.mode(PullUp);
    int x = switch_in0;
    int y = 1;

    char buffer[4]="OK";

    printf (" --------------------------------------------\r\n");
    printf ("| If you release RESET while turning on SW0, |\r\n");
    printf ("| ODIN will enter the condition change mode. |\r\n");
    printf (" --------------------------------------------\r\n");

    while( (strstr(buffer,"OK") == NULL) || ( x == 0 ) ) // if SW0 is PUSH -> Change Condition
    {
        x = 1 ;
        y = 0 ;
        printf ("Please Input Condition\r\n");

        printf("\r\nWiFi SSID = ");
        scanf ("%s" , ntrip_condition.wifi_ssid );
        printf("%s\r\nWiFi Password = "              ,ntrip_condition.wifi_ssid);
        scanf ("%s" , ntrip_condition.wifi_passwd );
        printf("%s\r\nNtrip Server (ex:rtk2go.com)= ",ntrip_condition.wifi_passwd);
        printf (" -------------------------\r\n");
        printf ("| Changed C099 conditions |\r\n");
        printf (" -------------------------\r\n");
        printf ("WiFi SSID      = %s \r\n" , ntrip_condition.wifi_ssid );
        printf ("WiFi Password  = %s \r\n" , ntrip_condition.wifi_passwd );
        printf (" -----------------------------------------------------------------------\r\n");
        printf ("| If this is all right, enter \"OK\". If not, enter \"NG\". (and ENTER) |\r\n");
        printf (" -----------------------------------------------------------------------\r\n");

        scanf("%s", buffer);
        printf("%s\r\n",buffer);
    } 

	if ( y == 0 ) {
        printf ("Write changed condition\r\n");
        writeFlash(address, (uint8_t*)&ntrip_condition, sizeof(NTRIP_Condition));
    }

    return ntrip_condition;
}


// Requests come in here
void request_handler(ParsedHttpRequest* request, TCPSocket* socket) {

    DigitalOut led(LED1);

    printf("Request came in: %s %s\n", http_method_str(request->get_method()), request->get_url().c_str());

    if (request->get_method() == HTTP_GET && request->get_url() == "/") {
        HttpResponseBuilder builder(200);
        builder.set_header("Content-Type", "text/html; charset=utf-8");

        char response[] = "<html><head><title>Hello from mbed</title></head>"
            "<body>"
                "<h1>mbed webserver</h1>"
                "<button id=\"toggle\">Toggle LED</button>"
                "<script>document.querySelector('#toggle').onclick = function() {"
                    "var x = new XMLHttpRequest(); x.open('POST', '/toggle'); x.send();"
                "}</script>"
            "</body></html>";

        builder.send(socket, response, sizeof(response) - 1);
    }
    else if (request->get_method() == HTTP_POST && request->get_url() == "/toggle") {
        printf("toggle LED called\n");
        led = !led;

        HttpResponseBuilder builder(200);
        builder.send(socket, NULL, 0);
    }
    else {
        HttpResponseBuilder builder(404);
        builder.send(socket, NULL, 0);
    }
}

int main() {

    NTRIP_Condition ntrip_condition = get_ntrip_condition();

    printf("Mbed OS version %d.%d.%d\r\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

    wifi = WiFiInterface::get_default_instance();
    int ret=1;
    ret = wifi->connect(ntrip_condition.wifi_ssid, ntrip_condition.wifi_passwd, NSAPI_SECURITY_WPA_WPA2);
    if (ret != 0) {
        printf("\nConnection error: %d\n", ret);
    }
    else {
        printf("Success\n\n");
        printf("MAC: %s\n", wifi->get_mac_address());
        printf("IP: %s\n", wifi->get_ip_address());
        printf("Netmask: %s\n", wifi->get_netmask());
        printf("Gateway: %s\n", wifi->get_gateway());
        printf("RSSI: %d\n\n", wifi->get_rssi());
    }

    // Connect to the network (see mbed_app.json for the connectivity method used)
    if (!wifi) {
        printf("Cannot connect to the network, see serial output\n");
        return 1;
    }
    
    HttpServer server(wifi);
    nsapi_error_t res = server.start(8080, &request_handler);

    if (res == NSAPI_ERROR_OK) {
        printf("Server is listening at http://%s:8080\n", wifi->get_ip_address());
    }
    else {
        printf("Server could not be started... %d\n", res);
    }

    wait(osWaitForever);
    
}