/* Copyright 2017-present Renesas Electronics Corporation and other contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef IOTJS_MODULE_GRAPHICS_H
#define IOTJS_MODULE_GRAPHICS_H

#include "iotjs_def.h"
#include "iotjs_handlewrap.h"
#include "iotjs_ext_module_periph_common.h"
#include "iotjs_reqwrap.h"

typedef struct iotjs_graphics_platform_data_s iotjs_graphics_platform_data_t;

typedef struct {
  int x;
  int y;
  int width;
  int height;
  char* bitmap;
} graphics_image_t;

typedef struct {
  int startX;
  int startY;
  int endX;
  int endY;
  uint32_t color;
} graphics_line_t;

typedef struct {
  int x;
  int y;
  int width;
  int height;
  uint32_t color;
  bool fill;
} graphics_rect_t;

typedef struct {
  int centerX;
  int centerY;
  int radius;
  int startAngle;
  int endAngle;
  uint32_t color;
} graphics_arc_t;

typedef struct {
  int centerX;
  int centerY;
  int radius;
  uint32_t color;
  bool fill;
} graphics_circle_t;

typedef struct {
  int centerX;
  int centerY;
  int radiusX;
  int radiusY;
  uint32_t color;
  bool fill;
} graphics_ellipse_t;

typedef struct {
  int centerX;
  int centerY;
  int radius;
  int sides;
  uint32_t color;
  bool fill;
} graphics_polygon_t;

typedef struct {
  iotjs_string_t text;
  int x;
  int y;
  int size;
  uint32_t color;
  uint32_t background;
} graphics_text_t;

typedef struct {
  uint8_t* addr;
  uint32_t width;
  uint32_t height;
  display_pixel_format_t format;
} graphics_frame_buffer_t;

typedef struct {
  jerry_value_t jobject;
  iotjs_graphics_platform_data_t* platform_data;

  graphics_frame_buffer_t frame_buffer;

  union {
    graphics_image_t image;
    graphics_line_t line;
    graphics_rect_t rect;
    graphics_arc_t arc;
    graphics_circle_t circle;
    graphics_ellipse_t ellipse;
    graphics_polygon_t polygon;
    graphics_text_t text;
  } shape;
} iotjs_graphics_t;

bool iotjs_graphics_init(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_image(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_line(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_rect(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_arc(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_circle(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_ellipse(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_polygon(iotjs_graphics_t* graphics);
bool iotjs_graphics_draw_text(iotjs_graphics_t* graphics);

void iotjs_graphics_create_platform_data(iotjs_graphics_t* graphics);
void iotjs_graphics_destroy_platform_data(iotjs_graphics_platform_data_t* pdata);
jerry_value_t iotjs_graphics_set_platform_config(iotjs_graphics_t* graphics,
                                             const jerry_value_t jconfig);

#endif /* IOTJS_MODULE_GRAPHICS_H */
