/* Copyright 2017-present Renesas Electronics Corporation and other contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "iotjs_def.h"

#if defined(USE_POSIX_BRIDGE)
  #include <stdlib.h>
  #include <nic.h>

JS_FUNCTION(Ifup) {
  JS_CHECK_ARGS(2, string, string);

  iotjs_string_t nic    = JS_GET_ARG(0, string);
  iotjs_string_t params = JS_GET_ARG(1, string);
  const char* cstr_nic    = iotjs_string_data( &nic );
  const char* cstr_params = iotjs_string_data( &params );
  int rc = nic_ifup( cstr_nic, cstr_params );
  iotjs_string_destroy( &params );
  iotjs_string_destroy( &nic );

  return jerry_create_boolean( rc == 0 );
}

JS_FUNCTION(Ifdown) {
  JS_CHECK_ARGS(1, string);

  iotjs_string_t nic = JS_GET_ARG(0, string);
  const char* cstr_nic = iotjs_string_data( &nic );
  int rc = nic_ifdown( cstr_nic );
  iotjs_string_destroy( &nic );

  return jerry_create_boolean( rc == 0 );
}

JS_FUNCTION(Ifconfig) {
  JS_CHECK_ARGS(1, string);

  iotjs_string_t nic = JS_GET_ARG(0, string);
  const char* cstr_nic = iotjs_string_data( &nic );
  char* json_conf = nic_ifconfig( cstr_nic );
  iotjs_string_t str = iotjs_string_create_with_size( json_conf, strlen( json_conf ) );
  jerry_value_t rc = iotjs_jval_create_string( &str );
  iotjs_string_destroy( &str );
  iotjs_string_destroy( &nic );
  free( json_conf );
  return rc;
}

JS_FUNCTION(Ntpdate) {
  JS_CHECK_ARGS(2, string, object);

  iotjs_string_t nic = JS_GET_ARG(0, string);
  const jerry_value_t obj = JS_GET_ARG(1, object);

  jerry_value_t prop_server = iotjs_jval_get_property( obj, "server" );
  iotjs_string_t server = iotjs_jval_as_string( prop_server );
  jerry_value_t prop_port = iotjs_jval_get_property( obj, "port" );
  int port = (int)( iotjs_jval_as_number( prop_port ) );
  jerry_value_t prop_timeout = iotjs_jval_get_property( obj, "timeout" );
  int timeout = (int)( iotjs_jval_as_number( prop_timeout ) );

  const char* cstr_nic = iotjs_string_data( &nic );
  const char* cstr_server = iotjs_string_data( &server );
  int rc = nic_ntpdate( cstr_nic, cstr_server, port, timeout );

  jerry_release_value( prop_timeout );
  jerry_release_value( prop_port );
  iotjs_string_destroy( &server );
  jerry_release_value( prop_server );
  iotjs_string_destroy( &nic );

  return jerry_create_boolean( rc == 0 );
}

JS_FUNCTION(Enumerate) {
  char* json_nics = nic_enumerate();
  iotjs_string_t str = iotjs_string_create_with_size( json_nics, strlen( json_nics ) );
  jerry_value_t rc = iotjs_jval_create_string( &str );
  iotjs_string_destroy( &str );
  free( json_nics );
  return rc;
}

#endif	// #if defined(USE_POSIX_BRIDGE)


jerry_value_t InitNic() {
  jerry_value_t nic = jerry_create_object();

#if defined(USE_POSIX_BRIDGE)
  iotjs_jval_set_method(nic, "ifup", Ifup);
  iotjs_jval_set_method(nic, "ifdown", Ifdown);
  iotjs_jval_set_method(nic, "ifconfig", Ifconfig);
  iotjs_jval_set_method(nic, "ntpdate", Ntpdate);
  iotjs_jval_set_method(nic, "enumerate", Enumerate);
#endif	// #if defined(USE_POSIX_BRIDGE)

  return nic;
}
