#include "mbed.h"
#include "MMA8451Q.h"
#include "MAG3110.h"
#include "TSISensor.h"
#include "SLCD.h"

#define ACC_ID 0
#define MAG_ID 1
#define ADC_ID 2
#define TSI_ID 3
#define NUM_OF_SENSOR 4

union Serial_Float{
    float f;
    char bt[4];
};

union Serial_Int{
    int i;
    char bt[4];
};

typedef union _data {
  float f;
  char  s[4];
} myData;

Serial pc(USBTX, USBRX);
MMA8451Q acc(PTE25,PTE24,0x1d<<1);
MAG3110 mag(PTE25,PTE24);
AnalogIn adc(PTE30);
TSISensor tsi;
SLCD slcd;
DigitalOut led2(LED2);
bool sirq_flags[NUM_OF_SENSOR] = {0}; // interrupt flags
int SR = 0; // total sampling rate

void send(float data, int id);
void send(int data, int id);
void read_cmd();
void display(int val);

Ticker acc_timer;
Ticker mag_timer;
Ticker adc_timer;
Ticker tsi_timer;
void acc_timer_irq();
void mag_timer_irq();
void adc_timer_irq();
void tsi_timer_irq();

// initial sampling frequency:
float sp_freq[NUM_OF_SENSOR] = {1,1,1,1};
// current sensor states:
int sp_flag[NUM_OF_SENSOR] = {0};

int main(){
    // Initialize sampling rate for display
    SR = 3*sp_freq[0]*sp_flag[0]+sp_freq[1]*sp_flag[1]+sp_freq[2]*sp_flag[2]+sp_freq[3]*sp_flag[3];
    led2 = 1;
    // set baud rate to 460800
    pc.baud(460800);
    // display current sampling rate on LCD
    display(SR);
    while(1){
        // check requests, execute and clear
        if(sirq_flags[ACC_ID]){
            send(acc.getAccX(),0);
            send(acc.getAccY(),1);
            send(acc.getAccZ(),2);
            sirq_flags[ACC_ID] = 0;
        }
        if(sirq_flags[MAG_ID]){
            send(mag.readVal(0x1d),3);
            sirq_flags[MAG_ID] = 0;
        }
        if(sirq_flags[ADC_ID]){
            send(float(adc.read()*3.3),4);
            sirq_flags[ADC_ID] = 0;
        }
        if(sirq_flags[TSI_ID]){
            send(tsi.readPercentage(),5);
            sirq_flags[TSI_ID] = 0;
        }
        if(pc.readable()){
            read_cmd();
            // refresh current sampling rate after the command is read:
            SR = 3*sp_freq[0]*sp_flag[0]+sp_freq[1]*sp_flag[1]+sp_freq[2]*sp_flag[2]+sp_freq[3]*sp_flag[3];
            display(SR);
        }
    }
}

// interrupt functions for each ticker
void acc_timer_irq(){
    sirq_flags[ACC_ID] = 1;
}
void mag_timer_irq(){
    sirq_flags[MAG_ID] = 1;
}
void adc_timer_irq(){
    sirq_flags[ADC_ID] = 1;
}
void tsi_timer_irq(){
    sirq_flags[TSI_ID] = 1;
}

// serial write (overloaded with different data type)
void send(float data,int id){
    Serial_Float srdata;
    srdata.f = data;
    for(int i = 0; i < 4; i++)
        pc.putc(srdata.bt[i]);
    pc.putc('0'+id);
}
void send(int data, int id){
    Serial_Int srdata;
    srdata.i = data;
    for(int i = 0; i < 4; i++)
        pc.putc(srdata.bt[i]);
    pc.putc('0'+id);
}

// read command
void read_cmd(){
    __disable_irq();
    led2 = !led2;
    char op; char id; myData freq;
    while(pc.readable()){//Flush input
        pc.getc();
    }
    pc.putc('r'); //confirm ready
    op = pc.getc(); // get operation
    id = pc.getc(); // get sensor id
    if(op == 'c') {
        if (id == '0'){
            acc_timer.detach();
            sp_flag[ACC_ID] = 0;
        }
        else if (id == '1'){
            mag_timer.detach();
            sp_flag[MAG_ID] = 0;
        }
        else if (id == '2'){
            adc_timer.detach();
            sp_flag[ADC_ID] = 0;
        }
        else if (id == '3'){
            tsi_timer.detach();
            sp_flag[TSI_ID] = 0;
        }
    }
    if(op == 'o') {
        for (int i=0; i<4; i++) {
            freq.s[i] = pc.getc();
        }
        switch(id){
            case '0':
                sp_freq[0] = freq.f;
                acc_timer.attach(&acc_timer_irq, 1.0/freq.f);
                sp_flag[ACC_ID] = 1; 
                break;
            case '1':
                sp_freq[1] = freq.f;
                mag_timer.attach(&mag_timer_irq, 1.0/freq.f); 
                sp_flag[MAG_ID] = 1;
                break;
            case '2':
                sp_freq[2] = freq.f;
                adc_timer.attach(&adc_timer_irq, 1.0/freq.f);
                sp_flag[ADC_ID] = 1;
                break;
            case '3':
                sp_freq[3] = freq.f;
                tsi_timer.attach(&tsi_timer_irq, 1.0/freq.f); 
                sp_flag[TSI_ID] = 1;
                break;
        }
    }
    led2 = !led2;
    __enable_irq();
}

// LCD function
void display(int val){
    char buf[16];
    sprintf(buf,"%d%d%d",val/100,(val%100)/10,val%10);
    slcd.printf(buf);
    slcd.Home();
}