/**
 ******************************************************************************
 * @file    Display.h
 * @author  AST / EST
 * @version V0.0.1
 * @date    14-April-2015
 * @brief   Header file for display
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/

#ifndef __DISPLAY_H
#define __DISPLAY_H

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include "stmpe1600_class.h"
#include "DevI2C.h"

#ifdef __cplusplus
extern "C" {
#endif

#define FAST_DISPLAY

#define DP  (1<<7)
#define NOT_7_NO_DP( ... ) (uint8_t) ~( __VA_ARGS__ + DP )
#define S0 (1<<0)
#define S1 (1<<1)
#define S2 (1<<2)
#define S3 (1<<3)
#define S4 (1<<4)
#define S5 (1<<5)
#define S6 (1<<6)

/* refer to http://www.twyman.org.uk/Fonts/ */
static const uint8_t ascii_to_display_lut[256]={
      [' ']=           0,
      ['-']=           S6,
      ['_']=           S3,
      ['=']=           S3+S6,
      ['~']=           S0+S3+S6, /* 3 h bar */
      ['^']=           S0, /* use as top bar */

      ['?']=           NOT_7_NO_DP(S5+S3+S2),
      ['*']=           NOT_7_NO_DP(),
      ['[']=           S0+S3+S4+S5,
      [']']=           S0+S3+S2+S1,
      ['@']=           S0+S3,

      ['0']=           NOT_7_NO_DP(S6),
      ['1']=           S1+S2,
      ['2']=           S0+S1+S6+S4+S3,
      ['3']=           NOT_7_NO_DP(S4+S5),
      ['4']=           S5+S1+S6+S2,
      ['5']=           NOT_7_NO_DP(S1+S4),
      ['6']=           NOT_7_NO_DP(S1),
      ['7']=           S0+S1+S2,
      ['8']=           NOT_7_NO_DP(0),
      ['9']=           NOT_7_NO_DP(S4),
    
      ['a']=           S2+ S3+ S4+ S6 ,
      ['b']=           NOT_7_NO_DP(S0+S1),
      ['c']=           S6+S4+S3,
      ['d']=           NOT_7_NO_DP(S0+S5),
      ['e']=           NOT_7_NO_DP(S2),
      ['f']=           S6+S5+S4+S0, /* same as F */
      ['g']=           NOT_7_NO_DP(S4), /* same as 9 */
      ['h']=           S6+S5+S4+S2,
      ['i']=           S4,
      ['j']=           S1+S2+S3+S4,
      ['k']=           S6+S5+S4+S2, /* a h */
      ['l']=           S3+S4,
      ['m']=           S0+S4+S2, /* same as  */
      ['n']=           S2+S4+S6,
      ['o']=           S6+S4+S3+S2,
      ['p']=           NOT_7_NO_DP(S3+S2), // same as P
      ['q']=           S0+S1+S2+S5+S6,
      ['r']=           S4+S6,
      ['s']=           NOT_7_NO_DP(S1+S4),
      ['t']=           NOT_7_NO_DP(S0+S1+S2),
      ['u']=           S4+S3+S2+S5+S1, // U
      ['v']=           S4+S3+S2, // is u but u use U
      ['w']=           S1+S3+S5,
      ['x']=           NOT_7_NO_DP(S0+S3), // similar to H
      ['y']=           NOT_7_NO_DP(S0+S4),
      ['z']=           S0+S1+S6+S4+S3, // same as 2
      
      ['A']=           NOT_7_NO_DP(S3),
      ['B']=           NOT_7_NO_DP(S0+S1), /* as b  */
      ['C']=           S0+S3+S4+S5, // same as [
      ['E']=           NOT_7_NO_DP(S1+S2),
      ['F']=           S6+S5+S4+S0,
      ['G']=           NOT_7_NO_DP(S4), /* same as 9 */
      ['H']=           NOT_7_NO_DP(S0+S3),
      ['I']=           S1+S2,
      ['J']=           S1+S2+S3+S4,
      ['K']=           NOT_7_NO_DP(S0+S3), /* same as H */
      ['L']=           S3+S4+S5,
      ['M']=           S0+S4+S2, /* same as  m*/
      ['N']=           S2+S4+S6, /* same as n*/
      ['O']=           NOT_7_NO_DP(S6),
      ['P']=           S0+S1+S2+S5+S6, // sames as 'q'
      ['Q']=           NOT_7_NO_DP(S3+S2),
      ['R']=           S4+S6,
      ['S']=           NOT_7_NO_DP(S1+S4), /* sasme as 5 */
      ['T']=           NOT_7_NO_DP(S0+S1+S2), /* sasme as t */
      ['U']=           NOT_7_NO_DP(S6+S0),
      ['V']=           S4+S3+S2, // is u but u use U
      ['W']=           S1+S3+S5,
      ['X']=           NOT_7_NO_DP(S0+S3), // similar to H
      ['Y']=           NOT_7_NO_DP(S0+S4),
      ['Z']=           S0+S1+S6+S4+S3, // same as 2


};

#undef S0
#undef S1
#undef S2
#undef S3
#undef S4
#undef S5
#undef S6
#undef DP

#define DISPLAY_DELAY   1  // in mSec 		

/* Classes -------------------------------------------------------------------*/
/** Class representing Display
 */ 
 
class Display
{
 public:

#ifndef FAST_DISPLAY
 
    Display(DevI2C &i2c, STMPE1600DigiOut &D1,	STMPE1600DigiOut &D2, STMPE1600DigiOut &D3, STMPE1600DigiOut &D4,
	    STMPE1600DigiOut &D_A, STMPE1600DigiOut &D_B, STMPE1600DigiOut &D_C, STMPE1600DigiOut &D_D,
	    STMPE1600DigiOut &D_E, STMPE1600DigiOut &D_F, STMPE1600DigiOut &D_G)
	    : Display_D1(D1), Display_D2(D2), Display_D3(D3), Display_D4(D4), Display_A(D_A), Display_B(D_B), 
            Display_C(D_C), Display_D(D_D), Display_E(D_E), Display_F(D_F), Display_G(D_G), dev_i2c(i2c)
	    {
	       pDisplay_DGT[0] = &Display_D1;
	       pDisplay_DGT[1] = &Display_D2;
	       pDisplay_DGT[2] = &Display_D3;
	       pDisplay_DGT[3] = &Display_D4;	
	       pDisplay_SEG[0] = &Display_A;
	       pDisplay_SEG[1] = &Display_B;
	       pDisplay_SEG[2] = &Display_C;
	       pDisplay_SEG[3] = &Display_D;
	       pDisplay_SEG[4] = &Display_E;
	       pDisplay_SEG[5] = &Display_F;
	       pDisplay_SEG[6] = &Display_G;		

	       for (int i=0; i<4; i++) { *pDisplay_DGT[i]=1; }
	       for (int i=0; i<7; i++) { *pDisplay_SEG[i]=1; }		
	    }
#else		
    Display(DevI2C &i2c) :  dev_i2c(i2c) {}
#endif		

#ifndef FAST_DISPLAY			
    void Digits_off (void)  // all 4 digits off
    {
       Display_D1=1;
       Display_D2=1; 
       Display_D3=1; 
       Display_D4=1;  			
    }

    void Digits_on (void)  // all 4 digits on
    {
       Display_D1=0;
       Display_D2=0; 
       Display_D3=0; 
       Display_D4=0;  			
    }

    void Digit_on (int digit)  // digits on
    {
       *pDisplay_DGT[digit] = 0;
    }

    void Digit_off (int digit)  // digits off
    {
       *pDisplay_DGT[digit] = 1;
    }
		
    void Segments_off ()   // all 7 segments off			
    {
       Display_A=1;
       Display_B=1;
       Display_C=1;
       Display_D=1;
       Display_E=1;
       Display_F=1;			
       Display_G=1;						
    }		
		
    void Segment_off (int seg)   // segment off			
    {
       *pDisplay_SEG[seg] = 1;
    }
		
    void Segment_on (int seg)   // segment on			
    {
       *pDisplay_SEG[seg] = 0;
    }
		
		
    void DisplayString (char str[4])
    {
       int i;
       const char *pc;
 
       for( i=0, pc=str; i<4 && *pc!=0 ; i++, pc++)
       {
          _V2_Set7Segment( ascii_to_display_lut[(uint8_t)*pc], i);
          if( *(pc+1)== '.')
          {
             pc++;
          }
          wait_ms(DISPLAY_DELAY);
          Digits_off();
       }			
    }
				
 
    void _V2_Set7Segment( int Leds, int digit )
    {
       //Digits_off();
       Segments_off();     // 7 seg off */
       int mask=1;
       for (int i=0; i<7; i++) 
       { 
          if (Leds & mask) Segment_on(i);
          mask = mask<<1;
       }		 
       Digit_on(digit);
    }


#else

    void DisplayDigit (char *val, char dgt)	
    {
       uint8_t data[2];
       uint16_t *pdata = (uint16_t*)data;				
       dev_i2c.i2c_read(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7					
       *pdata = *pdata | (uint16_t)0x0780;				// all digits off
       dev_i2c.i2c_write(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7		 
			
       _V2_Set7Segment( ascii_to_display_lut[(uint8_t)*val], dgt);						
    }
		
    void DisplayString (char str[4], char strlen)
    {
       int i, dgt;
       const char *pc;
       uint8_t data[2];
       uint16_t *pdata = (uint16_t*)data;	
			
       //dev_i2c.i2c_read(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7					
       //*pdata = *pdata | (uint16_t)0x0780;				// all digits off
       //dev_i2c.i2c_write(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7		 
			
       for(i=0, dgt=4-strlen, pc=str; i<strlen && *pc!=0; i++, pc++, dgt++)
       {
          _V2_Set7Segment( ascii_to_display_lut[(uint8_t)*pc], dgt);
          if( *(pc+1)== '.')
          {
             pc++;
          }
          wait_ms(DISPLAY_DELAY);
          dev_i2c.i2c_read(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7
          *pdata = *pdata | (uint16_t)0x0780;				// all digits off
          dev_i2c.i2c_write(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7	
       }			
    }
				
 
    void _V2_Set7Segment( int Leds, int digit )
    {
       //Digits_off();
       uint16_t dgt;
       dgt = 1<<digit;
       dgt = ((uint16_t)dgt)<<7;
       dgt = ~dgt;	
       uint8_t data[2];
       uint16_t *pdata = (uint16_t*)data;			 

		 
       /* set the exppinname state to lvl */
       dev_i2c.i2c_read(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7  		 
       *pdata = *pdata  | (uint16_t)0x007F;      // 7 segments off
       dev_i2c.i2c_write(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7		 
		 
       int mask=1;
       for (int i=0; i<7; i++) 
       { 
          if (Leds & mask) *pdata = *pdata & ~(uint16_t)mask;  
          mask = mask<<1;
       }		 
       *pdata = *pdata & dgt;
       dev_i2c.i2c_write(data, (uint8_t)STMPE1600_DEF_DEVICE_ADDRESS, (uint8_t)GPSR_0_7, 2);		//  GPSR_0_7		 			
   }
#endif		
		
		
 private:		
		
#ifndef FAST_DISPLAY
   STMPE1600DigiOut &Display_D1;
   STMPE1600DigiOut &Display_D2;
   STMPE1600DigiOut &Display_D3;
   STMPE1600DigiOut &Display_D4;
   STMPE1600DigiOut *pDisplay_DGT[4];		// the 4 display digits
		
   STMPE1600DigiOut &Display_A;
   STMPE1600DigiOut &Display_B;
   STMPE1600DigiOut &Display_C;
   STMPE1600DigiOut &Display_D;
   STMPE1600DigiOut &Display_E; 
   STMPE1600DigiOut &Display_F;
   STMPE1600DigiOut &Display_G; 
   STMPE1600DigiOut *pDisplay_SEG[7];		// the 7 digit segmments
#endif
    DevI2C &dev_i2c; 		
};

#ifdef __cplusplus
}
#endif
#endif // __DISPLAY_H