/**
 ******************************************************************************
 * @file    vl6180x_class.h
 * @author  AST / EST
 * @version V0.0.1
 * @date    14-April-2015
 * @brief   Header file for component VL6180X
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/

#ifndef __VL6180X_CLASS_H
#define __VL6180X_CLASS_H

/* Includes ------------------------------------------------------------------*/
#include "RangeSensor.h"
#include "LightSensor.h"
#include "DevI2C.h" 
#include "vl6180x_api.h"
#include "vl6180x_cfg.h"
#include "vl6180x_def.h"
#include "vl6180x_types.h"
#include "vl6180x_platform.h"
//#include "vl6180x_appcfg.h"
#include "stmpe1600_class.h"

/* data struct containing range measure, light measure and type of error provided to the user
   in case of invalid data range_mm=0xFFFFFFFF and lux=0xFFFFFFFF */	
typedef struct MeasureData 
{
   uint32_t range_mm;
   uint32_t lux;
   uint32_t range_error;
   uint32_t als_error;
   uint32_t int_error;
}MeasureData_t;

/* sensor operating modes */ 
typedef enum
{
   range_single_shot_polling=1,
   als_single_shot_polling,
   range_continuous_polling,
   als_continuous_polling,
   range_continuous_interrupt,
   als_continuous_interrupt,
   interleaved_mode_interrupt,
   range_continuous_polling_low_threshold,
   range_continuous_polling_high_threshold,
   range_continuous_polling_out_of_window,
   als_continuous_polling_low_threshold,
   als_continuous_polling_high_threshold,
   als_continuous_polling_out_of_window,
   range_continuous_interrupt_low_threshold,
   range_continuous_interrupt_high_threshold,
   range_continuous_interrupt_out_of_window,
   als_continuous_interrupt_low_threshold,
   als_continuous_interrupt_high_threshold,
   als_continuous_interrupt_out_of_window,
   range_continuous_als_single_shot,
   range_single_shot_als_continuous,
}OperatingMode;

/** default device address */
#define DEFAULT_DEVICE_ADDRESS		0x29

/* Classes -------------------------------------------------------------------*/
/** Class representing a VL6180X sensor component
 */
class VL6180X : public RangeSensor, public LightSensor 
{
 public:
    /** Constructor
     * @param[in] i2c device I2C to be used for communication
     * @param[in] digital out pin/STMPE1600DigiOut pin to be used for GPIO expander communication
     * @param[in] gpio1 pin
     * @param[in] device address, 0x29 by default  
     */
    VL6180X(DevI2C &i2c, DigitalOut &pin, PinName pin_gpio1, uint8_t DevAddr=DEFAULT_DEVICE_ADDRESS) : RangeSensor(), LightSensor(), dev_i2c(i2c), gpio0(&pin), interrupt_measure(pin_gpio1)
    {
       MyDevice.I2cAddr=DevAddr;		 
       MyDevice.Present=0;
       MyDevice.Ready=0;
       Device=&MyDevice;;
       expgpio0=NULL;
    }  
		
    VL6180X(DevI2C &i2c, STMPE1600DigiOut &pin, PinName pin_gpio1, uint8_t DevAddr=DEFAULT_DEVICE_ADDRESS) : RangeSensor(), LightSensor(), dev_i2c(i2c), expgpio0(&pin), interrupt_measure(pin_gpio1)
    {
       MyDevice.I2cAddr=DevAddr;		 
       MyDevice.Present=0;
       MyDevice.Ready=0;
       Device=&MyDevice;
       gpio0=NULL;			 		 
    }  	 
    
   /** Destructor
    */
    //virtual ~VL6180X(){} 
    /* warning: VL6180X class inherits from GenericSensor, RangeSensor and LightSensor, that haven`t a destructor.
       The warning should request to introduce a virtual destructor to make sure to delete the object */
		
    /* turns on the sensor */		 
    void VL6180x_On(void)
    {
       if(gpio0) 
	  *gpio0=1;
       else if(expgpio0) 
	  *expgpio0=1;
    } 
		
    /* turns off the sensor */
    void VL6180x_Off(void) 
    {
       if(gpio0) 
	  *gpio0=0;
       else if(expgpio0) 
	  *expgpio0=0;			
    }
		
    int InitSensor(uint8_t NewAddr);
    int StartMeasurement(OperatingMode operating_mode, void (*fptr)(void), uint16_t low, uint16_t high);
    int GetMeasurement(OperatingMode operating_mode, MeasureData_t *Data);		
    int StopMeasurement(OperatingMode operating_mode);

    /* handling functions of the interrupt_measure */
		
    /** Attach a function to call when an interrupt is detected, i.e. measurement is ready
     *
     *  @param[in] fptr A pointer to a void function, or 0 to set as none
     */
    void AttachInterruptMeasureDetectionIRQ(void (*fptr)(void))
    {
       interrupt_measure.rise(fptr);
    }

    /** Enable interrupt measure IRQ
     */
    void EnableInterruptMeasureDetectionIRQ(void) 
    {
       interrupt_measure.enable_irq();
    }

    /** Disable interrupt measure IRQ
     */
    void DisableInterruptMeasureDetectionIRQ(void) 
    {
       interrupt_measure.disable_irq();
    }
		
    int HandleIRQ(OperatingMode operating_mode, MeasureData_t *Data);
		
    unsigned Present()
    {
       return Device->Present;
    }
		
    /* Wrapper functions */	
    int WaitDeviceBooted()
    {
       return VL6180x_WaitDeviceBooted(Device);
    }
		
    int Init() 
    {
       return VL6180x_InitData(Device);
    }
		
    int SetupGPIO1(uint8_t InitFunction, int ActiveHigh)
    {
       return VL6180x_SetupGPIO1(Device, InitFunction, ActiveHigh);
    }
		
    int Prepare()
    {
       return VL6180x_Prepare(Device);
    }
		
    int RangeStartContinuousMode()
    {
       return VL6180x_RangeStartContinuousMode(Device);
    }
		
    int RangeStartSingleShot()
    {
       return VL6180x_RangeStartSingleShot(Device);
    }
		
    int RangeSetMaxConvergenceTime(uint8_t MaxConTime_msec)
    {
       return VL6180x_RangeSetMaxConvergenceTime(Device, MaxConTime_msec);
    }
		
    int RangePollMeasurement(VL6180x_RangeData_t *pRangeData)
    {
       return VL6180x_RangePollMeasurement(Device, pRangeData);
    }
		
    int RangeGetMeasurementIfReady(VL6180x_RangeData_t *pRangeData)
    {
       return VL6180x_RangeGetMeasurementIfReady(Device, pRangeData);
    }
		
    int RangeGetMeasurement(VL6180x_RangeData_t *pRangeData)
    {
       return VL6180x_RangeGetMeasurement(Device, pRangeData);
    }
		
    int GetRange(int32_t *piData)
    {
       return VL6180x_RangeGetResult(Device, piData);
    }
		
    int RangeConfigInterrupt(uint8_t ConfigGpioInt)
    {
       return VL6180x_RangeConfigInterrupt(Device, ConfigGpioInt);
    }
		
    int RangeGetInterruptStatus(uint8_t *pIntStatus)
    {
       return VL6180x_RangeGetInterruptStatus(Device, pIntStatus);
    }
		
    int AlsPollMeasurement(VL6180x_AlsData_t *pAlsData)
    {
       return VL6180x_AlsPollMeasurement(Device, pAlsData);
    }
		
    int AlsGetMeasurement(VL6180x_AlsData_t *pAlsData)
    {
       return VL6180x_AlsGetMeasurement(Device, pAlsData);
    }
	
    int AlsConfigInterrupt(uint8_t ConfigGpioInt)
    {
       return VL6180x_AlsConfigInterrupt(Device, ConfigGpioInt);
    }
		
    int AlsSetIntegrationPeriod(uint16_t period_ms)
    {
       return VL6180x_AlsSetIntegrationPeriod(Device, period_ms);	
    }
		
    int AlsSetInterMeasurementPeriod(uint16_t intermeasurement_period_ms)
    {
       return VL6180x_AlsSetInterMeasurementPeriod(Device, intermeasurement_period_ms);
    }

    int AlsSetAnalogueGain(uint8_t gain)
    {
       return VL6180x_AlsSetAnalogueGain(Device, gain);
    }
		
    int AlsSetThresholds(uint16_t lux_threshold_low, uint16_t lux_threshold_high);

    int AlsGetInterruptStatus(uint8_t *pIntStatus)
    {
       return VL6180x_AlsGetInterruptStatus(Device, pIntStatus);
    }

    int StaticInit()
    {
       return VL6180x_StaticInit(Device);
    }
		
    int RangeWaitDeviceReady(int MaxLoop )
    {
       return VL6180x_RangeWaitDeviceReady(Device, MaxLoop);
    }
		
    int RangeSetInterMeasPeriod(uint32_t  InterMeasTime_msec)
    {
       return VL6180x_RangeSetInterMeasPeriod(Device, InterMeasTime_msec);
    }
		
    int UpscaleSetScaling(uint8_t scaling)
    {
       return VL6180x_UpscaleSetScaling(Device, scaling);
    }
		
    int UpscaleGetScaling()
    {
       return VL6180x_UpscaleGetScaling(Device);
    }
		
    uint16_t GetUpperLimit()
    {
       return VL6180x_GetUpperLimit(Device);
    }
		
    int RangeSetThresholds(uint16_t low, uint16_t high, int SafeHold)
    {
       return VL6180x_RangeSetThresholds(Device, low, high, SafeHold);
    }

    int RangeGetThresholds(uint16_t *low, uint16_t *high)
    {
       return VL6180x_RangeGetThresholds(Device, low, high);
    }
			
    int RangeSetRawThresholds(uint8_t low, uint8_t high)
    {
       return VL6180x_RangeSetRawThresholds(Device, low, high);
    }
		
    int RangeSetEceFactor(uint16_t  FactorM, uint16_t FactorD)
    {
       return VL6180x_RangeSetEceFactor(Device, FactorM, FactorD);
    }
		
    int RangeSetEceState(int enable)
    {
       return VL6180x_RangeSetEceState(Device, enable);
    }
			
    int FilterSetState(int state)
    {
       return VL6180x_FilterSetState(Device, state);
    }
			
    int FilterGetState()
    {
       return VL6180x_FilterGetState(Device);
    }
		
    int DMaxSetState(int state)
    {
       return VL6180x_DMaxSetState(Device, state);
    }
		
    int DMaxGetState()
    {
       return VL6180x_DMaxGetState(Device);
    }
		
    int RangeSetSystemMode(uint8_t mode)
    {
       return VL6180x_RangeSetSystemMode(Device, mode);
    }
		
    int8_t GetOffsetCalibrationData()
    {
       return VL6180x_GetOffsetCalibrationData(Device);
    }
		
    void SetOffsetCalibrationData(int8_t offset)
    {
       return VL6180x_SetOffsetCalibrationData(Device, offset);
    }
		
    int SetXTalkCompensationRate(FixPoint97_t Rate)
    {
       return VL6180x_SetXTalkCompensationRate(Device, Rate);
    }
		
    int AlsWaitDeviceReady(int MaxLoop)
    {
       return VL6180x_AlsWaitDeviceReady(Device, MaxLoop);
    }
		
    int AlsSetSystemMode(uint8_t mode)
    {
       return VL6180x_AlsSetSystemMode(Device, mode);
    }

    int SetGroupParamHold(int Hold)
    {
       return VL6180x_SetGroupParamHold(Device, Hold);
    }		
		
    int SetI2CAddress(int NewAddr)
    {
       int status;
			
       status=VL6180x_SetI2CAddress(Device, NewAddr);
       if(!status)
          Device->I2cAddr=NewAddr;
       return status;
    }
		
    int SetupGPIOx(int pin, uint8_t IntFunction, int ActiveHigh)
    {
       return VL6180x_SetupGPIOx(Device, pin, IntFunction, ActiveHigh);
    }
		
    int SetGPIOxPolarity(int pin, int active_high)
    {
       return VL6180x_SetGPIOxPolarity(Device, pin, active_high);
    }
		 
    int SetGPIOxFunctionality(int pin, uint8_t functionality)
    {
       return VL6180x_SetGPIOxFunctionality(Device, pin, functionality);
    }
	
    int DisableGPIOxOut(int pin)
    {
       return VL6180x_DisableGPIOxOut(Device, pin);
    }
		
    int GetInterruptStatus(uint8_t *status)
    {
       return VL6180x_GetInterruptStatus(Device, status);
    }
		
    int ClearInterrupt(uint8_t IntClear)
    {
       return VL6180x_ClearInterrupt(Device, IntClear );
    }
		
    int GetLight(uint32_t *piData)
    {
       return VL6180x_AlsGetLux(Device, piData);
    }
		
    int AlsStartContinuousMode()
    {
       return VL6180x_AlsSetSystemMode(Device, MODE_START_STOP|MODE_CONTINUOUS);
    }
    
    int AlsStartSingleShot()
    {
       return VL6180x_AlsSetSystemMode(Device, MODE_START_STOP|MODE_SINGLESHOT);
    }
		
 private:		
    /* api.h functions */
    int VL6180x_WaitDeviceBooted(VL6180xDev_t dev);
    int VL6180x_InitData(VL6180xDev_t dev );
    int VL6180x_SetupGPIO1(VL6180xDev_t dev, uint8_t IntFunction, int ActiveHigh);
    int VL6180x_Prepare(VL6180xDev_t dev);
    int VL6180x_RangeStartContinuousMode(VL6180xDev_t dev);
    int VL6180x_RangeStartSingleShot(VL6180xDev_t dev);
    int VL6180x_RangeSetMaxConvergenceTime(VL6180xDev_t dev, uint8_t  MaxConTime_msec);
    int VL6180x_RangePollMeasurement(VL6180xDev_t dev, VL6180x_RangeData_t *pRangeData);
    int VL6180x_RangeGetMeasurementIfReady(VL6180xDev_t dev, VL6180x_RangeData_t *pRangeData);
    int VL6180x_RangeGetMeasurement(VL6180xDev_t dev, VL6180x_RangeData_t *pRangeData);
    int VL6180x_RangeGetResult(VL6180xDev_t dev, int32_t *pRange_mm);
    int VL6180x_RangeConfigInterrupt(VL6180xDev_t dev, uint8_t ConfigGpioInt);
    int VL6180x_RangeGetInterruptStatus(VL6180xDev_t dev, uint8_t *pIntStatus);
    int VL6180x_AlsPollMeasurement(VL6180xDev_t dev, VL6180x_AlsData_t *pAlsData);
    int VL6180x_AlsGetMeasurement(VL6180xDev_t dev, VL6180x_AlsData_t *pAlsData);
    int VL6180x_AlsConfigInterrupt(VL6180xDev_t dev, uint8_t ConfigGpioInt);
    int VL6180x_AlsSetIntegrationPeriod(VL6180xDev_t dev, uint16_t period_ms);
    int VL6180x_AlsSetInterMeasurementPeriod(VL6180xDev_t dev,  uint16_t intermeasurement_period_ms);
    int VL6180x_AlsSetAnalogueGain(VL6180xDev_t dev, uint8_t gain);
    int VL6180x_AlsSetThresholds(VL6180xDev_t dev, uint16_t low, uint16_t high);
    int VL6180x_AlsGetInterruptStatus(VL6180xDev_t dev, uint8_t *pIntStatus);
    int VL6180x_StaticInit(VL6180xDev_t dev);
    int VL6180x_RangeWaitDeviceReady(VL6180xDev_t dev, int MaxLoop );
    int VL6180x_RangeSetInterMeasPeriod(VL6180xDev_t dev, uint32_t  InterMeasTime_msec);
    int VL6180x_UpscaleSetScaling(VL6180xDev_t dev, uint8_t scaling);
    int VL6180x_UpscaleGetScaling(VL6180xDev_t dev);
    uint16_t VL6180x_GetUpperLimit(VL6180xDev_t dev);
    int VL6180x_RangeSetThresholds(VL6180xDev_t dev, uint16_t low, uint16_t high, int SafeHold);
    int VL6180x_RangeGetThresholds(VL6180xDev_t dev, uint16_t *low, uint16_t *high);
    int VL6180x_RangeSetRawThresholds(VL6180xDev_t dev, uint8_t low, uint8_t high);
    int VL6180x_RangeSetEceFactor(VL6180xDev_t dev, uint16_t  FactorM, uint16_t FactorD);
    int VL6180x_RangeSetEceState(VL6180xDev_t dev, int enable );
    int VL6180x_FilterSetState(VL6180xDev_t dev, int state);
    int VL6180x_FilterGetState(VL6180xDev_t dev);
    int VL6180x_DMaxSetState(VL6180xDev_t dev, int state);
    int VL6180x_DMaxGetState(VL6180xDev_t dev);
    int VL6180x_RangeSetSystemMode(VL6180xDev_t dev, uint8_t mode);
    int8_t VL6180x_GetOffsetCalibrationData(VL6180xDev_t dev);
    void VL6180x_SetOffsetCalibrationData(VL6180xDev_t dev, int8_t offset);
    int VL6180x_SetXTalkCompensationRate(VL6180xDev_t dev, FixPoint97_t Rate);
    int VL6180x_AlsWaitDeviceReady(VL6180xDev_t dev, int MaxLoop );
    int VL6180x_AlsSetSystemMode(VL6180xDev_t dev, uint8_t mode); 
    int VL6180x_SetGroupParamHold(VL6180xDev_t dev, int Hold);
    int VL6180x_SetI2CAddress(VL6180xDev_t dev, uint8_t NewAddr);
    int VL6180x_SetupGPIOx(VL6180xDev_t dev, int pin, uint8_t IntFunction, int ActiveHigh);
    int VL6180x_SetGPIOxPolarity(VL6180xDev_t dev, int pin, int active_high);
    int VL6180x_SetGPIOxFunctionality(VL6180xDev_t dev, int pin, uint8_t functionality);
    int VL6180x_DisableGPIOxOut(VL6180xDev_t dev, int pin);
    int VL6180x_GetInterruptStatus(VL6180xDev_t dev, uint8_t *status);
    int VL6180x_ClearInterrupt(VL6180xDev_t dev, uint8_t IntClear );
		
    /*  Other functions defined in api.c */
    int VL6180x_RangeStaticInit(VL6180xDev_t dev); 
    int VL6180x_UpscaleRegInit(VL6180xDev_t dev);
    int VL6180x_UpscaleStaticInit(VL6180xDev_t dev); 
    int VL6180x_AlsGetLux(VL6180xDev_t dev, lux_t *pLux);
    int _UpscaleInitPatch0(VL6180xDev_t dev); 
    int VL6180x_RangeGetDeviceReady(VL6180xDev_t dev, int * Ready);
    int VL6180x_RangeSetEarlyConvergenceEestimateThreshold(VL6180xDev_t dev);
    int32_t _GetAveTotalTime(VL6180xDev_t dev); 
    int32_t _filter_Start(VL6180xDev_t dev, uint16_t m_trueRange_mm, uint16_t m_rawRange_mm, uint32_t m_rtnSignalRate, uint32_t m_rtnAmbientRate, uint16_t errorCode);
    int _filter_GetResult(VL6180xDev_t dev, VL6180x_RangeData_t *pRangeData);
    int _GetRateResult(VL6180xDev_t dev, VL6180x_RangeData_t *pRangeData); 
    int _DMax_InitData(VL6180xDev_t dev);
		
    /* Read function of the ID device */
    int ReadID();
    
    /* Write and read functions from I2C */
    int VL6180x_WrByte(VL6180xDev_t dev, uint16_t index, uint8_t data);
    int VL6180x_WrWord(VL6180xDev_t dev, uint16_t index, uint16_t data);
    int VL6180x_WrDWord(VL6180xDev_t dev, uint16_t index, uint32_t data);
    int VL6180x_RdByte(VL6180xDev_t dev, uint16_t index, uint8_t *data);
    int VL6180x_RdWord(VL6180xDev_t dev, uint16_t index, uint16_t *data);
    int VL6180x_RdDWord(VL6180xDev_t dev, uint16_t index, uint32_t *data);
    int VL6180x_UpdateByte(VL6180xDev_t dev, uint16_t index, uint8_t AndData, uint8_t OrData);
    int VL6180x_I2CWrite(uint8_t DeviceAddr, uint16_t RegisterAddr, uint8_t *pBuffer, uint16_t NumByteToWrite);
    int VL6180x_I2CRead(uint8_t DeviceAddr, uint16_t RegisterAddr, uint8_t *pBuffer, uint16_t NumByteToRead);
		
		
    int IsPresent()
    {
       int status;
			
       status=ReadID();
       if(status)
          VL6180x_ErrLog("Failed to read ID device. Device not present!\n\r");
       return status;
    }	
    int StopRangeMeasurement(OperatingMode operating_mode);
    int StopAlsMeasurement(OperatingMode operating_mode);
    int GetRangeMeas(OperatingMode operating_mode, MeasureData_t *Data);	
    int GetAlsMeas(OperatingMode operating_mode, MeasureData_t *Data);
    int GetRangeAlsMeas(MeasureData_t *Data);
    int RangeSetLowThreshold(uint16_t threshold);
    int RangeSetHighThreshold(uint16_t threshold);
    int AlsSetLowThreshold(uint16_t threshold);	
    int AlsSetHighThreshold(uint16_t threshold);
    int GetRangeError(MeasureData_t *Data, VL6180x_RangeData_t RangeData);
    int GetAlsError(MeasureData_t *Data, VL6180x_AlsData_t AlsData);
    int RangeMeasPollSingleShot();
    int AlsMeasPollSingleShot();		
    int RangeMeasPollContinuousMode();	
    int AlsMeasPollContinuousMode();
    int AlsGetMeasurementIfReady(VL6180xDev_t dev, VL6180x_AlsData_t *pAlsData);
    int RangeMeasIntContinuousMode(void (*fptr)(void));
    int AlsMeasIntContinuousMode(void (*fptr)(void));
    int InterleavedMode(void (*fptr)(void));
    int StartInterleavedMode();
    int AlsGetThresholds(VL6180xDev_t dev, lux_t *low, lux_t *high);

		
    /* IO Device */
    DevI2C &dev_i2c;
    /* Digital out pin */
    DigitalOut *gpio0;
    /* GPIO expander */
    STMPE1600DigiOut *expgpio0;
    /* Measure detection IRQ */
    InterruptIn interrupt_measure;
    /* Device data */
    MyVL6180Dev_t MyDevice;
    VL6180xDev_t Device;  
};

#endif // __VL6180X_CLASS_H
