/////////////////////////////////////////////////////////////////////
//  Capacitor Dose Meter                                          //
//  for 6CH Sensor Charge / Discharge / Measure                    //
//   Since 2019.11.22  H.Tsunemoto                                 //
//   Sample Program Import (1) Serial_interrupt                    //
//                         (2) Timer_Interrupt(100mSec) Sample     //
//                         (3) Internal_ADC_with_interrupt         //
/////////////////////////////////////////////////////////////////////

#include "mbed.h"
#include "stdio.h"
#include "math.h"
#include "LPC11Uxx.h" 

// MBED NXP-LPC11U24 Bord I/O Port Status Define
Serial device(USBTX, USBRX);
// On_Board Debug LED  
DigitalOut myled1(P1_8);
DigitalOut myled2(P1_9);
DigitalOut myled3(P1_10);
DigitalOut myled4(P1_11);

DigitalOut P36_Int1msec(p36);
DigitalOut P35_SampleTiming(p35);

// Common Relay Connect SW
DigitalOut pCOM_Vref_Conn_p5(p5);
DigitalOut pCOM_GND_Conn_p6(p6);
DigitalOut pCOM_AD_OFF_DRV_p13(p13);

// Sensor Relay Connect SW 1 - 6 
DigitalOut pS1_Conn_p7(p7);
DigitalOut pS2_Conn_p8(p8);
DigitalOut pS3_Conn_p9(p9);
DigitalOut pS4_Conn_p10(p10);
DigitalOut pS5_Conn_p11(p11);
DigitalOut pS6_Conn_p12(p12);

// Sensor Error LED 1 - 6
DigitalOut pS1_LED_p30(p30);
DigitalOut pS2_LED_p29(p29);
DigitalOut pS3_LED_p28(p28);
DigitalOut pS4_LED_p27(p27);
DigitalOut pS5_LED_p26(p26);
DigitalOut pS6_LED_p25(p25);

//Sensor 6CH AnalogIn input 1 - 6 (p15,p16,p17,p18,p19,p20 );
AnalogIn    pS1_AD0__p15(p15);
AnalogIn    pS2_AD1__p16(p16);
AnalogIn    pS3_AD2__p17(p17);
AnalogIn    pS4_AD3__p18(p18);
AnalogIn    pS5_AD5__p19(p19);
AnalogIn    pS6_AD6__p20(p20);
//DigitalInOut p10_Power(p10,PIN_OUTPUT,PullNone,0);

/////////////////////////////////
//  Parameter Data //////////////
/////////////////////////////////

// --------------  Capacitor Dose Meter Parameter Declaration  ----------------//
typedef struct  st_CapDose_Action_param{
    int i_CPD_Mode;             // Main Job Mode 
    int i_Meas_Job_Stage;        // Measurement Job Job Stage
    int i_Charge_Job_Stage;     // Charge Job Stage
    int i_DisCharge_Job_Stage;  // DisCharge Job Stage
    
}ST_CAPDOSE_ACTION_PARAMETER;
//    int i_CPD_Mode;         // Main Job Mode 
enum {
    CPD_STAT_Ready=0
    ,CPD_STAT_MEAS
    ,CPD_STAT_CHARGE
    ,CPD_STAT_DISCHARGE
};
//    i_Meas_Job_Stage;         // 測定動作状態遷移 
enum {
    MEAS_STAGE_Ready=0
    ,MEAS_STAGE_AD_Open
    ,MEAS_STAGE_Connect_Start
    ,MEAS_STAGE_Sn_Measure
    ,MEAS_STAGE_Sn_End
};
//   int i_Charge_Job_Stage;     // Charge Job Stage
enum {
    CHARGE_STAGE_Ready=0
    ,CHARGE_STAGE_CHG_START
    ,CHARGE_STAGE_SnMeasure
    ,CHARGE_STAGE_EndChack
    ,CHARGE_STAGE_SnRetry
    ,CHARGE_STAGE_END
};
//    int i_DisCharge_Job_Stage;  // DisCharge Job Stage
enum {
    DISCHARGE_STAGE_Ready=0
    ,DISCHARGE_STAGE_DCH_START
    ,DISCHARGE_STAGE_SnMeasure
    ,DISCHARGE_STAGE_SnEND_Delay
    ,DISCHARGE_STAGE_END
};

// 初期値ｾｯﾄ　st_CapDose_Action_param　
const ST_CAPDOSE_ACTION_PARAMETER const_st_CapDose_Action_param_Default=
{
//     int i_CPD_Mode;             // Main Job Mode 
    CPD_STAT_Ready
//   int i_Meas_Job_Stage;        // Measurement Job Job Stage
    ,MEAS_STAGE_Ready
//    int i_Charge_Job_Stage;     // Charge Job Stage
    ,CHARGE_STAGE_Ready
//    int i_DisCharge_Job_Stage;  // DisCharge Job Stage
    ,DISCHARGE_STAGE_Ready
};


/////////////////////////////// 
//    設定ﾃﾞｰﾀ                //
///////////////////////////////
// --------------  Capacitor Dose Meter Setting Parameter Declaration  ----------------//
typedef struct  st_CapDoseSetting_param{
    // 共通設定項目
    int i_RelaySW_Delay;            // 共通　Relay 切り替え時遅延時間[x1msec]
    //Measure 専用ﾊﾟﾗﾒｰﾀ
    int   i_Meas_Interval_Time;         //  測定値　AD測定間隔
    int   i_Meas_Connect_Time;          //  測定時　Sensor　接続時間
    int   i_Meas_StartDelayTime;        //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
    int   i_Meas_SampleAveTimes;         // 測定開始から　AD測定回数　
    int   i_Meas_SampleAveInterval;      // 　AD測定間隔　
                                        //  （i_Meas_StartDelayTime　＋　（i_Meas_SampleAveTim　x　i_Meas_SampleAveInterval)）
                                        //  が　i_Meas_Connect_Time より小さい
    // Charge/DisCharge 共通
    int   i_CDC_Meas_IntervalTime;      //  充放電時電圧監視　AD測定間隔
     // Charge 専用ﾊﾟﾗﾒｰﾀ
    float   f_ChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
    int     i_ChargeLimitTime;              //  充電時間　ﾘﾐｯﾄ制限
    int     i_ChargeAflerDelay;             // 充電しきい値達成後Wait
    
    // DisCharge 専用   
    float   f_DisChargeTargetVolt;          // 放電時　放電終了　ﾚﾍﾞﾙしきい値[ADC_Level ]
    int     i_DisChargeLimitTime;           //  放電時間　ﾘﾐｯﾄ制限
    int     i_DisChargeAfterDelay;          //　放電時　しきい値達成後　Delay
    
}ST_CAPDOSE_SETTING_PARAMETER;
    // 共通設定項目
    //int i_RelaySW_Delay;                  // 共通　Relay 切り替え時遅延時間[x1msec]
#define Def_RELAY_SW_DELAY_5ms       5       // Relay SW 切り替え時間 初期値　5msec
#define Def_RELAY_SW_DELAY_SetMax_1S  1000       // Relay SW 切り替え時間 最大値

    //Measure 専用ﾊﾟﾗﾒｰﾀ
    //int   i_Meas_Interval_Time;           //  測定値　AD測定間隔
#define Def_MEAS_INTERVAL_100ms        100     // １00msec毎に　再接続　Sample Ave
#define Def_MEAS_INTERVAL_SET_MAX_10min 600000  // 測定間隔　ﾘﾐｯﾄ
    //int   i_Meas_Connect_Time;            //  測定時　SeSample AveSample Ave    
#define Def_MEAS_CONNECT_TIME_50ms    50      //1回　測定での接続時間　　50msec
#define Def_MEAS_CONNECT_TIME_Max1m  60000      //1回　測定での接Max1mi 6000ax 60000sec = 1min

    //int   i_Meas_StartDelayTime;          //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
#define Def_MEAS_START_DELAY_10ms     10      // Sensor 接続からAD測定開始　Delay時間 初期値　１4msec
#define Def_MEAS_START_DELAY_SetMax1sec     1000      // Sensor 接続からAD測定開始　Delay時間 最大1sec

    //int   i_Meas_SampleAveTimes;           // 測定開始から　AD測定回数　（i_Meas_StartDelayTime　＋　i_Meas_SampleAveTimes)
#define Def_MEAS_SAMPLE_AVE_32Times   32      // AD測定回数 初期値　32回 
#define Def_MEAS_SAMPLE_AVE_Times_SetMax_60000   60000     // AD測定回数 /1測定　60000Times 

    //int   i_Meas_SampleAveInterval;       // 　AD測定間隔　
 #define Def_MEAS_SAMPLE_INETRVAL_1ms    1   // AD測定回数 初期値　毎回　1msec
 #define Def_MEAS_SAMPLE_INETRVAL_SetMax_1sec   1000// AD測定間隔 初期値　毎回　1sec
    //int  i_CDC_Meas_IntervalTime;         //  充放電時電圧監視　AD測定間隔
#define Def_CDC_MEAS_INTERVAL_5ms  5             // しきい値監視 初期値　5msec
#define Def_CDC_MEAS_INTERVAL_SetMax  1000       // しきい値監視 最大　1sec

     // Charge 専用ﾊﾟﾗﾒｰﾀ
    //float f_ChargeTargetVolt;             // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
#define Def_CHARGE_TARGETVolt3V  3.0f         // Charge [V] Power Charge Target Limit Volt
#define Def_CHARGE_TARGET_VoltMIN    1.0f
#define Def_CHARGE_TARGET_VoltMAX    3.3f

    // int i_ChargeLimitTime;               //  充電時間　ﾘﾐｯﾄ制限
#define Def_CHARGE_LIMIT_TIME_10sec 10000         // 10000msec = 10sec 
#define Def_CHARGE_LIMIT_TIME_SetMax_1min 60000         // 60000msec = 1minutes 

    //int i_ChargeAflerDelay;               // 充電しきい値達成後Wait
#define Def_CHARGE_AFTER_DELAY5sec      5000    // 5000msec = 5sec 
#define Def_CHARGE_AFTER_DELAY_SetMax_1min    60000         // 60000msec = 1minutes 
    // DisCharge 専用   
   //float f_DisChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
#define Def_DisCHARGE_TARGETVolt05V  0.5f      // DisCharge [V] Power Charge Target Limit Volt
#define Def_DisCHARGE_TARGET_VoltMIN    0.0f
#define Def_DisCHARGE_TARGET_VoltMAX    3.0f
    //int i_DisChargeLimitTime;             //  放電時間　ﾘﾐｯﾄ制限
#define Def_DISCHARGE_LIMIT_TIME10sec 10000      // 10000msec = 10sec 
#define Def_DISCHARGE_LIMIT_TIMEMax_1min   60000         // 60000msec = 1minutes 
    //int i_DisChargeAfterDelay;            //　放電時　しきい値達成後　Delay
#define Def_DISCHARGE_AFTER_DELAY5sec 5000      // 5000msec = 5sec 
#define Def_DISCHARGE_AFTER_DELAY_SetMax_1min 60000         // 60000msec = 1minutes  


///////////////////////////////
//    測定及び状態監視ﾃﾞｰﾀ      //
///////////////////////////////
//DigitalOut pCOM_Vref_Conn_p5(p5) ,pCOM_GND_Conn_p6(p6) ,pCOM_AD_OFF_DRV_p13(p13);
// Sensor Relay Connect SW 1 - 6 
//DigitalOut pS1_Conn_p7(p7) ,pS2_Conn_p8(p8), pS3_Conn_p9(p9), pS4_Conn_p10(p10), pS5_Conn_p11(p11), pS6_Conn_p12(p12);
// Sensor Error LED 1 - 6
//DigitalOut pS1_LED_p30(p30), pS2_LED_p29(p29), pS3_LED_p28(p28), pS4_LED_p27(p27), pS5_LED_p26(p26), pS6_LED_p25(p25);
//---------------  I/O Port Setting Now -----------------------//
#define nSENSOR_6CH     6
#define Sel_S1  0
#define Sel_S2  1
#define Sel_S3  2
#define Sel_S4  3
#define Sel_S5  4
#define Sel_S6  5
#define No_S1  1
#define No_S2  2
#define No_S3  3
#define No_S4  4
#define No_S5  5
#define No_S6  6
typedef struct st_PortStat{
    int i_pCOM_Vref_Conn_p5;
    int i_pCOM_GND_Conn_p6;
    int i_pCOM_AD_OFF_DRV_p13;
    int i_pSn_Conn[nSENSOR_6CH];
    int i_pSn_LED[nSENSOR_6CH];
}ST_PORT_STATUS;
    // Relay SW ON/OFF
    #define RelayON     1
    #define RelayOFF    0
    #define LED_ON      0
    #define LED_OFF     1
    #define Def_PortSET_Stat_MAX 1

// --------------  Capacitor Dose Meter Measurement Parameter  ----------------//

typedef struct  st_MeasStat{
    int i_Sn_ADMeas[nSENSOR_6CH];
    int i_Sn_ERR[nSENSOR_6CH];
    int i_Sn_MeasCount[nSENSOR_6CH];
    
}ST_MEAS_STAT;

// Work Space Declaration
ST_CAPDOSE_ACTION_PARAMETER st_CapDoseMain;
ST_CAPDOSE_SETTING_PARAMETER st_CapDoseSetting;
ST_MEAS_STAT st_Meas;
ST_PORT_STATUS st_IOPort;


//////////////////////////////////////////
//　　Timer 割り込み　定義
//////////////////////////////////////////
// Timer Interrupt //
///////////////
Ticker Int1mSec_Timer;
Timer t;

//////////////////////////////////////////////////////////
//------------------------------------//
// --- USB　Serial Communication ---  //
//------------------------------------//
void Tx_interrupt();
void Rx_interrupt();
void send_line();
int read_line();  // Return Rec CHAR Count 2013.08.08 Tsunemoto Append
void Tick_1mSec_Interrupt();

// Circular buffers for serial TX and RX data - used by interrupt routines
const int ser_buffer_size = 255;
// might need to increase buffer size for high baud rates
char tx_buffer[ser_buffer_size];
char rx_buffer[ser_buffer_size];
// Circular buffer pointers
// volatile makes read-modify-write atomic
volatile int tx_in=0;
volatile int tx_out=0;
volatile int rx_in=0;
volatile int rx_out=0;
// Line buffers for sprintf and sscanf
char tx_line[80];
char rx_line[80];
//---  2013.08.08 Tsunemoto ------//
//--  rx Data Cr Rec Counter
volatile int rx_cr_Rec = 0;
//-----------------------------------------------------------//
//--------- Timer Innterrupt For DAC Control  ---------------//
//-----------------------------------------------------------//
int timer_count=0;
int timer_1Sec=0;
int main_loop_count=0;
//////////////////////////////
////  Measurement Mode   /////
//////////////////////////////
#define SAMPLE_MODE_NotActive   0
#define SAMPLE_MODE_ACTIVE      1
int i_SampleMode=0;
#define CHARGE_MODE_NotActive   0
#define CHARGE_MODE_AfterDelay  1
#define CHARGE_MODE_ACTIVE      2
int i_ChargeMode=0;

#define DISCHARGE_MODE_NotActive   0
#define DISCHARGE_MODE_ACTIVE      1
int i_DisChargeMode=0;

// 100usec Decrement Counter
int i_ADC_IntCount =0;
int i_ADC_Sample_Count=0;
int i_ChargeAfterDelayCount =0;
int i_Charge_TermOverCount=0;
int i_DisChargeTermCount =0;


//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//////////////////////////////////////////////////////////////////////////////////
void CapDose_param_init();

float ADC_Sample_Send();
void Ser_Command_Input();
void com_Table_Param_Send();
void com_Stat_Table_Param_Send();
// USB Serial Command Job
bool com_Check_SW_PCV_COMVref(int i_RecCharCount);
bool com_Check_SW_PCG_COMGND(int i_RecCharCount);
bool com_Check_SW_PCA_COMADG(int i_RecCharCount);
bool com_Check_SW_Psn_SensConn(int i_RecCharCount);
bool com_Check_SW_PLn_LEDset(int i_RecCharCount);
bool com_Check_RDL_ConnDelayAfterRSW(int i_RecCharCount);
bool com_Check_MIT_MeasurementInterval(int i_RecCharCount);
bool com_Check_MCT_MeasurementConnectTime(int i_RecCharCount);
bool com_Check_MSD_MeasSampleStartDelay(int i_RecCharCount);
bool com_Check_MAT_MeasAveTimes(int i_RecCharCount);
bool com_Check_MAI_MeasSampleAveInterval(int i_RecCharCount);
bool com_Check_CDI_Charge_DisCharge_SampleInterval(int i_RecCharCount);
bool com_Check_CTV_ChargeTargetVoltSet(int i_RecCharCount);
bool com_Check_CLT_ChargeLimitTimeSet(int i_RecCharCount);
bool com_Check_CAD_ChargeAfterDelaySet(int i_RecCharCount);
bool com_Check_DTV_DisChargeTargetVoltSet(int i_RecCharCount);
bool com_Check_DLT_DisChargeLimitTimeSet(int i_RecCharCount);
bool com_Check_DAD_DisChargeAfterDelaySet(int i_RecCharCount);

//----------------------------------------------------------------//
//////////////////////////////////////////////////
//  Relay SW Control Initial Set                //
//      Common Relay Connect SW                 //
//      DigitalOut pCOM_Vref_Conn_p5(p5);       //
//      DigitalOut pCOM_GND_Conn_p6(p6);        //
//      DigitalOut pCOM_AD_OFF_DRV_p13(p13);    //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//DigitalOut pCOM_Vref_Conn_p5(p5) ,pCOM_GND_Conn_p6(p6) ,pCOM_AD_OFF_DRV_p13(p13);                                         //
// Sensor Relay Connect SW 1 - 6                                                                                            //
//DigitalOut pS1_Conn_p7(p7) ,pS2_Conn_p8(p8), pS3_Conn_p9(p9), pS4_Conn_p10(p10), pS5_Conn_p11(p11), pS6_Conn_p12(p12);    //
// Sensor Error LED 1 - 6                                                                                                   //
//DigitalOut pS1_LED_p30(p30), pS2_LED_p29(p29), pS3_LED_p28(p28), pS4_LED_p27(p27), pS5_LED_p26(p26), pS6_LED_p25(p25);    //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  ---------------  I/O Port Setting Now -----------------------  ///
//typedef struct st_PortStat{                                       //
//    int i_pCOM_Vref_Conn_p5;                                      //
//    int i_pCOM_GND_Conn_p6;                                       //
//    int i_pCOM_AD_OFF_DRV_p13;                                    //
//    int i_pSn_Conn[nSENSOR_6CH];                                  //
//    int i_pSn_LED[nSENSOR_6CH];                                   //
//////////////////////////////////////////////////////////////////////
void set_Initial_IOPort()  
{
    pCOM_Vref_Conn_p5 = RelayOFF;       st_IOPort.i_pCOM_Vref_Conn_p5 = RelayOFF;
    pCOM_GND_Conn_p6 = RelayOFF;        st_IOPort.i_pCOM_GND_Conn_p6 = RelayOFF;
     pS1_Conn_p7 = RelayOFF;             st_IOPort.i_pSn_Conn[Sel_S1] = RelayOFF;
     pS2_Conn_p8 = RelayOFF;             st_IOPort.i_pSn_Conn[Sel_S2] = RelayOFF;
     pS3_Conn_p9 = RelayOFF;             st_IOPort.i_pSn_Conn[Sel_S3] = RelayOFF;
     pS4_Conn_p10 = RelayOFF;            st_IOPort.i_pSn_Conn[Sel_S4] = RelayOFF;
     pS5_Conn_p11 = RelayOFF;            st_IOPort.i_pSn_Conn[Sel_S5] = RelayOFF;
     pS6_Conn_p12 = RelayOFF;            st_IOPort.i_pSn_Conn[Sel_S6] = RelayOFF;
    pCOM_AD_OFF_DRV_p13 = RelayON; st_IOPort.i_pCOM_AD_OFF_DRV_p13 = RelayON;
     pS1_LED_p30 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S1] = LED_OFF;
     pS2_LED_p29 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S2] = LED_OFF;
     pS3_LED_p28 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S3] = LED_OFF;
     pS4_LED_p27 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S4] = LED_OFF;
     pS5_LED_p26 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S5] = LED_OFF;
     pS6_LED_p25 = LED_OFF;             st_IOPort.i_pSn_LED[Sel_S6] = LED_OFF;
   
}
//////////////////////////////////////////////////////////////////////
//typedef struct st_PortStat{                                       //
//    int i_pCOM_Vref_Conn_p5;                                      //
//    int i_pCOM_GND_Conn_p6;                                       //
//    int i_pCOM_AD_OFF_DRV_p13;                                    //
//////////////////////////////////////////////////////////////////////
void set_RelaySW_COM(int i_sw_Vref,int i_sw_GND,int i_sw_ADOFF)
{
    st_IOPort.i_pCOM_Vref_Conn_p5 = i_sw_Vref;
    pCOM_Vref_Conn_p5 = i_sw_Vref;
    st_IOPort.i_pCOM_GND_Conn_p6 = i_sw_GND;
    pCOM_GND_Conn_p6 = i_sw_GND;
    st_IOPort.i_pCOM_AD_OFF_DRV_p13 = i_sw_ADOFF;
    pCOM_AD_OFF_DRV_p13 = i_sw_ADOFF;
}
//////////////////////////////////////////////////////////////////////
//          ---   Sensor n Relay Control  ---                       //
//typedef struct st_PortStat{                                       //
//    int i_pCOM_Vref_Conn_p5;                                      //
//    int i_pCOM_GND_Conn_p6;                                       //
//    int i_pCOM_AD_OFF_DRV_p13;                                    //
//>>    int i_pSn_Conn[nSENSOR_6CH];                                //
//    int i_pSn_LED[nSENSOR_6CH];                                   //
/////////////////////////////////////////////////////////////////////
void set_RelaySW_SensConn(int i_Sn,int i_sw)
{
    switch(i_Sn){
        case No_S1:
            st_IOPort.i_pSn_Conn[Sel_S1] = i_sw;
            pS1_Conn_p7 = i_sw;
            break;
        case No_S2:
            st_IOPort.i_pSn_Conn[Sel_S2] = i_sw;
            pS2_Conn_p8 = i_sw;
           break;
        case No_S3:
            st_IOPort.i_pSn_Conn[Sel_S3] = i_sw;
            pS3_Conn_p9 = i_sw;
           break;
        case No_S4:
            st_IOPort.i_pSn_Conn[Sel_S4] = i_sw;
            pS4_Conn_p10 = i_sw;
           break;
        case No_S5:
            st_IOPort.i_pSn_Conn[Sel_S5] = i_sw;
            pS5_Conn_p11 = i_sw;
            break;
        case No_S6:
            st_IOPort.i_pSn_Conn[Sel_S6] = i_sw;
            pS6_Conn_p12 = i_sw;
            break;
        default:
            break;    
    }
}
//////////////////////////////////////////////////////////////////////
//          ---   LED n ON/OFF Control  ---                       //
//typedef struct st_PortStat{                                       //
//    int i_pCOM_Vref_Conn_p5;                                      //
//    int i_pCOM_GND_Conn_p6;                                       //
//    int i_pCOM_AD_OFF_DRV_p13;                                    //
//    int i_pSn_Conn[nSENSOR_6CH];                                  //
//>>    int i_pSn_LED[nSENSOR_6CH];                                   //
/////////////////////////////////////////////////////////////////////
void set_LEDnSW_port(int i_Sn,int i_sw)
{
    switch(i_Sn){
        case No_S1:
            st_IOPort.i_pSn_LED[Sel_S1] = i_sw;
            pS1_LED_p30 = i_sw;
            break;
        case No_S2:
            st_IOPort.i_pSn_LED[Sel_S2] = i_sw;
            pS2_LED_p29 = i_sw;
           break;
        case No_S3:
             st_IOPort.i_pSn_LED[Sel_S3] = i_sw;
           pS3_LED_p28 = i_sw;
           break;
        case No_S4:
             st_IOPort.i_pSn_LED[Sel_S4] = i_sw;
           pS4_LED_p27 = i_sw;
           break;
        case No_S5:
            st_IOPort.i_pSn_LED[Sel_S5] = i_sw;
            pS5_LED_p26 = i_sw;
            break;
        case No_S6:
            st_IOPort.i_pSn_LED[Sel_S6] = i_sw;
            pS6_LED_p25 = i_sw;
            break;
        default:
            break;    
    }
}

///////////////////////////////////////////////////////////
//--------------Parameter Initial Set ------------------//
void CapDose_param_init()
{
    st_CapDoseMain.i_CPD_Mode = const_st_CapDose_Action_param_Default.i_CPD_Mode;
    st_CapDoseMain.i_Meas_Job_Stage = const_st_CapDose_Action_param_Default.i_Meas_Job_Stage;
    st_CapDoseMain.i_Charge_Job_Stage = const_st_CapDose_Action_param_Default.i_Charge_Job_Stage;
    st_CapDoseMain.i_DisCharge_Job_Stage = const_st_CapDose_Action_param_Default.i_DisCharge_Job_Stage;
    // 共通設定項目
    //int i_RelaySW_Delay;                  // 共通　Relay 切り替え時遅延時間[x1msec]
    st_CapDoseSetting.i_RelaySW_Delay = Def_RELAY_SW_DELAY_5ms;   // Relay SW 切り替え時間 初期値　5msec
    //Measure 専用ﾊﾟﾗﾒｰﾀ
    //int   i_Meas_Interval_Time;           //  測定値　AD測定間隔
    st_CapDoseSetting.i_Meas_Interval_Time = Def_MEAS_INTERVAL_100ms; 
    //int   i_Meas_Connect_Time;            //  測定時　SeSample AveSample Ave  
    st_CapDoseSetting.i_Meas_Connect_Time = Def_MEAS_CONNECT_TIME_50ms;
    //int   i_Meas_StartDelayTime;          //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
    st_CapDoseSetting.i_Meas_StartDelayTime = Def_MEAS_START_DELAY_10ms;
    //int   i_Meas_SampleAveTimes;           // 測定開始から　AD測定回数　（i_Meas_StartDelayTime　＋　i_Meas_SampleAveTimes)
    st_CapDoseSetting.i_Meas_SampleAveTimes = Def_MEAS_SAMPLE_AVE_32Times;
    //int   i_Meas_SampleAveInterval;       // 　AD測定間隔　
    st_CapDoseSetting.i_Meas_SampleAveInterval = Def_MEAS_SAMPLE_INETRVAL_1ms;
    //int  i_CDC_Meas_IntervalTime;         //  充放電時電圧監視　AD測定間隔
    st_CapDoseSetting.i_CDC_Meas_IntervalTime = Def_CDC_MEAS_INTERVAL_5ms;
     // Charge 専用ﾊﾟﾗﾒｰﾀ
    //float f_ChargeTargetVolt;             // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
    st_CapDoseSetting.f_ChargeTargetVolt = Def_CHARGE_TARGETVolt3V;
    // int i_ChargeLimitTime;               //  充電時間　ﾘﾐｯﾄ制限
    st_CapDoseSetting.i_ChargeLimitTime = Def_CHARGE_LIMIT_TIME_10sec;
    //int i_ChargeAflerDelay;               // 充電しきい値達成後Wait
    st_CapDoseSetting.i_ChargeAflerDelay = Def_CHARGE_AFTER_DELAY5sec;
    // DisCharge 専用   
    //float f_DisChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
    st_CapDoseSetting.f_DisChargeTargetVolt = Def_DisCHARGE_TARGETVolt05V;
    //int i_DisChargeLimitTime;             //  放電時間　ﾘﾐｯﾄ制限
    st_CapDoseSetting.i_DisChargeLimitTime = Def_DISCHARGE_LIMIT_TIME10sec;
    //int i_DisChargeAfterDelay;            //　放電時　しきい値達成後　Delay
    st_CapDoseSetting.i_DisChargeAfterDelay = Def_DISCHARGE_AFTER_DELAY5sec;

 }
//    i_Meas_Job_Stage;         // 測定動作状態遷移 
//enum {
//    MEAS_STAGE_Ready=0
//    ,MEAS_STAGE_AD_Open
//    ,MEAS_STAGE_Connect_Start
//    ,MEAS_STAGE_Sn_Measure
//    ,MEAS_STAGE_Sn_End
//};
////////////////////////////////////////////
//     --  Main Action Sequence  ---      //
//           <<<  Measure  >>>            //
////////////////////////////////////////////
void MainAction_Measurement(void)
{
    
}
//-------------------------------------------------------//
//----------   "SM" ﾚﾍﾞﾙ測定スタート      ---------------------//
//  //  
//int i_ADC_IntCount =0;
//int i_ADC_Sample_Count=0;
//-------------------------------------------------------//
void Start_SampleMeasure()
{
    st_CapDoseMain.i_CPD_Mode = CPD_STAT_MEAS;         // Main Action Measurement Stage Set
    sprintf(tx_line,"SM:START\r\n" );
    send_line();

    AnalogIn p20_Adc(p20);
   i_SampleMode = SAMPLE_MODE_ACTIVE;
 //   i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
//    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
}
void Start_MeasureSingle()
{
}
void Start_MeasureNtimes()
{
}
void Start_MeasureContinuous()
{
}
////////////////////////////////////////////
//     --  Main Action Sequence  ---      //
//           <<<  Charge  >>>             //
////////////////////////////////////////////
void MainAction_Charge(void)
{
    
}
////////////////////////////////////////////
//     --  Main Action Sequence  ---      //
//           <<<  DisCharge  >>>          //
////////////////////////////////////////////
void MainAction_DisCharge(void)
{
    
}
    
////////////////////////////////////////////
//////////////////////////////////////////////////////////
//     Main                                             //
//////////////////////////////////////////////////////////
int main(){
    float f_ADC_inp =0.0;
        
    set_Initial_IOPort();
    device.baud(115200);
    timer_count = 0;
    CapDose_param_init();
    Int1mSec_Timer.attach_us(&Tick_1mSec_Interrupt, (1000));       // 1msec Timer 割り込み
    t.reset();
    t.start();

    // Setup a serial interrupt function to receive data
    device.attach(&Rx_interrupt, Serial::RxIrq);
    // Setup a serial interrupt function to transmit data
    device.attach(&Tx_interrupt, Serial::TxIrq);
    timer_count = 0;

    //--- ADC Measurement Control Parameter Initial Set ---//
    //    adc_param_init(); 
    //--- DAC Control Parameter Init --- //
    //   dac1_param_init();
    // -- Main Loop -- //
    while (1) {
        if(rx_cr_Rec > 0){
            Ser_Command_Input();
        }
        // Sample Measurement Active
        if(i_SampleMode == SAMPLE_MODE_ACTIVE){
             myled4 = 1.0;
            if(i_ADC_IntCount == 0){
 //               i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
                f_ADC_inp =ADC_Sample_Send();
                i_ADC_Sample_Count++;
//               if(i_ADC_Sample_Count >= st_dosimeterDock_param.i_SampleTimes_MAX){
//                    i_SampleMode = SAMPLE_MODE_NotActive;           // Sample End
//                   myled4 = 0;
//                    sprintf(tx_line,"SM:END\r\n" );
//                    send_line();
//                }
            }

        }
        // 動作　モード　ｼｰｹﾝｽ
        //enum {
        //CPD_STAT_Ready=0
        //,CPD_STAT_MEAS
        //,CPD_STAT_CHARGE
        //,CPD_STAT_DISCHARGE
        //};

        switch(st_CapDoseMain.i_CPD_Mode){
            case CPD_STAT_Ready:
                break;
            case CPD_STAT_MEAS:
                MainAction_Measurement();
                break;
            case CPD_STAT_CHARGE:
                MainAction_Charge();
                break;
            case CPD_STAT_DISCHARGE:
                MainAction_DisCharge();
                break;
            default:
                break;
        }
            
        
       // MainLoop LED Blink
        main_loop_count++;
        if(main_loop_count>=100000){
            myled1 = (myled1+1) & 1;
            main_loop_count = 0;
        }
            /////////////////////////////////
    }
}

//-------------------------------------------------------//
//----------    Chargeスタート      ---------------------//
//-------------------------------------------------------//
void Start_ChargeMode()
{
    sprintf(tx_line,"SC:START\r\n" );
   send_line();
 //   AnalogIn p20_Adc(p20);
 //  i_ChargeMode = CHARGE_MODE_ACTIVE;
 //   i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
//    i_Charge_TermOverCount = (int)( (st_dosimeterDock_param.f_ChargeOver_Time) * 10000.0f);
//    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
 //   P10_PWR_Pin_1_High();
 //      sprintf(tx_line,"TIMECOUNT %4d\r\n",i_Charge_TermOverCount );
 //  send_line();
 
}
//-------------------------------------------------------//
//----------    DisChargeスタート      ---------------------//
//-------------------------------------------------------//
void Start_DisChargeMode()
{
    sprintf(tx_line,"SD:START\r\n" );
    send_line();
    i_DisChargeMode = DISCHARGE_MODE_ACTIVE;
//    i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
 //   i_DisChargeTermCount=(int)( (st_dosimeterDock_param.f_DisCharge_Time) * 10000.0f);
//    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
//    P10_PWR_Pin_0_GND();
}

////////////////////////////////////////////////////////////////////////////////////////////////
// 動作　ﾊﾟﾗﾒｰﾀ　 初期設定
///
//st_CapDoseMain
// --------------  Capacitor Dose Meter Parameter Declaration  ----------------//
//typedef struct  st_CapDose_Action_param{
//    int i_CPD_Mode;             // Main Job Mode 
//    int i_Meas_Job_Stage;        // Measurement Job Job Stage
//    int i_Charge_Job_Stage;     // Charge Job Stage
//    int i_DisCharge_Job_Stage;  // DisCharge Job Stage
//    
//}ST_CAPDOSE_ACTION_PARAMETER;
// 初期値ｾｯﾄ　st_CapDose_Action_param　
//const ST_CAPDOSE_ACTION_PARAMETER const_st_CapDose_Action_param_Default=
//{
////     int i_CPD_Mode;             // Main Job Mode 
//   CPD_STAT_Ready
// //   int i_Meas_Job_Stage;        // Measurement Job Job Stage
//   ,MEAS_STAGE_Ready
////    int i_Charge_Job_Stage;     // Charge Job Stage
//   ,CHARGE_STAGE_Ready
////    int i_DisCharge_Job_Stage;  // DisCharge Job Stage
//    ,DISCHARGE_STAGE_Ready
//};

////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////
/////      ADC 1 Sample Input & Send        //
//////////////////////////////////////////////
float ADC_Sample_Send()
{
 //   int i;
    float f_ADC_V =0.0;
    
    AnalogIn p20_Adc(p20);
    f_ADC_V  = (float)p20_Adc*3.3;
    sprintf(tx_line,"ADC,%3i,%2.3f\r\n",i_ADC_Sample_Count, f_ADC_V  );
    send_line();
    return(f_ADC_V);
}

////////////////////////////
//  P10 Power Port Set    //
////////////////////////////
void P10_PWR_Pin_0_GND()
{
 }
void P10_PWR_Pin_1_High()
{
}
void P10_PWR_Pin_2_Open()
{
 
}

void P10_PWR_Pin_Setting(int i_selMode)
{
    switch(i_selMode){
        case 0:
            P10_PWR_Pin_0_GND();
        break;
        case 1:
            P10_PWR_Pin_1_High();
        break;
        case 2:
            P10_PWR_Pin_2_Open();
       break;
        default:
        break;
    }
        
}
////////////////////////////
//  P20 ADC Port Set    //
////////////////////////////
void P20_ADC_Pin_0_Open()
{
    DigitalInOut p20_GPIO_Open(p20,PIN_OUTPUT,OpenDrain,1);
}
void P20_ADC_Pin_1_ADCin()
{
    AnalogIn    p20_Adc(p20);

}
void P20_ADC_Pin_2_AD_Open()
{
    DigitalInOut p20_GPIO_Open(p20,PIN_OUTPUT,OpenDrain,1);

}
void P20_ADC_Pin_3_ADPin_GNDTest()
{
    DigitalInOut p20_GPIO_GND(p20,PIN_OUTPUT,PullNone,0);

}
void P20_ADC_Pin_4_ADPin_HighTest()
{
    DigitalInOut p20_GPIO_High(p20,PIN_OUTPUT,PullNone,1);

}
           
void P20_ADC_Pin_Setting(int i_selMode)
{
    switch(i_selMode){
        case 0:
            P20_ADC_Pin_0_Open();
            break;
        case 1:
            P20_ADC_Pin_1_ADCin();
            break;
        case 2:
            P20_ADC_Pin_2_AD_Open();
            break;
        case 3:
            P20_ADC_Pin_3_ADPin_GNDTest();
            break;
         case 4:
            P20_ADC_Pin_4_ADPin_HighTest();
            break;
       default:
            break;
    }
        
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------//
//------------------------------------------------------------------------------//
//-----    Serial tx/rx Communication  
//------------------------------------------------------------------------------//
// Copy tx line buffer to large tx buffer for tx interrupt routine
void send_line() {
    int i;
    char temp_char;
    bool empty;
    i = 0;
// Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(UART_IRQn);
    empty = (tx_in == tx_out);
    while ((i==0) || (tx_line[i-1] != '\n')) {
// Wait if buffer full
        if (((tx_in + 1) % ser_buffer_size) == tx_out) {
// End Critical Section - need to let interrupt routine empty buffer by sending
            NVIC_EnableIRQ(UART_IRQn);
            while (((tx_in + 1) % ser_buffer_size) == tx_out) {
            }
// Start Critical Section - don't interrupt while changing global buffer variables
            NVIC_DisableIRQ(UART_IRQn);
        }
        tx_buffer[tx_in] = tx_line[i];
        i++;
        tx_in = (tx_in + 1) % ser_buffer_size;
    }
    if (device.writeable() && (empty)) {
        temp_char = tx_buffer[tx_out];
        tx_out = (tx_out + 1) % ser_buffer_size;
// Send first character to start tx interrupts, if stopped
        device.putc(temp_char);
    }
// End Critical Section
    NVIC_EnableIRQ(UART_IRQn);
    return;
}


// Read a line from the large rx buffer from rx interrupt routine
// 2013.08.08 H.Tsunemoto 
// Append Return Chear Number
int read_line(){
//void read_line() {
    int i;
    i = 0;
    // Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(UART_IRQn);
    while((rx_in != rx_out) && (rx_buffer[rx_out]<0x20)){
          rx_out = (rx_out + 1) % ser_buffer_size;
     }
     while(rx_in != rx_out){
        rx_line[i] = rx_buffer[rx_out];
         rx_out = (rx_out + 1) % ser_buffer_size;
        if((rx_line[i] == '\r') || (rx_line[i] == '\n')){
            break;
       }
             i++;

    }
    rx_line[i] = 0;
// End Critical Section
    NVIC_EnableIRQ(UART_IRQn);
    return(i);
}
// Interupt Routine to read in data from serial port
void Rx_interrupt() {
    //led1=1;
// Loop just in case more than one character is in UART's receive FIFO buffer
// Stop if buffer full
    while ((device.readable()) || (((rx_in + 1) % ser_buffer_size) == rx_out)) {
        rx_buffer[rx_in] = device.getc();
// Uncomment to Echo to USB serial to watch data flow
//        monitor_device.putc(rx_buffer[rx_in]);
        //------- 2013.08.08 Tsunemoto ------------//
        // -- Char CR Rec Counter ----//
        if(rx_buffer[rx_in]== '\r'){
           //led2 = 1;
           rx_cr_Rec ++;
        }
        //----------------------------//
        rx_in = (rx_in + 1) % ser_buffer_size;
    }
    //led1=0;
    return;
}

// Interupt Routine to write out data to serial port
void Tx_interrupt() {
    //led2=1;
// Loop to fill more than one character in UART's transmit FIFO buffer
// Stop if buffer empty
    while ((device.writeable()) && (tx_in != tx_out)) {
        device.putc(tx_buffer[tx_out]);
        tx_out = (tx_out + 1) % ser_buffer_size;
    }
    //led2=0;
    return;
}

////////////////////////////////////////////////
//   1msec Timer Interrupt
//
// 1msec Decrement Counter
//int i_ADC_IntCount =0;
//int i_ADC_Sample_Count=0;
//int i_Charge_Term_Count=0;
//int i_ChargeAfterDelayCount =0;
//int i_Charge_TermOverCount=0;
//int i_DisChargeTermCount =0;

///////////////////////////////////////////////
void Tick_1mSec_Interrupt() 
{
    P36_Int1msec=1;
    // Sampling Interval 
    if(i_ADC_IntCount >0){
        i_ADC_IntCount--;
    }
    // i_Charge_Term_Count 
    if(i_ChargeAfterDelayCount >0){
        i_ChargeAfterDelayCount--;
    }
    if(i_Charge_TermOverCount >0){
        i_Charge_TermOverCount--;
    }

    // Discharge Term Counter
    if(i_DisChargeTermCount >0){
        i_DisChargeTermCount--;
    }
    
    timer_count++;                  //increment timer_count
    if(timer_count >= 10000){
        timer_count = 0;
        timer_1Sec++;
    }
    P36_Int1msec=0;

}

//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//  Input :i_RecCharCount :Command Stringth Length                              //
//         rx_line[80] :(Global) Rec Data Stringth                              //
//  Return :bool b_CommadERR  0= ACK                                            //
//                            1= ERR                                            //
//////////////////////////////////////////////////////////////////////////////////

//-------- DosimeterDock Parameter declaration  --------//
/////////////////////////////// 
//    設定ﾊﾟﾗﾒｰﾀ応答               //
///////////////////////////////
// --------------  Capacitor Dose Meter I・O Port ON/OFF Test  ----------------//
 //  MBED Port (Relay Control Pin Replay Status & Set I/O Port
 //     bool com_Check_SW_RCV_COMVref(int i_RecCharCount)
 //     bool com_Check_SW_RCG_COMGND(int i_RecCharCount)
 //     bool com_Check_SW_RCA_COMADG(int i_RecCharCount)
 //     bool com_Check_SW_RSn_SnConn(int i_RecCharCount)n=1-6
 // Setting Parameter Replay & Setting
 //     bool com_Check_SW_RCV_COMVref(int i_RecCharCount)
//------------------------------------------------------------------------------//
//  Command No.1  "Relay SW ON/OFF "  P5 COM_Vref3.2V Pin Status  Set             //
//typedef struct st_PortStat{
//>>    int i_pCOM_Vref_Conn_p5;
//    int i_pCOM_GND_Conn_p6;
//    int i_pCOM_AD_OFF_DRV_p13;
//    int i_pSn_Conn[nSENSOR_6CH];
//    int i_pSn_LED[nSENSOR_6CH];
//}ST_PORT_STATUS;
//
//int st_IOPort.i_pCOM_Vref_Conn_p5 = 1;  // P5 Vref3.2V Connect
//------------------------------------------------------------------------------//
bool com_Check_SW_PCV_COMVref(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    }
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'N') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'N') )){
        set_RelaySW_COM(RelayON,st_IOPort.i_pCOM_GND_Conn_p6,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
    }
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'F') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'F') )){
        set_RelaySW_COM(RelayOFF,st_IOPort.i_pCOM_GND_Conn_p6,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
    }
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if((i_num >= 0 ) && (i_num <= Def_PortSET_Stat_MAX)){
                 set_RelaySW_COM(i_num,st_IOPort.i_pCOM_GND_Conn_p6,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
 //------------------------------------------------------------------------------//
//  Command No.2  "Relay SW ON/OFF "  P6 COM GND  Pin Status  Set             //
//typedef struct st_PortStat{
//>>    int i_pCOM_GND_Conn_p6;
//}ST_PORT_STATUS;
//------------------------------------------------------------------------------//
bool com_Check_SW_PCG_COMGND(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'N') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'N') )){
        set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,RelayON,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
    }
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'F') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'F') )){
        set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,RelayOFF,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
    }
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if((i_num >= 0 ) && (i_num <= Def_PortSET_Stat_MAX)){
                 set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,i_num,st_IOPort.i_pCOM_AD_OFF_DRV_p13);
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.3  "Relay SW ON/OFF "  P13 ADC Inp Short GND  Pin Status  Set    //
//typedef struct st_PortStat{
//>>    int i_pCOM_AD_OFF_DRV_p13;
//}ST_PORT_STATUS;
///------------------------------------------------------------------------------//
bool com_Check_SW_PCA_COMADG(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'N') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'N') )){
        set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,st_IOPort.i_pCOM_GND_Conn_p6,RelayON);
    }
    else if( ((rx_line[3] == 'O') && (rx_line[4] == 'F') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'F') )){
        set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,st_IOPort.i_pCOM_GND_Conn_p6,RelayOFF);
    }
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if((i_num >= 0 ) && (i_num <= Def_PortSET_Stat_MAX)){
                 set_RelaySW_COM(st_IOPort.i_pCOM_Vref_Conn_p5,st_IOPort.i_pCOM_GND_Conn_p6,i_num);
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.4  "Relay SW ON/OFF "  Sensor 1 - 6 Relay  Pin Status  Set    //
//typedef struct st_PortStat{
//>>    int i_pSn_Conn[nSENSOR_6CH];
//}ST_PORT_STATUS;
//------------------------------------------------------------------------------//
bool com_Check_SW_Psn_SensConn(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_SensorNo=0;
    int i_num=0;
    char *pt_comRec;
    #define SENSOR_S0   0x30 
        i_SensorNo = (int)(rx_line[2]);
        
    if((i_SensorNo >= 0x31) && (i_SensorNo <= 0x36)){
        if(i_RecCharCount < 3){
         b_CommadERR = 1;
        } 
        else if( ((rx_line[3] == 'O') && (rx_line[4] == 'N') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'N') )){
            set_RelaySW_SensConn((i_SensorNo - SENSOR_S0) ,RelayON);
        }
        else if( ((rx_line[3] == 'O') && (rx_line[4] == 'F') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'F') )){
            set_RelaySW_SensConn((i_SensorNo - SENSOR_S0) ,RelayOFF);
        }
        else{
            pt_comRec = (char *)&rx_line[3];
            i_num= atoi(pt_comRec);
            if((i_num >= 0 ) && (i_num <= Def_PortSET_Stat_MAX)){
                 set_RelaySW_SensConn((i_SensorNo - SENSOR_S0) ,i_num);
            }
            else{
                b_CommadERR = 1;
            }
        }
    }
    else{
        b_CommadERR = 1;
    }
    return(b_CommadERR);
 }
//set_LEDnSW_port
//------------------------------------------------------------------------------//
//  Command No.5  "Sensor LED  ON/OFF "  LED 1 - 6   Pin Status  Set    //
//typedef struct st_PortStat{
///>>    int i_pSn_LED[nSENSOR_6CH];
//}ST_PORT_STATUS;
//------------------------------------------------------------------------------//
bool com_Check_SW_PLn_LEDset(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_SensorNo=0;
    int i_num=0;
    char *pt_comRec;
    #define SENSOR_S0   0x30 
        i_SensorNo = (int)(rx_line[2]);
        
    if((i_SensorNo >= 0x31) && (i_SensorNo <= 0x36)){
        if(i_RecCharCount < 3){
         b_CommadERR = 1;
        } 
        else if( ((rx_line[3] == 'O') && (rx_line[4] == 'N') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'N') )){
            set_LEDnSW_port((i_SensorNo - SENSOR_S0)  ,LED_ON);
        }
        else if( ((rx_line[3] == 'O') && (rx_line[4] == 'F') )
             || ((rx_line[3] == ' ') &&(rx_line[4] == 'O') && (rx_line[5] == 'F') )){
            set_LEDnSW_port((i_SensorNo - SENSOR_S0)  ,LED_OFF);
        }
        else{
            pt_comRec = (char *)&rx_line[3];
            i_num= atoi(pt_comRec);
            if((i_num >= 0 ) && (i_num <= Def_PortSET_Stat_MAX)){
                 set_LEDnSW_port((i_SensorNo - SENSOR_S0) ,i_num);
            }
            else{
                b_CommadERR = 1;
            }
        }
    }
    else{
        b_CommadERR = 1;
    }
    return(b_CommadERR);
 }
 
//------------------------------------------------------------------------------//
//  Command No.11  Setteing Parameter Set No.11    //
// --------------  Capacitor Dose Meter Setting Parameter Declaration  ----------------//
//////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
//typedef struct  st_CapDoseSetting_param{
//    // 共通設定項目
//    int i_RelaySW_Delay;            // 共通　Relay 切り替え時遅延時間[x1msec]
//    //Measure 専用ﾊﾟﾗﾒｰﾀ
//    int   i_Meas_Interval_Time;         //  測定値　AD測定間隔
//    int   i_Meas_Connect_Time;          //  測定時　Sensor　接続時間
//    int   i_Meas_StartDelayTime;        //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
//    int   i_Meas_SampleAveTimes;         // 測定開始から　AD測定回数　
//    int   i_Meas_SampleAveInterval;      // 　AD測定間隔　//  
//        （i_Meas_StartDelayTime　＋　（i_Meas_SampleAveTim　x　i_Meas_SampleAveInterval)） が　i_Meas_Connect_Time より小さい
//    // Charge/DisCharge 共通
//    int   i_CDC_Meas_IntervalTime;      //  充放電時電圧監視　AD測定間隔
//     // Charge 専用ﾊﾟﾗﾒｰﾀ
//    float f_ChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
//     int i_ChargeLimitTime;             //  充電時間　ﾘﾐｯﾄ制限
//    int i_ChargeAflerDelay;             // 充電しきい値達成後Wait
//    
//    // DisCharge 専用   
//    float f_DisChargeTargetVolt;          // 放電時　放電終了　ﾚﾍﾞﾙしきい値[ADC_Level ]
//    int i_DisChargeLimitTime;       //  放電時間　ﾘﾐｯﾄ制限
//    int i_DisChargeAfterDelay;      //　放電時　しきい値達成後　Delay
//}ST_CAPDOSE_SETTING_PARAMETER;
//////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//------------------------------------------------------------------------------//
//  Command No.101  "RDL xxx"  [msec] Relay SW After Delay Relay 切替後接続Delay               //
//int st_CapDoseSetting.i_RelaySW_Delay  // 
//------------------------------------------------------------------------------//
bool com_Check_RDL_ConnDelayAfterRSW(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if((i_num > 0 ) && (i_num <= Def_RELAY_SW_DELAY_SetMax_1S)){
                 st_CapDoseSetting.i_RelaySW_Delay = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.102  "MIT xxxx"  [msec] 測定間隔                                 //
//  int st_CapDoseSetting.i_Meas_Interval_Time                                  //
//        条件　>　測定時接続時間　<　MaxLimit　（<600000msec = 10minutes          // 
//------------------------------------------------------------------------------//
bool com_Check_MIT_MeasurementInterval(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if((i_num > st_CapDoseSetting.i_Meas_Connect_Time ) && (i_num <= Def_MEAS_INTERVAL_SET_MAX_10min)){
                 st_CapDoseSetting.i_Meas_Interval_Time = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.103  "MCT xxxx"  [msec] １測定時1接続時間                          //
//  int st_CapDoseSetting.i_Meas_Connect_Time                                  //
//        条件　>　測定時接続時間　<　MaxLimit　（<60000msec = 1minutes            // 
//              <測定間隔
//------------------------------------------------------------------------------//
bool com_Check_MCT_MeasurementConnectTime(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) && (i_num < st_CapDoseSetting.i_Meas_Interval_Time ) && (i_num <= Def_MEAS_CONNECT_TIME_Max1m)){
                 st_CapDoseSetting.i_Meas_Connect_Time = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.104  "MSD xxxx"  [msec] ｓ測定開始　遅延時間                       //
//  int st_CapDoseSetting.i_Meas_StartDelayTime                                 //
//        条件　<　測定時接続時間　<　MaxLimit　（< = 10sec            // 
//              <測定間隔
//------------------------------------------------------------------------------//
bool com_Check_MSD_MeasSampleStartDelay(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) && (i_num < st_CapDoseSetting.i_Meas_Connect_Time ) && (i_num <= Def_MEAS_START_DELAY_SetMax1sec)){
                 st_CapDoseSetting.i_Meas_StartDelayTime = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.105  "MAT xxxx"  [msec] ｓ測定開始　遅延時間                       //
//  int st_CapDoseSetting.i_Meas_SampleAveTimes                                 //
//        条件　<　測定　Ave回数　            // 
//              <測定間隔
//------------------------------------------------------------------------------//
bool com_Check_MAT_MeasAveTimes(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) && (i_num < st_CapDoseSetting.i_Meas_Connect_Time ) && (i_num <= Def_MEAS_SAMPLE_AVE_Times_SetMax_60000)){
                 st_CapDoseSetting.i_Meas_SampleAveTimes = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.106  "MAI xxxx"  [msec] 測定　繰返しAve　Sample間隔                //
//  int st_CapDoseSetting.i_Meas_SampleAveInterval                                 //
//        条件　<　測定時接続時間　<　MaxLimit　（< = 10sec            // 
//              <測定間隔
//------------------------------------------------------------------------------//
bool com_Check_MAI_MeasSampleAveInterval(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_MEAS_SAMPLE_INETRVAL_SetMax_1sec)){
                 st_CapDoseSetting.i_Meas_SampleAveInterval = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

////  Charge / DisCharge Setting Command
//    int   i_CDC_Meas_IntervalTime;      //  充放電時電圧監視　AD測定間隔
//    float f_ChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
//     int i_ChargeLimitTime;             //  充電時間　ﾘﾐｯﾄ制限
//    int i_ChargeAflerDelay;             // 充電しきい値達成後Wait
//    float f_DisChargeTargetVolt;          // 放電時　放電終了　ﾚﾍﾞﾙしきい値[ADC_Level ]
//    int i_DisChargeLimitTime;       //  放電時間　ﾘﾐｯﾄ制限
//    int i_DisChargeAfterDelay;      //　放電時　しきい値達成後　Delay
//------------------------------------------------------------------------------//
//  Command No.107  "CDI xxxx"  [msec] 測定　Charge/DisCharge 　Sample間隔                //
//  int st_CapDoseSetting.i_Meas_SampleAveInterval                                 //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_CDI_Charge_DisCharge_SampleInterval(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_CDC_MEAS_INTERVAL_SetMax)){
                 st_CapDoseSetting.i_CDC_Meas_IntervalTime = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.108  "CTV xxxx"  [V] 測定　Charge しきい値 V              //
//  int st_CapDoseSetting.f_ChargeTargetVolt                                     //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_CTV_ChargeTargetVoltSet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    float f_num=0.00f;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1; } 
    else{
        pt_comRec = (char *)&rx_line[3];
        f_num = atof(pt_comRec);
        if((f_num >= Def_CHARGE_TARGET_VoltMIN ) && (f_num <= Def_CHARGE_TARGET_VoltMAX)){
                 st_CapDoseSetting.f_ChargeTargetVolt = ( f_num ) ;
        }
        else{b_CommadERR = 1;}
   }
   return(b_CommadERR);
}
//------------------------------------------------------------------------------//
//  Command No.109  "CLT xxxx"  [msec] 測定　Charge 時間　limits             //
//  int st_CapDoseSetting.i_ChargeLimitTime                                     //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_CLT_ChargeLimitTimeSet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_CHARGE_LIMIT_TIME_SetMax_1min)){
                 st_CapDoseSetting.i_ChargeLimitTime = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.110  "CAD xxxx"  [msec] 測定　Charge 後　接続解除Delay             //
//  int st_CapDoseSetting.i_ChargeAflerDelay                                    //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_CAD_ChargeAfterDelaySet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_CHARGE_AFTER_DELAY_SetMax_1min)){
                 st_CapDoseSetting.i_ChargeAflerDelay = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.111  "DTV xxxx"  [V] 測定　Charge しきい値 V              //
//  int st_CapDoseSetting.f_DisChargeTargetVolt                                    //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_DTV_DisChargeTargetVoltSet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    float f_num=0.00f;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1; } 
    else{
        pt_comRec = (char *)&rx_line[3];
        f_num = atof(pt_comRec);
        if((f_num >= Def_DisCHARGE_TARGET_VoltMIN ) && (f_num <= Def_DisCHARGE_TARGET_VoltMAX)){
                 st_CapDoseSetting.f_DisChargeTargetVolt = ( f_num ) ;
        }
        else{b_CommadERR = 1;}
   }
   return(b_CommadERR);
}
//------------------------------------------------------------------------------//
//  Command No.112  "DLT xxxx"  [msec] 測定　Charge 時間　limits             //
//  int st_CapDoseSetting.i_DisChargeLimitTime                                     //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_DLT_DisChargeLimitTimeSet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_DISCHARGE_LIMIT_TIMEMax_1min)){
                 st_CapDoseSetting.i_DisChargeLimitTime = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }

//------------------------------------------------------------------------------//
//  Command No.113  "DAD xxxx"  [msec] 測定　Charge 後　接続解除Delay             //
//  int st_CapDoseSetting.i_ChargeAflerDelay                                    //
//        条件　<　
//------------------------------------------------------------------------------//
bool com_Check_DAD_DisChargeAfterDelaySet(int i_RecCharCount)
{
    bool b_CommadERR=0;
    int i_num=0;
    char *pt_comRec;
    if(i_RecCharCount < 4){ b_CommadERR = 1;} 
    else{
        pt_comRec = (char *)&rx_line[3];
        i_num= atoi(pt_comRec);
        if( (i_num > 0 ) &&  (i_num <= Def_DISCHARGE_AFTER_DELAY_SetMax_1min)){
                 st_CapDoseSetting.i_DisChargeAfterDelay = i_num;
        }
        else{ b_CommadERR = 1; }
   }
    return(b_CommadERR);
 }





//------------------------------------------------------------------//
//-----    Serial rx Commmand Input & Parameter Set Function   -----//
//  Tsunemoto Scince 2013.08.08                                     //
//------------------------------------------------------------------//
void Ser_Command_Input()
{
    int i_RecCharCount;
    bool b_CommadERR = 0;
 
//    int i;
 //   while(rx_cr_Rec > 0){
         // Read a line from the large rx buffer from rx interrupt routine
        i_RecCharCount = read_line();
            
        if(i_RecCharCount >0)
        {
            switch (rx_line[0]){
                case 'P':
                    if(rx_line[1] == 'C'){
                        switch(rx_line[2]){
                            case 'V':
                                b_CommadERR = com_Check_SW_PCV_COMVref(i_RecCharCount);
                                break;
                            case 'G':
                                b_CommadERR = com_Check_SW_PCG_COMGND(i_RecCharCount);
                                break;
                            case 'A':
                                b_CommadERR = com_Check_SW_PCA_COMADG(i_RecCharCount);
                                break;
                            default:
                                break;
                        }
                    }
                    else if (rx_line[1] == 'S'){
                        b_CommadERR = com_Check_SW_Psn_SensConn(i_RecCharCount);
                    }
                    else if (rx_line[1] == 'L'){
                        b_CommadERR = com_Check_SW_PLn_LEDset(i_RecCharCount);
                    }
                    else{
                    }
                    break;
                case 'R':
                    if((rx_line[1] == 'D') && (rx_line[2] == 'L')){
                        b_CommadERR = com_Check_RDL_ConnDelayAfterRSW(i_RecCharCount);
                    }
                    break;
                 case 'M':
                    if((rx_line[1] == 'I') && (rx_line[2] == 'T')){
                        b_CommadERR = com_Check_MIT_MeasurementInterval(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'C') && (rx_line[2] == 'T')){
                        b_CommadERR = com_Check_MCT_MeasurementConnectTime(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'S') && (rx_line[2] == 'D')){
                        b_CommadERR = com_Check_MSD_MeasSampleStartDelay(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'A') && (rx_line[2] == 'T')){
                        b_CommadERR = com_Check_MAT_MeasAveTimes(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'A') && (rx_line[2] == 'I')){
                        b_CommadERR = com_Check_MAT_MeasAveTimes(i_RecCharCount);
                    }
                    break;
                case 'C':
                    if((rx_line[1] == 'D') && (rx_line[2] == 'I')){
                        b_CommadERR = com_Check_CDI_Charge_DisCharge_SampleInterval(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'T') && (rx_line[2] == 'V')){
                        b_CommadERR = com_Check_CTV_ChargeTargetVoltSet(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'L') && (rx_line[2] == 'T')){
                        b_CommadERR = com_Check_CLT_ChargeLimitTimeSet(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'A') && (rx_line[2] == 'D')){
                        b_CommadERR = com_Check_CAD_ChargeAfterDelaySet(i_RecCharCount);
                    }
                   break;
                case 'D':
                    if((rx_line[1] == 'T') && (rx_line[2] == 'V')){
                        b_CommadERR = com_Check_DTV_DisChargeTargetVoltSet(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'L') && (rx_line[2] == 'T')){
                        b_CommadERR = com_Check_DLT_DisChargeLimitTimeSet(i_RecCharCount);
                    }
                    else if((rx_line[1] == 'A') && (rx_line[2] == 'D')){
                        b_CommadERR = com_Check_DAD_DisChargeAfterDelaySet(i_RecCharCount);
                    }
                    break;
                case 'S':
                   switch (rx_line[1]){
                        case 'M':
                            switch (rx_line[2]){
                                case 'S':
                                    Start_MeasureSingle();
                                    
                                    break;
                                case 'N':
                                    Start_MeasureNtimes();
                                    break;
                                case 'C':
                                     Start_MeasureContinuous();
                                   break;
                                default:
                                    b_CommadERR = 2;
                                    break;
                            }        
                            //Start_SampleMeasure();
                            break;
                        case 'C':
                            Start_ChargeMode();
                            break;
                        case 'D':
                            Start_DisChargeMode();
                            break;
                        default:
                            break;
                    }
                    break;

                case '?':
                    com_Stat_Table_Param_Send();
                    /////////////////////////////
                    // Iwate Pattern
                    break;
               default:
                    break;
            }
            //}
        }
 
        if(b_CommadERR == 0){
                sprintf(tx_line,"ACK%d \r\n",rx_cr_Rec); 
                // Copy tx line buffer to large tx buffer for tx interrupt routine
                send_line();
        }
        else{
                sprintf(tx_line,"ERR%d \r\n",b_CommadERR); 
                // Copy tx line buffer to large tx buffer for tx interrupt routine
                send_line();
        }

        rx_cr_Rec--;
          
 //   }
}
//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//  Input :i_RecCharCount :Command Stringth Length                              //
//         rx_line[80] :(Global) Rec Data Stringth                              //
//  Return :bool b_CommadERR  0= ACK                                            //
//                            1= ERR                                            //
//////////////////////////////////////////////////////////////////////////////////
//------------------------------------------------------------------------------//
//  CaｐDose  ?"  Setting  Parameter Repry                                          //
/////////////////////////////// 
//    設定ﾊﾟﾗﾒｰﾀ応答               //
///////////////////////////////
// --------------  Capacitor Dose Meter Setting Parameter Declaration  ----------------//
//typedef struct  st_CapDoseSetting_param{
//    // 共通設定項目
//    int i_RelaySW_Delay;            // 共通　Relay 切り替え時遅延時間[x1msec]
//    //Measure 専用ﾊﾟﾗﾒｰﾀ
//    int   i_Meas_Interval_Time;         //  測定値　AD測定間隔
//    int   i_Meas_Connect_Time;          //  測定時　Sensor　接続時間
//    int   i_Meas_StartDelayTime;        //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
//    int   i_Meas_SampleAveTimes;         // 測定開始から　AD測定回数　
//    int   i_Meas_SampleAveInterval;      // 　AD測定間隔　
//                                        //  （i_Meas_StartDelayTime　＋　（i_Meas_SampleAveTim　x　i_Meas_SampleAveInterval)）
//                                        //  が　i_Meas_Connect_Time より小さい
//    // Charge/DisCharge 共通
//    int   i_CDC_Meas_IntervalTime;      //  充放電時電圧監視　AD測定間隔
//     // Charge 専用ﾊﾟﾗﾒｰﾀ
//    float f_ChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
//     int i_ChargeLimitTime;             //  充電時間　ﾘﾐｯﾄ制限
//    int i_ChargeAflerDelay;             // 充電しきい値達成後Wait
//    
//    // DisCharge 専用   
//    float f_DisChargeTargetVolt;          // 放電時　放電終了　ﾚﾍﾞﾙしきい値[ADC_Level ]
//    int i_DisChargeLimitTime;       //  放電時間　ﾘﾐｯﾄ制限
//    int i_DisChargeAfterDelay;      //　放電時　しきい値達成後　Delay
//    
//}ST_CAPDOSE_SETTING_PARAMETER;
//
//------------------------------------------------------------------------------//
void com_Stat_Table_Param_Send()
{
float f_num;
int i_num;

    sprintf(tx_line,"Capacitor DoseMeter MBED_LPC11U24 Ver0.01 \r\n");
    send_line();

    sprintf(tx_line,"?: Parameter & Status Request \r\n");
    send_line();
    
   // 共通設定項目
    //int i_RelaySW_Delay;                  // 共通　Relay 切り替え時遅延時間[x1msec]
    //st_CapDoseSetting.i_RelaySW_Delay = Def_RELAY_SW_DELAY_5ms;   // Relay SW 切り替え時間 初期値　5msec
    i_num = ((int)st_CapDoseSetting.i_RelaySW_Delay );
   sprintf(tx_line,"Com:RelaySW Change Delay Time [RDL]=%3d[msec]\r\n",i_num);
   send_line();
       //Measure 専用ﾊﾟﾗﾒｰﾀ
    //int   i_Meas_Interval_Time;           //  測定値　AD測定間隔
    //st_CapDoseSetting.i_Meas_Interval_Time = Def_MEAS_INTERVAL_100ms; 
    i_num = ((int)st_CapDoseSetting.i_Meas_Interval_Time );
    sprintf(tx_line,"Meas:Meas Interval Time [MIT]=%6d[msec]\r\n",i_num);
    send_line();   
    //int   i_Meas_Connect_Time;            //  測定時　SeSample AveSample Ave  
    //st_CapDoseSetting.i_Meas_Connect_Time = Def_MEAS_CONNECT_TIME_50ms;
    i_num = ((int)st_CapDoseSetting.i_Meas_Connect_Time );
    sprintf(tx_line,"Meas:1Meas Sensor Connect Time [MCT]=%5d[msec]\r\n",i_num);
    send_line();
    //int   i_Meas_StartDelayTime;          //  ｾﾝｻｰ接続からｾﾝｻｰ値測定開始Delay
    //st_CapDoseSetting.i_Meas_StartDelayTime = Def_MEAS_START_DELAY_10ms;
    i_num = ((int)st_CapDoseSetting.i_Meas_StartDelayTime );
    sprintf(tx_line,"Meas:Sample Delay Time after Sensor Connect [MSD]=%3d[msec]\r\n",i_num);
    send_line();    
    //int   i_Meas_SampleAveTimes;           // 測定開始から　AD測定回数　（i_Meas_StartDelayTime　＋　i_Meas_SampleAveTimes)
    //st_CapDoseSetting.i_Meas_SampleAveTimes = Def_MEAS_SAMPLE_AVE_32Times;
    i_num = ((int)st_CapDoseSetting.i_Meas_SampleAveTimes );
    sprintf(tx_line,"Meas:1Meas Sample Average Times  [MAT]=%3d[times]\r\n",i_num);
    send_line();
    //int   i_Meas_SampleAveInterval;       // 　AD測定間隔　
    //st_CapDoseSetting.i_Meas_SampleAveInterval = Def_MEAS_SAMPLE_INETRVAL_1ms;
    i_num = ((int)st_CapDoseSetting.i_Meas_SampleAveInterval );
    sprintf(tx_line,"Meas:nSample Average Interval Time  [MAI]=%3d[msec]\r\n",i_num);
    send_line();
    //int  i_CDC_Meas_IntervalTime;         //  充放電時電圧監視　AD測定間隔
    //st_CapDoseSetting.i_CDC_Meas_IntervalTime = Def_CDC_MEAS_INTERVAL_5ms;
    i_num = ((int)st_CapDoseSetting.i_CDC_Meas_IntervalTime );
    sprintf(tx_line,"Charge/Discharge: threshold Sample Interval Time [CDI]=%3d[msec]\r\n",i_num);
    send_line();
    // Charge 専用ﾊﾟﾗﾒｰﾀ
    //float f_ChargeTargetVolt;             // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
    //st_CapDoseSetting.f_ChargeTargetVolt = Def_CHARGE_TARGETVolt3V;
    f_num = ((float)st_CapDoseSetting.f_ChargeTargetVolt );
    sprintf(tx_line,"Charge:Charge Target Level [CTV]=%2.3f[V]\r\n",f_num);
    send_line();
     // int i_ChargeLimitTime;               //  充電時間　ﾘﾐｯﾄ制限
    //st_CapDoseSetting.i_ChargeLimitTime = Def_CHARGE_LIMIT_TIME_10sec;
    i_num = ((int)st_CapDoseSetting.i_ChargeLimitTime );
    sprintf(tx_line,"Charge: Charge Wait Time Max Limit [CLT]=%5d[msec]\r\n",i_num);
    send_line();
    //int i_ChargeAflerDelay;               // 充電しきい値達成後Wait
    //st_CapDoseSetting.i_ChargeAflerDelay = Def_CHARGE_AFTER_DELAY5sec;
    i_num = ((int)st_CapDoseSetting.i_ChargeAflerDelay);
    sprintf(tx_line,"Charge: Delay Time After Charge [CAD]=%5d[msec]\r\n",i_num);
    send_line();
    // DisCharge 専用   
   //float f_DisChargeTargetVolt;           // 充電完了判断しきい値ﾚﾍﾞﾙ 設定値
   //st_CapDoseSetting.f_DisChargeTargetVolt = Def_DisCHARGE_TARGETVolt05V;
    f_num = ((float)st_CapDoseSetting.f_DisChargeTargetVolt );
    sprintf(tx_line,"Discharge:Discharge Target Level [DTV]=%2.3f[V]\r\n",f_num);
    send_line();
    //int i_DisChargeLimitTime;             //  放電時間　ﾘﾐｯﾄ制限
    //st_CapDoseSetting.i_DisChargeLimitTime = Def_DISCHARGE_LIMIT_TIME10sec;
    i_num = ((int)st_CapDoseSetting.i_DisChargeLimitTime);
    sprintf(tx_line,"Discharge: Discharge Wait Time Max Limit [DLT]=%5d[msec]\r\n",i_num);
    send_line();
     //int i_DisChargeAfterDelay;            //　放電時　しきい値達成後　Delay
    //st_CapDoseSetting.i_DisChargeAfterDelay = Def_DISCHARGE_AFTER_DELAY5sec;
    i_num = ((int)st_CapDoseSetting.i_DisChargeAfterDelay);
    sprintf(tx_line,"Discharge: Delay Time After Discharge [DAD]=%5d[msec]\r\n",i_num);
    send_line();

// typedef struct st_PortStat{
//    int i_pCOM_Vref_Conn_p5;
//    int i_pCOM_GND_Conn_p6;
//    int i_pCOM_AD_OFF_DRV_p13;
//    int i_pSn_Conn[nSENSOR_6CH];
//    int i_pSn_LED[nSENSOR_6CH];
//}ST_PORT_STATUS;
//ST_PORT_STATUS st_IOPort;
    sprintf(tx_line," Relay & LED Control StatusControl Status\r\n");
    send_line();
    sprintf(tx_line,"Common Relay Control\r\n" );
    send_line();
    sprintf(tx_line," COM_Ref3.2V=[%1d],COM_GND=[%1d],COM_ADGND=[%1d]\r\n"
                    ,st_IOPort.i_pCOM_Vref_Conn_p5
                    ,st_IOPort.i_pCOM_GND_Conn_p6
                    ,st_IOPort.i_pCOM_AD_OFF_DRV_p13
                    );
    send_line();
    sprintf(tx_line," S1=[%1d],S2=[%1d],S3=[%1d],S4=[%1d],S5=[%1d],S6=[%1d]\r\n"
                    ,st_IOPort.i_pSn_Conn[Sel_S1]
                    ,st_IOPort.i_pSn_Conn[Sel_S2]
                    ,st_IOPort.i_pSn_Conn[Sel_S3]
                    ,st_IOPort.i_pSn_Conn[Sel_S4]
                    ,st_IOPort.i_pSn_Conn[Sel_S5]
                    ,st_IOPort.i_pSn_Conn[Sel_S6]
                    );
    send_line();
    sprintf(tx_line," LED1=[%1d],LED2=[%1d],LED3=[%1d],LED4=[%1d],LED5=[%1d],LED6=[%1d]\r\n"
                    ,st_IOPort.i_pSn_LED[Sel_S1]
                    ,st_IOPort.i_pSn_LED[Sel_S2]
                    ,st_IOPort.i_pSn_LED[Sel_S3]
                    ,st_IOPort.i_pSn_LED[Sel_S4]
                    ,st_IOPort.i_pSn_LED[Sel_S5]
                    ,st_IOPort.i_pSn_LED[Sel_S6]
                    );
    send_line();


 
}

