/////////////////////////////////////////////////////////////////////
//  DCapacitor Dose Meter                                          //
//  for 6CH Sensor Charge / Discharge / Measure                    //
//   Since 2019.11.22  H.Tsunemoto                                 //
//   Sample Program Import (1) Serial_interrupt                    //
//                         (2) Timer_Interrupt(100mSec) Sample     //
//                         (3) Internal_ADC_with_interrupt         //
/////////////////////////////////////////////////////////////////////

#include "mbed.h"
#include "stdio.h"
#include "math.h"
#include "LPC11Uxx.h" 

// MBED NXP-LPC11U24 Bord I/O Port Status Define
Serial device(USBTX, USBRX);
// On_Board Debug LED  
DigitalOut myled1(P1_8);
DigitalOut myled2(P1_9);
DigitalOut myled3(P1_10);
DigitalOut myled4(P1_11);

DigitalOut P36_Int1msec(p36);
DigitalOut P35_SampleTiming(p35);

// Common Relay Connect SW
DigitalOut pCOM_Vref_Conn_p5(p5);
DigitalOut pCOM_GND_Conn_p6(p6);
DigitalOut pCOM_AD_OFF_DRV_p13(p13);

// Sensor Relay Connect SW 1 - 6 
DigitalOut pS1_Conn_p7(p7);
DigitalOut pS2_Conn_p8(p8);
DigitalOut pS3_Conn_p9(p9);
DigitalOut pS4_Conn_p10(p10);
DigitalOut pS5_Conn_p11(p11);
DigitalOut pS6_Conn_p12(p12);

// Sensor Error LED 1 - 6
DigitalOut pS1_LED_p30(p30);
DigitalOut pS2_LED_p29(p29);
DigitalOut pS3_LED_p28(p28);
DigitalOut pS4_LED_p27(p27);
DigitalOut pS5_LED_p26(p26);
DigitalOut pS6_LED_p25(p25);

//Sensor 6CH AnalogIn input 1 - 6 (p15,p16,p17,p18,p19,p20 );
AnalogIn    pS1_AD0__p15(p15);
AnalogIn    pS2_AD1__p16(p16);
AnalogIn    pS3_AD2__p17(p17);
AnalogIn    pS4_AD3__p18(p18);
AnalogIn    pS5_AD5__p19(p19);
AnalogIn    pS6_AD6__p20(p20);
//DigitalInOut p10_Power(p10,PIN_OUTPUT,PullNone,0);

///////////////////////////////////////
//  Parameter Data //////////////
/////////////////////////////////
//-------- DosimeterDock Parameter declaration  --------//
typedef struct  st_DosiDock_param{
    int   i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
    int i_P20ADC_Mode;              // P20 ADC Pin Mode Set
                                    //0:P20_Open(No Measure) 1:P20_ADC  2:P20_ADC/Open AutoChange
    float f_ChargeLimitVolt;        // 「V] Power Charge Limit Volt
    float f_ChargeLimitAfter_Delay; // [sec] Power Charge Delay しきい値を超えてからのDelay 
    float f_ChargeOver_Time;        //[sec] Charge Time Max 
    float f_DisCharge_Time;         //「sec] Discharge Time
    int   i_SampleInterval_msec;     // [msec] 測定周期
    int   i_SampleTimes_MAX;        // 測定回数　
 }ST_DOSIDOCK_PARAM;

ST_DOSIDOCK_PARAM st_dosimeterDock_param;
// Parameter Data Define
//i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
#define Def_P10Power_0_GND 0
#define Def_P10Power_1_Vcc 1
#define Def_P10Power_2_Open 2
#define Def_P10Power_2_MAX Def_P10Power_2_Open

#define Def_P20ADC_0_Open 0
#define Def_P20ADC_1_ADC 1
#define Def_P20ADC_2_AutoChange 2
#define Def_P20ADC_4_MAX 4

#define Def_CHARGE_Limit_VoltMIN    1.0f
#define Def_CHARGE_Limit_VoltMAX    3.4f

#define Def_CHARGE_AfterDelay_MIN    0.0f
#define Def_CHARGE_AfterDelay_MAX    3600.0f  // 1H

#define Def_CHARGE_TIME_MIN    1.0f
#define Def_CHARGE_TIME_MAX    3600.0f  // 1H

#define Def_SAMPLE_INTERVAL_MIN    10  // 10[msec]
#define Def_SAMPLE_INTERVAL_MAX    10000  // 10000[msec]

//-------- DosimeterDock Parameter Default Set  --------//
const ST_DOSIDOCK_PARAM const_DosiDock_Param_Default=
{
        1           //i_P10PowerMode　LPC11U24 P10PowerMode Pin Mode
        ,1          //i_P20ADC_Mode　P20 ADC Pin Mode Set
        ,3.2f       //f_ChargeLimitVolt 「V] Power Charge Limit Volt
        ,5.0f       //f_ChargeLimitAfter_Delay [sec] Power Charge Delay しきい値を超えてからのDelay 
        ,15.0f       //f_ChargeOver_Time[sec] Charge Time Max 
        ,5.0f       //f_DisCharge_Time 「sec] Discharge Time
        ,100        //i_SampleInterval_msec     // [msec] 測定周期
        ,5          //i_SampleTimes_MAX;        // 測定回数
};

#define Def_CHARGE_LImitVolt  3.2f           // 「V] Power Charge Limit Volt
#define Def_CHARGE_LimitAfter_Delay 5.0f     // [sec] Power Charge Delay しきい値を超えてからのDelay 
#define Def_Sample_Interval  0.01f           // [sec] ADC Sample Interval
#define Def_CHARGE_OverTime  15.0f           // [sec] Charge Time Max 
//////////////////////////////////////////
//
//////////////////////////////////////////
// Timer Interrupt //
///////////////
Ticker Int1mSec_Timer;
Timer t;

//////////////////////////////////////////////////////////
//--------------------------------//
// --- Serial Communication  ---  //
//--------------------------------//
void Tx_interrupt();
void Rx_interrupt();
void send_line();
int read_line();  // Return Rec CHAR Count 2013.08.08 Tsunemoto Append
void Tick_1mSec_Interrupt();

// Circular buffers for serial TX and RX data - used by interrupt routines
const int ser_buffer_size = 255;
// might need to increase buffer size for high baud rates
char tx_buffer[ser_buffer_size];
char rx_buffer[ser_buffer_size];
// Circular buffer pointers
// volatile makes read-modify-write atomic
volatile int tx_in=0;
volatile int tx_out=0;
volatile int rx_in=0;
volatile int rx_out=0;
// Line buffers for sprintf and sscanf
char tx_line[80];
char rx_line[80];
//---  2013.08.08 Tsunemoto ------//
//--  rx Data Cr Rec Counter
volatile int rx_cr_Rec = 0;
//-----------------------------------------------------------//
//--------- Timer Innterrupt For DAC Control  ---------------//
//-----------------------------------------------------------//
int timer_count=0;
int timer_1Sec=0;
int main_loop_count=0;
//////////////////////////////
////  Measurement Mode   /////
//////////////////////////////
#define SAMPLE_MODE_NotActive   0
#define SAMPLE_MODE_ACTIVE      1
int i_SampleMode=0;
#define CHARGE_MODE_NotActive   0
#define CHARGE_MODE_AfterDelay  1
#define CHARGE_MODE_ACTIVE      2
int i_ChargeMode=0;

#define DISCHARGE_MODE_NotActive   0
#define DISCHARGE_MODE_ACTIVE      1
int i_DisChargeMode=0;

// 100usec Decrement Counter
int i_ADC_IntCount =0;
int i_ADC_Sample_Count=0;
int i_ChargeAfterDelayCount =0;
int i_Charge_TermOverCount=0;
int i_DisChargeTermCount =0;


//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//////////////////////////////////////////////////////////////////////////////////
float ADC_Sample_Send();
void Ser_Command_Input();
void DosiDock_param_init();
void com_Table_Param_Send();
void com_Stat_Table_Param_Send();
//  Command No.1  "PW x"  P10 PowerMode Pin Status  Set                              //
bool com_Check_PW_P10Mode(int i_RecCharCount);
//  Command No.2  "AD x"  P20 ADC Pin Status  Set                              //
bool com_Check_AD_P20Mode(int i_RecCharCount);
//  Command No.3  "LV xx.xx"  [V] Power Charge Limit Volt  Set                              //
bool com_Check_LV_PW_ChargeLimit_V(int i_RecCharCount);
//  Command No.4  "LW xx.xx"  [sec] Power Charge Delay しきい値を超えてからのDelay                             //
bool com_Check_LW_PW_ChargeAfterDelay(int i_RecCharCount);
//  Command No.5  "LT xx.xx"  [sec] Charge Time Max  Set                              //
bool com_Check_LT_ChargeOverTime(int i_RecCharCount);
//  Command No.6  "LD xx.xx"  [「sec] Discharge Time Set                              //
bool com_Check_LD_PW_DisCharge_Time(int i_RecCharCount);
void P10_PWR_Pin_0_GND();
void P10_PWR_Pin_1_High();
void P10_PWR_Pin_2_Open();
void P20_ADC_Pin_0_Open();
void P20_ADC_Pin_1_ADCin();
void P20_ADC_Pin_2_AD_Open();
void P20_ADC_Pin_3_ADPin_GNDTest();
void P20_ADC_Pin_4_ADPin_HighTest();

//----------------------------------------------------------------//

//////////////////////////////////////////////////////////
int main(){
    float f_ADC_inp =0.0;
 
    // Serial Baud 115200
      device.baud(115200);
         timer_count = 0;
         DosiDock_param_init();
    Int1mSec_Timer.attach_us(&Tick_1mSec_Interrupt, (1000));
    t.reset();
     t.start();

// Setup a serial interrupt function to receive data
    device.attach(&Rx_interrupt, Serial::RxIrq);
// Setup a serial interrupt function to transmit data
    device.attach(&Tx_interrupt, Serial::TxIrq);
   timer_count = 0;

//--- ADC Measurement Control Parameter Initial Set ---//
//    adc_param_init(); 
//--- DAC Control Parameter Init --- //
 //   dac1_param_init();
// -- Main Loop -- //
    while (1) {
        if(rx_cr_Rec > 0){
            Ser_Command_Input();
        }
        // Sample Measurement Active
        if(i_SampleMode == SAMPLE_MODE_ACTIVE){
             myled4 = 1.0;
            if(i_ADC_IntCount == 0){
                i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
                f_ADC_inp =ADC_Sample_Send();
                i_ADC_Sample_Count++;
                if(i_ADC_Sample_Count >= st_dosimeterDock_param.i_SampleTimes_MAX){
                    i_SampleMode = SAMPLE_MODE_NotActive;           // Sample End
                   myled4 = 0;
                    sprintf(tx_line,"SM:END\r\n" );
                    send_line();
                }
            }

        }
        // Charge Mode Active
        if(i_ChargeMode > CHARGE_MODE_NotActive){
            myled3 = 1.0;
            if(i_ADC_IntCount == 0){
                i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
                f_ADC_inp =ADC_Sample_Send();
                i_ADC_Sample_Count++;
                if(i_Charge_TermOverCount == 0){                                    // Time Over Check
                    i_ChargeMode = CHARGE_MODE_NotActive; 
                        sprintf(tx_line,"SC:END_Over\r\n");
                        send_line();
               }
                else if (i_ChargeMode == CHARGE_MODE_ACTIVE){                       // Charge Level Check
                    if(f_ADC_inp >= st_dosimeterDock_param.f_ChargeLimitVolt){
                        i_ChargeAfterDelayCount = (int)((st_dosimeterDock_param.f_ChargeLimitAfter_Delay)*10000.0f);
                        i_ChargeMode = CHARGE_MODE_AfterDelay;
                        sprintf(tx_line,"CLV,MS%2.3f,LM%2.3f\r\n",f_ADC_inp,st_dosimeterDock_param.f_ChargeLimitVolt  );
                        send_line();
                    }
                }
                else if (i_ChargeMode == CHARGE_MODE_AfterDelay){
                    if(i_ChargeAfterDelayCount ==0) {                                                      // Charge After Delay
                        i_ChargeMode = CHARGE_MODE_NotActive;
                         sprintf(tx_line,"SC:END\r\n");
                        send_line();
                    }
               } 
            }
            if(i_ChargeMode == CHARGE_MODE_NotActive){
                    P10_PWR_Pin_2_Open();
                     myled3 = 0;
           }
        }
        // DisCharge Mode Active
        if(i_DisChargeMode == DISCHARGE_MODE_ACTIVE){
             myled4 = 1.0;
            if(i_ADC_IntCount == 0){
                i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
                f_ADC_inp =ADC_Sample_Send();
                i_ADC_Sample_Count++;
                if(i_DisChargeTermCount == 0){                                    // Time Over Check
                    i_DisChargeMode = DISCHARGE_MODE_NotActive; 
                    P10_PWR_Pin_2_Open();
                    myled4 = 0;
                         sprintf(tx_line,"SD:END\r\n");
                        send_line();
               }   
            }
       }
        // MainLoop LED Blink
        main_loop_count++;
        if(main_loop_count>=100000){
            myled1 = (myled1+1) & 1;
            main_loop_count = 0;
        }
            /////////////////////////////////
    }
}
//-------------------------------------------------------//
//----------   "SM" ﾚﾍﾞﾙ測定スタート      ---------------------//
// 100usec Decrement Counter
//int i_ADC_IntCount =0;
//int i_ADC_Sample_Count=0;
//-------------------------------------------------------//
void Start_SampleMeasure()
{
    sprintf(tx_line,"SM:START\r\n" );
    send_line();

    AnalogIn p20_Adc(p20);
   i_SampleMode = SAMPLE_MODE_ACTIVE;
    i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
}
//-------------------------------------------------------//
//----------    Chargeスタート      ---------------------//
//-------------------------------------------------------//
void Start_ChargeMode()
{
    sprintf(tx_line,"SC:START\r\n" );
   send_line();
    AnalogIn p20_Adc(p20);
   i_ChargeMode = CHARGE_MODE_ACTIVE;
    i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
    i_Charge_TermOverCount = (int)( (st_dosimeterDock_param.f_ChargeOver_Time) * 10000.0f);
    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
    P10_PWR_Pin_1_High();
 //      sprintf(tx_line,"TIMECOUNT %4d\r\n",i_Charge_TermOverCount );
 //  send_line();
 
}
//-------------------------------------------------------//
//----------    DisChargeスタート      ---------------------//
//-------------------------------------------------------//
void Start_DisChargeMode()
{
    sprintf(tx_line,"SD:START\r\n" );
    send_line();
    i_DisChargeMode = DISCHARGE_MODE_ACTIVE;
    i_ADC_IntCount= (st_dosimeterDock_param.i_SampleInterval_msec)*10;
    i_DisChargeTermCount=(int)( (st_dosimeterDock_param.f_DisCharge_Time) * 10000.0f);
    i_ADC_Sample_Count= 0 ;   //st_dosimeterDock_param.i_SampleTimes_MAX;
    P10_PWR_Pin_0_GND();
}

////////////////////////////////////////////////////////////////////////////////////////////////
//ST_DosimeterDock_PARAM 初期設定
//typedef struct  st_DosiDock_param{
//    int   i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
//                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
//    int i_P20ADC_Mode;              // P20 ADC Pin Mode Set
//                                    //0:P20_Open(No Measure) 1:P20_ADC  2:P20_ADC/Open AutoChange
//    float f_ChargeLimitVolt;        // 「V] Power Charge Limit Volt
//    float f_ChargeLimitAfter_Delay; // [sec] Power Charge Delay しきい値を超えてからのDelay 
//    float f_ChargeOver_Time;        //[sec] Charge Time Max 
//    float f_DisCharge_Time;         //「sec] Discharge Time
//    int   i_SampleInterval_msec;     // [msec] 測定周期
//    int   i_SampleTimes_MAX;        // 測定回数　
// }ST_DOSIDOCK_PARAM;
//
//ST_DOSIDOCK_PARAM st_dosimeterDock_param;
////////////////////////////////////////////////////////////////////////////////////////////////
//-------------------------------------------------------//
void DosiDock_param_init()
{
    st_dosimeterDock_param.i_P10PowerMode = const_DosiDock_Param_Default.i_P10PowerMode;
    st_dosimeterDock_param.i_P20ADC_Mode = const_DosiDock_Param_Default.i_P20ADC_Mode;
    st_dosimeterDock_param.f_ChargeLimitVolt = const_DosiDock_Param_Default.f_ChargeLimitVolt;
    st_dosimeterDock_param.f_ChargeLimitAfter_Delay = const_DosiDock_Param_Default.f_ChargeLimitAfter_Delay;
    st_dosimeterDock_param.f_ChargeOver_Time = const_DosiDock_Param_Default.f_ChargeOver_Time;
    st_dosimeterDock_param.f_DisCharge_Time = const_DosiDock_Param_Default.f_DisCharge_Time;
    st_dosimeterDock_param.i_SampleInterval_msec = const_DosiDock_Param_Default.i_SampleInterval_msec;
    st_dosimeterDock_param.i_SampleTimes_MAX = const_DosiDock_Param_Default.i_SampleTimes_MAX;
 }
//////////////////////////////////////////////
/////      ADC 1 Sample Input & Send        //
//////////////////////////////////////////////
float ADC_Sample_Send()
{
 //   int i;
    float f_ADC_V =0.0;
    
    AnalogIn p20_Adc(p20);
    f_ADC_V  = (float)p20_Adc*3.3;
    sprintf(tx_line,"ADC,%3i,%2.3f\r\n",i_ADC_Sample_Count, f_ADC_V  );
    send_line();
    return(f_ADC_V);
}

////////////////////////////
//  P10 Power Port Set    //
////////////////////////////
void P10_PWR_Pin_0_GND()
{
 }
void P10_PWR_Pin_1_High()
{
}
void P10_PWR_Pin_2_Open()
{
 
}

void P10_PWR_Pin_Setting(int i_selMode)
{
    switch(i_selMode){
        case 0:
            P10_PWR_Pin_0_GND();
        break;
        case 1:
            P10_PWR_Pin_1_High();
        break;
        case 2:
            P10_PWR_Pin_2_Open();
       break;
        default:
        break;
    }
        
}
////////////////////////////
//  P20 ADC Port Set    //
////////////////////////////
void P20_ADC_Pin_0_Open()
{
    DigitalInOut p20_GPIO_Open(p20,PIN_OUTPUT,OpenDrain,1);
}
void P20_ADC_Pin_1_ADCin()
{
    AnalogIn    p20_Adc(p20);

}
void P20_ADC_Pin_2_AD_Open()
{
    DigitalInOut p20_GPIO_Open(p20,PIN_OUTPUT,OpenDrain,1);

}
void P20_ADC_Pin_3_ADPin_GNDTest()
{
    DigitalInOut p20_GPIO_GND(p20,PIN_OUTPUT,PullNone,0);

}
void P20_ADC_Pin_4_ADPin_HighTest()
{
    DigitalInOut p20_GPIO_High(p20,PIN_OUTPUT,PullNone,1);

}
           
void P20_ADC_Pin_Setting(int i_selMode)
{
    switch(i_selMode){
        case 0:
            P20_ADC_Pin_0_Open();
            break;
        case 1:
            P20_ADC_Pin_1_ADCin();
            break;
        case 2:
            P20_ADC_Pin_2_AD_Open();
            break;
        case 3:
            P20_ADC_Pin_3_ADPin_GNDTest();
            break;
         case 4:
            P20_ADC_Pin_4_ADPin_HighTest();
            break;
       default:
            break;
    }
        
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------//
//------------------------------------------------------------------------------//
//-----    Serial tx/rx Communication  
//------------------------------------------------------------------------------//
// Copy tx line buffer to large tx buffer for tx interrupt routine
void send_line() {
    int i;
    char temp_char;
    bool empty;
    i = 0;
// Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(UART_IRQn);
    empty = (tx_in == tx_out);
    while ((i==0) || (tx_line[i-1] != '\n')) {
// Wait if buffer full
        if (((tx_in + 1) % ser_buffer_size) == tx_out) {
// End Critical Section - need to let interrupt routine empty buffer by sending
            NVIC_EnableIRQ(UART_IRQn);
            while (((tx_in + 1) % ser_buffer_size) == tx_out) {
            }
// Start Critical Section - don't interrupt while changing global buffer variables
            NVIC_DisableIRQ(UART_IRQn);
        }
        tx_buffer[tx_in] = tx_line[i];
        i++;
        tx_in = (tx_in + 1) % ser_buffer_size;
    }
    if (device.writeable() && (empty)) {
        temp_char = tx_buffer[tx_out];
        tx_out = (tx_out + 1) % ser_buffer_size;
// Send first character to start tx interrupts, if stopped
        device.putc(temp_char);
    }
// End Critical Section
    NVIC_EnableIRQ(UART_IRQn);
    return;
}


// Read a line from the large rx buffer from rx interrupt routine
// 2013.08.08 H.Tsunemoto 
// Append Return Chear Number
int read_line(){
//void read_line() {
    int i;
    i = 0;
    // Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(UART_IRQn);
    while((rx_in != rx_out) && (rx_buffer[rx_out]<0x20)){
          rx_out = (rx_out + 1) % ser_buffer_size;
     }
     while(rx_in != rx_out){
        rx_line[i] = rx_buffer[rx_out];
         rx_out = (rx_out + 1) % ser_buffer_size;
        if((rx_line[i] == '\r') || (rx_line[i] == '\n')){
            break;
       }
             i++;

    }
    rx_line[i] = 0;
// End Critical Section
    NVIC_EnableIRQ(UART_IRQn);
    return(i);
}
// Interupt Routine to read in data from serial port
void Rx_interrupt() {
    //led1=1;
// Loop just in case more than one character is in UART's receive FIFO buffer
// Stop if buffer full
    while ((device.readable()) || (((rx_in + 1) % ser_buffer_size) == rx_out)) {
        rx_buffer[rx_in] = device.getc();
// Uncomment to Echo to USB serial to watch data flow
//        monitor_device.putc(rx_buffer[rx_in]);
        //------- 2013.08.08 Tsunemoto ------------//
        // -- Char CR Rec Counter ----//
        if(rx_buffer[rx_in]== '\r'){
           //led2 = 1;
           rx_cr_Rec ++;
        }
        //----------------------------//
        rx_in = (rx_in + 1) % ser_buffer_size;
    }
    //led1=0;
    return;
}

// Interupt Routine to write out data to serial port
void Tx_interrupt() {
    //led2=1;
// Loop to fill more than one character in UART's transmit FIFO buffer
// Stop if buffer empty
    while ((device.writeable()) && (tx_in != tx_out)) {
        device.putc(tx_buffer[tx_out]);
        tx_out = (tx_out + 1) % ser_buffer_size;
    }
    //led2=0;
    return;
}

////////////////////////////////////////////////
//   1msec Timer Interrupt
//
// 1msec Decrement Counter
//int i_ADC_IntCount =0;
//int i_ADC_Sample_Count=0;
//int i_Charge_Term_Count=0;
//int i_ChargeAfterDelayCount =0;
//int i_Charge_TermOverCount=0;
//int i_DisChargeTermCount =0;

///////////////////////////////////////////////
void Tick_1mSec_Interrupt() 
{
    P36_Int1msec=1;
    // Sampling Interval 
    if(i_ADC_IntCount >0){
        i_ADC_IntCount--;
    }
    // i_Charge_Term_Count 
    if(i_ChargeAfterDelayCount >0){
        i_ChargeAfterDelayCount--;
    }
    if(i_Charge_TermOverCount >0){
        i_Charge_TermOverCount--;
    }

    // Discharge Term Counter
    if(i_DisChargeTermCount >0){
        i_DisChargeTermCount--;
    }
    
    timer_count++;                  //increment timer_count
    if(timer_count >= 10000){
        timer_count = 0;
        timer_1Sec++;
    }
    P36_Int1msec=0;

}

//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//  Input :i_RecCharCount :Command Stringth Length                              //
//         rx_line[80] :(Global) Rec Data Stringth                              //
//  Return :bool b_CommadERR  0= ACK                                            //
//                            1= ERR                                            //
//////////////////////////////////////////////////////////////////////////////////
/*
//-------- DosimeterDock Parameter declaration  --------//
typedef struct  st_DosiDock_param{
    int   i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
    int i_P20ADC_Mode;              // P20 ADC Pin Mode Set
                                    //0:P20_Open(No Measure) 1:P20_ADC  2:P20_ADC/Open AutoChange
    float f_ChargeLimitVolt;        // 「V] Power Charge Limit Volt
    float f_ChargeLimitAfter_Delay; // [sec] Power Charge Delay しきい値を超えてからのDelay 
    float f_ChargeOver_Time;        //[sec] Charge Time Max 
    float f_DisCharge_Time;         //「sec] Discharge Time
    int   i_SampleInterval_msec;     // [msec] 測定周期
    int   i_SampleTimes_MAX;        // 測定回数　
 }ST_DOSIDOCK_PARAM;

ST_DOSIDOCK_PARAM st_dosimeterDock_param;
// Parameter Data Define
//i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
#define Def_P10Power_0_GND 0
#define Def_P10Power_1_Vcc 1
#define Def_P10Power_2_Open 2
#define Def_P10Power_2_MAX Def_P10Power_2_Open

#define Def_P20ADC_0_Open 0
#define Def_P20ADC_1_ADC 1
#define Def_P20ADC_2_AutoChange 2
#define Def_P20ADC_2_MAX Def_P20ADC_2_AutoChange

#define Def_CHARGE_Limit_VoltMIN    1.0f
#define Def_CHARGE_Limit_VoltMAX    3.4f

#define Def_CHARGE_AfterDelay_MIN    0.0f
#define Def_CHARGE_AfterDelay_MAX    3600.0f  // 1H

#define Def_CHARGE_TIME_MIN    1.0f
#define Def_CHARGE_Time_MAX    3600.0f  // 1H

#define Def_SAMPLE_INTERVAL_MIN    10d  // 10[msec]
#define Def_SAMPLE_INTERVAL_MAX    10000d  // 10000[msec]

*/
//------------------------------------------------------------------------------//
//  Command No.1  "PW x"  P10 PowerMode Pin Status  Set                              //
//int st_dosimeterDock_param.i_P10PowerMode = 1;  // P10PW Pin Mode Set
//------------------------------------------------------------------------------//
bool com_Check_PW_P10Mode(int i_RecCharCount)
{
bool b_CommadERR=0;
int i_num=0;
char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        i_num= atoi(pt_comRec);
        if((i_num >= 0 ) && (i_num <= Def_P10Power_2_MAX)){
                 st_dosimeterDock_param.i_P10PowerMode = i_num;
                 P10_PWR_Pin_Setting(i_num);
        }
        else{
            b_CommadERR = 1;
        }
        
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.2  "AD x"  P20 ADC Pin Status  Set                              //
//int st_dosimeterDock_param.i_P20ADC_Mode = 1;  // P20 ADC Pin Mode Set
//------------------------------------------------------------------------------//
bool com_Check_AD_P20Mode(int i_RecCharCount)
{
bool b_CommadERR=0;
int i_num=0;
char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        i_num= atoi(pt_comRec);
        if((i_num >= 0 ) && (i_num <= Def_P20ADC_4_MAX)){
                 st_dosimeterDock_param.i_P20ADC_Mode = i_num;
                 P20_ADC_Pin_Setting(i_num);
        }
        else{
            b_CommadERR = 1;
        }
        
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.3  "LV xx.xx"  [V] Power Charge Limit Volt  Set                              //
//int st_dosimeterDock_param.f_ChargeLimitVolt = 3.2;  // [V] Power Charge Limit Volt
//------------------------------------------------------------------------------//
bool com_Check_LV_PW_ChargeLimit_V(int i_RecCharCount)
{
bool b_CommadERR=0;
float f_num=0.00f;
char *pt_comRec;
    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        f_num = atof(pt_comRec);
        if((f_num >= Def_CHARGE_Limit_VoltMIN ) && (f_num <= Def_CHARGE_Limit_VoltMAX)){
                 st_dosimeterDock_param.f_ChargeLimitVolt = ( f_num ) ;
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.4  "LW xx.xx"  [sec] Power Charge Delay しきい値を超えてからのDelay                             //
//int st_dosimeterDock_param.f_ChargeLimitAfter_Delay = 5.0;  // [sec] Power Charge Delay
//------------------------------------------------------------------------------//
bool com_Check_LW_PW_ChargeAfterDelay(int i_RecCharCount)
{
bool b_CommadERR=0;
float f_num=0.00f;
char *pt_comRec;
    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        f_num = atof(pt_comRec);
        if((f_num >= Def_CHARGE_AfterDelay_MIN ) && (f_num <= Def_CHARGE_AfterDelay_MAX)){
                 st_dosimeterDock_param.f_ChargeLimitAfter_Delay = ( f_num ) ;
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.5  "LT xx.xx"  [sec] Charge Time Max  Set                              //
//int st_dosimeterDock_param.f_DisCharge_Time = 5.0;  // [sec] Charge Time Max
//------------------------------------------------------------------------------//
bool com_Check_LT_ChargeOverTime(int i_RecCharCount)
{
bool b_CommadERR=0;
float f_num=0.00f;
char *pt_comRec;
    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        f_num = atof(pt_comRec);
        if((f_num >= Def_CHARGE_TIME_MIN ) && (f_num <= Def_CHARGE_TIME_MAX)){
                 st_dosimeterDock_param.f_ChargeOver_Time = ( f_num ) ;
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.6  "LD xx.xx"  [「sec] Discharge Time Set                              //
//int st_dosimeterDock_param.f_DisCharge_Time = 5.0;  // 「sec] Discharge Time
//------------------------------------------------------------------------------//
bool com_Check_LD_PW_DisCharge_Time(int i_RecCharCount)
{
bool b_CommadERR=0;
float f_num=0.00f;
char *pt_comRec;
    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        f_num = atof(pt_comRec);
        if((f_num >= Def_CHARGE_TIME_MIN ) && (f_num <= Def_CHARGE_TIME_MAX)){
                 st_dosimeterDock_param.f_DisCharge_Time = ( f_num ) ;
        }
        else{
            b_CommadERR = 1;
        }
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.7  "MT xxxx"  [[msec] 測定周期                             //
//int st_dosimeterDock_param.i_SampleInterval_msec = 100msec;  // [msec] 測定周期
//------------------------------------------------------------------------------//
bool com_Check_MT_Sampling_Time(int i_RecCharCount)
{
bool b_CommadERR=0;
int i_num=0;
char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        i_num= atoi(pt_comRec);
        if((i_num > 0 ) && (i_num <= Def_SAMPLE_INTERVAL_MAX)){
                 st_dosimeterDock_param.i_SampleInterval_msec = i_num;
        }
        else{
            b_CommadERR = 1;
        }
        
   }
    return(b_CommadERR);
 }
//------------------------------------------------------------------------------//
//  Command No.8  "MN xxxx"  [[Times] // 測定回数                        //
//int st_dosimeterDock_param.i_SampleTimes_MAX = 5 Times;  // [Times] // 測定回数
//------------------------------------------------------------------------------//
bool com_Check_MN_Sample_CountTime(int i_RecCharCount)
{
bool b_CommadERR=0;
int i_num=0;
char *pt_comRec;

    if(i_RecCharCount < 3){
        b_CommadERR = 1;
    } 
    else{
        pt_comRec = (char *)&rx_line[2];
        i_num= atoi(pt_comRec);
        if((i_num > 0 ) && (i_num <= Def_SAMPLE_INTERVAL_MAX)){
                 st_dosimeterDock_param.i_SampleTimes_MAX = i_num;
        }
        else{
            b_CommadERR = 1;
        }
        
   }
    return(b_CommadERR);
 }



//------------------------------------------------------------------//
//-----    Serial rx Commmand Input & Parameter Set Function   -----//
//  Tsunemoto Scince 2013.08.08                                     //
//------------------------------------------------------------------//
void Ser_Command_Input()
{
    int i_RecCharCount;
    bool b_CommadERR = 0;
 
//    int i;
 //   while(rx_cr_Rec > 0){
         // Read a line from the large rx buffer from rx interrupt routine
        i_RecCharCount = read_line();
            
        if(i_RecCharCount >0)
        {
            switch (rx_line[0]){
                case 'P':
                    if(rx_line[1] == 'W'){
                        com_Check_PW_P10Mode(i_RecCharCount);
                    }
                    break;
                case 'A':
                    if(rx_line[1] == 'D'){
                        com_Check_AD_P20Mode(i_RecCharCount);
                    }
                    break;
                case 'L':
                    switch (rx_line[1]){
                        case 'V':
                            com_Check_LV_PW_ChargeLimit_V( i_RecCharCount);
                        break;
                        case 'W':
                            com_Check_LW_PW_ChargeAfterDelay( i_RecCharCount);
                        break;
                        case 'T':
                            com_Check_LT_ChargeOverTime(i_RecCharCount);
                        break;
                        case 'D':
                            com_Check_LD_PW_DisCharge_Time(i_RecCharCount);
                        break;
                        default:
                        break;
                   }
                    break;
                case 'M':
                    switch (rx_line[1]){
                        case 'T':
                            com_Check_MT_Sampling_Time(i_RecCharCount);
                        break;
                        case 'N':
                            com_Check_MN_Sample_CountTime(i_RecCharCount);
                        break;
                        default:
                        break;
                    }
                    break;
                case 'S':
                   switch (rx_line[1]){
                        case 'M':
                            Start_SampleMeasure();
                            break;
                        case 'C':
                            Start_ChargeMode();
                            break;
                        case 'D':
                            Start_DisChargeMode();
                            break;
                        default:
                            break;
                    }
                    break;

                case '?':
                    com_Stat_Table_Param_Send();
                    /////////////////////////////
                    // Iwate Pattern
                    break;
               default:
                    break;
            }
            //}
        }
 
        if(b_CommadERR == 0){
                sprintf(tx_line,"ACK%d \r\n",rx_cr_Rec); 
                // Copy tx line buffer to large tx buffer for tx interrupt routine
                send_line();
        }
        else{
                sprintf(tx_line,"ERR%d \r\n",rx_cr_Rec); 
                // Copy tx line buffer to large tx buffer for tx interrupt routine
                send_line();
        }

        rx_cr_Rec--;
          
 //   }
}
//////////////////////////////////////////////////////////////////////////////////
//------------    Command Check & Set Function ---------------------------------//
//  Input :i_RecCharCount :Command Stringth Length                              //
//         rx_line[80] :(Global) Rec Data Stringth                              //
//  Return :bool b_CommadERR  0= ACK                                            //
//                            1= ERR                                            //
//////////////////////////////////////////////////////////////////////////////////
//------------------------------------------------------------------------------//
//  ADC No.5  "A?"  DAC Parameter Repry                                          //
//-------- DosimeterDock Parameter declaration  --------//
//typedef struct  st_DosiDock_param{
//    int   i_P10PowerMode;           // LPC11U24 P10PowerMode Pin Mode Set
//                                    //0:GND 1:Vcc(Charge)(Digital High) 2:OFF(Open)
//    int i_P20ADC_Mode;              // P20 ADC Pin Mode Set
//                                    //0:P20_Open(No Measure) 1:P20_ADC  2:P20_ADC/Open AutoChange
//    float f_ChargeLimitVolt;        // 「V] Power Charge Limit Volt
//    float f_ChargeLimitAfter_Delay; // [sec] Power Charge Delay しきい値を超えてからのDelay 
//    float f_ChargeOver_Time;        //[sec] Charge Time Max 
//    float f_DisCharge_Time;         //「sec] Discharge Time
//    int   i_SampleInterval_msec;     // [msec] 測定周期
//    int   i_SampleTimes_MAX;        // 測定回数　
// }ST_DOSIDOCK_PARAM;
//
//ST_DOSIDOCK_PARAM st_dosimeterDock_param;
//
//------------------------------------------------------------------------------//
void com_Stat_Table_Param_Send()
{
float f_num;
int i_num;

        sprintf(tx_line,"DosimeterDock MBED_LPC11U24 Ver0.95 \r\n");
        send_line();

        sprintf(tx_line,"?: Parameter & Status Request \r\n");
        send_line();
      
        f_num = ((float)st_dosimeterDock_param.f_ChargeLimitVolt );
        sprintf(tx_line,"LV:Charge Volt Check Level =%2.3f[V]\r\n",f_num);
        send_line();
        f_num = ((float)st_dosimeterDock_param.f_ChargeLimitAfter_Delay );
        sprintf(tx_line,"LW:Charge Delay Time after Charge  =%4.2f[sec]\r\n",f_num);
        send_line();
        f_num = ((float)st_dosimeterDock_param.f_ChargeOver_Time );
        sprintf(tx_line,"LT:Charge Time Limit Max  =%4.2f[sec]\r\n",f_num);
        send_line();
        f_num = ((float)st_dosimeterDock_param.f_DisCharge_Time );
        sprintf(tx_line,"LD:DisCharge Time Set  =%4.2f[sec]\r\n",f_num);
        send_line();
        
        sprintf(tx_line,"Sample:\r\n");
        send_line();

        i_num = ((int)st_dosimeterDock_param.i_SampleInterval_msec );
        sprintf(tx_line,"MT:Sample Interval Time =%4d[msec]\r\n",i_num);
        send_line();
        i_num = ((int)st_dosimeterDock_param.i_SampleTimes_MAX );
        sprintf(tx_line,"MN:Sample Count Time =%4d[Times]\r\n",i_num);
        send_line();

        sprintf(tx_line,"Pin Active Mode:\r\n");
        send_line();
        if(st_dosimeterDock_param.i_P10PowerMode == 0){
            sprintf(tx_line,"PW: P10 Power Pin : 0 GND\r\n");
        }
        else if(st_dosimeterDock_param.i_P10PowerMode == 1){
            sprintf(tx_line,"PW: P10 Power Pin : 1 Vcc(High)\r\n");
        }
        else if(st_dosimeterDock_param.i_P10PowerMode == 2){
             sprintf(tx_line,"PW: P10 Power Pin : 2 Open\r\n");
       }
        else{
            sprintf(tx_line,"PW: P10 Power Pin : ??\r\n");
         }
        send_line();
        
        if(st_dosimeterDock_param.i_P20ADC_Mode == 0){
            sprintf(tx_line,"AD: P20 ADC Inp Pin : 0 Open No Measure\r\n");
        }
        else if(st_dosimeterDock_param.i_P20ADC_Mode == 1){
             sprintf(tx_line,"AD: P20 ADC Inp Pin : 1 ADC Input\r\n");
       }
        else if(st_dosimeterDock_param.i_P20ADC_Mode == 2){
             sprintf(tx_line,"AD: P20 ADC Inp Pin : 2 ADC/Open Change\r\n");
        }
        else if(st_dosimeterDock_param.i_P20ADC_Mode == 3){
             sprintf(tx_line,"AD: P20 ADC Inp Pin : 3 GND(Test)\r\n");
        }
        else if(st_dosimeterDock_param.i_P20ADC_Mode == 4){
             sprintf(tx_line,"AD: P20 ADC Inp Pin : 4 High(Test)\r\n");
        }
        else{
              sprintf(tx_line,"AD: P20 ADC Inp Pin : ???\r\n");
       }
        send_line();
 
}

