#include "common.h"

void ls(mfs *fs);
void listReservedBlocks(mfs *fs);
void printStatus(char status);

int main() {
    char data[1];
    char buf[44];
    unsigned int n;

    mfs fs(0xA0);
    pc.printf("\n\r\n\r[mFS] Formatting... %u bad block headers.\n\r", fs.mkfs(true)); // Quick format EEPROM

    // Mark block 2 as bad
    fs.write((char[]){'\x00','\x00','\x00', '\x00', '\x00'}, 2, 0, 3);
    pc.printf("Block 2 marked as a bad block.\n\r");

    fs.read(data, 0, 0, 1);
    pc.printf("\n\rFirst byte: %X\n\r", data[0]);

    pc.printf("Create file testi.txt: "); printStatus(fs.createFile("testi.txt"));
    pc.printf("Create file testi.txt: "); printStatus(fs.createFile("testi.txt"));
    pc.printf("Create file koe.txt: "); printStatus(fs.createFile("koe.txt"));
    pc.printf("Rm file testi.txt: "); printStatus(fs.removeFile("testi.txt"));
    pc.printf("Create file new.txt: "); printStatus(fs.createFile("new.txt"));
    
    // Try to write multi block file new.txt 
    file *fp3 = new file(&fs, "new.txt", AWRITE);
    pc.printf("Trying to write multi-block file new.txt...\n\r");
    for (n=0; n < BS; n++)
        fp3->write("F", 1);
    
    pc.printf("Create file testi.txt: "); printStatus(fs.createFile("testi.txt"));
    file *fp = new file(&fs, "testi.txt", AWRITE);
    pc.printf("File new.txt opened successfully\n\r");
    fp->write("The quick brown fox jumps over the lazy dog", 44);
    pc.printf("File write OK\n\r");
    fp->flush(); 
    
    pc.printf("Continue to write new.txt...");
    for (n=0; n < 400; n++)
        fp3->write("S", 1);
    pc.printf(" End\n\r");
    
    fp3->rewind();
    fp3->read(buf, 20);
    pc.printf("Rewind test (new.txt): %s\n\r", buf);
    pc.printf("Forwarding new.txt: "); printStatus(fp3->forward(BS-25-22));
    fp3->read(buf, 43);
    pc.printf("Forward test (new.txt): %s\n\r", buf);
    delete fp3;

    pc.printf("Create file yykaa.txt: "); printStatus(fs.createFile("yykaa.txt"));

    fp->rewind();
    fp->read(buf, 44);
    pc.printf("\n\rRead file testi.txt: %s\n\r", buf);
    delete fp;

    // Open read-only file
    file *fp2 = new file(&fs, "koe.txt", RO); /* Keep in mind that error will  *
                                             * be thrown if RO file is       *
                                              * opened in RW mode. So it's    *
                                              * better to check flags before  *
                                              * creating a handle.            */
    // This should fail
    pc.printf("Trying to write RO file koe.txt: "); printStatus(fp2->write("aa", 3));
    delete fp2;
    
    pc.printf("Set file flags 0x07 for file new.txt\n\r");
    fs.setFileFlags((char[]){'\x07'}, "new.txt");

    pc.printf("Rename yykaa.xt => nameChanged.txt...\n\r");
    fs.renameFile("yykaa.txt", "nameChanged.txt");

    ls(&fs);
    listReservedBlocks(&fs);
    
    fs.setFileFlags(0, "new.txt");
    file *fp4 = new file(&fs, "new.txt", DWRITE);
    pc.printf("Trying destructive write with new.txt... ");
    for (n=0; n < 3*(BS-25); n++)
        fp3->write("*", 1);
    pc.printf("End\n\r");
    listReservedBlocks(&fs);
    
}

/** Lists files stored in file system
*
* @param *fs Gets handle to file system in use.
*/
void ls(mfs *fs) {
    unsigned int n, fileCount=0;
    char name[20];
    char flags[1];

    pc.printf("\n\rls:\n\rName:\t\tBlock:\tFlags:\n\r");
    n=0;
    while (1)
    {
        if (fs->findNextFile(n, name, &n) == 0 )
        {
            fs->getFileFlags(flags, name);
            if ((flags[0] & 0x2) != 0)
                goto next; // Skip hidden file
            
            pc.printf("%s", name);
            char len = strlen(name);
            for (char i=0; i < 20-len; i++)
                pc.printf(" ");
            pc.printf("%X\t%X\n\r", n, flags[0]);
        } else break; // Reach end of fs
        fileCount++;
        next:
        n++;
    }

    // Get number of free blocks
    char iFreeBlocks = fs->free();
    // Calculate amount of space used
    unsigned int iSpaceUsed = VOL_SIZE - (iFreeBlocks*BS);
    // Summary
    pc.printf("%u files total\n\r%u/%u kB space used, %u blocks free\n\r\n\r", fileCount, iSpaceUsed/1024, VOL_SIZE/1024, iFreeBlocks);
}

/** Lists blocks marked as used
*
* @param *fs Gets handle to file system in use.
*/
void listReservedBlocks(mfs *fs)
{
    char data[5];
    
    pc.printf("\n\rReserved blocks and list of related links:\n\r");
    for (unsigned int n=0; n < BC; n++)
    {
        fs->read(data, n, 0, 5);
        if (data[0] != 4)
            pc.printf("b%XB0: %X, %X, %X\n\r", n, data[0], data[1]<<8|data[2], data[3]<<8|data[4]);
    }
}

void printStatus(char status)
{
    if (status == 0)
        pc.printf("OK\n\r");
    else
        pc.printf("Failed: %u\n\r", status);
}