/** @file mfs.h */ 
/*H****************************************************************************
 * FILENAME :        mfs.h                                                    *
 *                                                                            *
 * DESCRIPTION :                                                              *
 *       mFS file system implementation for mBED with external I2C EEEPROM.   *
 *                                                                            *
 * ---------------------------------------------------------------------------*
 * "THE BEER-WARE LICENSE" (Revision 42):                                     *
 * <olli.vanhoja@gmail.com> wrote this file. As long as you retain this notice*
 * you can do whatever you want with this stuff. If we meet some day, and you *
 * think this stuff is worth it, you can buy me a beer in return Olli Vanhoja *
 * ---------------------------------------------------------------------------*
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 *                                                                            *
 *                                                                            *
 * Block Flags:                                                               *
 * 7:FBOF      Begining of file                                               *
 * 6:LBOF      Last block of file                                             *
 * 5:RO        Read only file (Used only with FBOF)                           *
 * 4:HIDDEN    Hidden file (Used only with FBOF)                              *
 * 3:INUSE     Block in use                                                   *
 * 2:NBAD      Bad block (INV)                                                *
 * 1:VOL       Volume label (Used only with FBOF)                             *
 * 0:LOCK      Locked file (Used only with FBOF)                              *
 *                                                                            *
 * AUTHOR :         Olli Vanhoja       START DATE :    2011-02-18             *
 ******************************************************************************
 *
 * CHANGES :
 *
 * VERSION DATE       WHO            DETAIL
 * 0.1     2011-02-21 Olli Vanhoja   Initial release version
 * 0.2     2011-02-21 Olli Vanhoja   Documentational comments added
 * 0.3     2011-02-21 Olli Vanhoja   * File::read issues fixed, rewind/forward
 *                                     functions improved
 *                                   * Added possibility change I2C speed
 *                                   * I2C autoreset on failure
 * 0.4     2011-02-22 Olli Vanhoja   * mfs::renameFile(char [20], char [20] function added
 *                                   * Incresed fault tolerance by first allocating new
 *                                     block and then linking to it from previous block
 *                                   * Reconstructed initialization and use of some variables
 * 0.5     2011-02-22 Olli Vanhoja   * Improved documentation
 *                                   * Block variables changed from char to uint32_t for
 *                                     code optimization (and for possible 16bit update which
 *                                     would technically allow 256 TB volume sizes)
 *                                   * Optimized file searching algorithms
 * 0.6     2011-02-22 Olli Vanhoja   * Fixed file remove issue
 *                                   * Fixed mkfs bug
 * 0.7     2011-02-23 Olli Vanhoja   * file::seek(uint32_t) added
 * 0.7     2011-02-23 Olli Vanhoja   * Fixed destroying prev link issue on flush
 *                                   * Fixed Forwarding issue which moved cursor at the begining
 *                                     of the filename block
 *                                   * Separated locating of next/prev links functionality
 *                                     into its own private function
 *                                   * 16 bit block number pointers 256 TB theoretical maximum
 *                                     volume size, WHOA \O/
 *                                   * Remove and rename respects RO bit
 *                                   * SetFileFlags fixed
 *                                   * New file write mode DWRITE
 * 0.8     2011-02-24 Olli Vanhoja   * EOF should be now found in AWRITE mode and EOF is
 *                                   * also written by mfs::createFile() so there should
 *                                     be no more confusing situations with new files.
 *
 * TODO :
 *          * Directory support (VOL blocks?)
 *          * RAID 0 & 1
 *H*/

#ifndef MFS_H
#define MFS_H

#include "i2c_eeprom.h"

const unsigned int VOL_SIZE=65536; /**< EEPROM chip size in bytes */
const unsigned int BS=1024; /**< How many bytes per block (default: 4096 bytes) */
const unsigned int BC=VOL_SIZE / BS; // block count
const char mEOF='\x01'; // End Of File/Section marked
const unsigned int BUF=400; /**< File buffer length */

/** mFS File System class
 * 
 * This class is used as a handle for the fs in use.
 */
class mfs {
private:
    i2c_eeprom *mem; // Only 512 kB I2C EEPROM is supported ATM
public:
    /** Create a new file system object
    *
    * @param xi2c_address a Physical I2C address of the EEPROM chip  
    */
    mfs(int i2c_address);
    
    /** Read data from specified fs block
    *
    * @param *data Pointer for readed data
    * @param block Block number.
    * @param byte Selected byte.
    * @param n Bytes to be read.
    * @returns Error code: 0 = OK, 1 = Incorrect input
    */
    char read(char *data, uint32_t block, uint32_t byte, uint32_t n);
    
    /** Write data to specified fs block
    *
    * @param *data Pointer for readed data
    * @param block Block number.
    * @param byte Selected byte.
    * @param n Bytes to be read.
    * @returns Error code: 0 = OK, 1 = Incorrect input
    */
    char write(char *data, uint32_t block, uint32_t byte, uint32_t n);
    
    /** Locate next free block
    *
    * @param *blockOut Returns next free block from begining of the fs.
    * @returns Error code: 0 = OK, 1 = Out of space
    */
    char getNextFreeBlock(uint32_t *blockOut);
    
    /** Locates next starting file from parameter block
    *
    * @param block Start scanning from this block.
    * @param *filenameOut Return name of the file found.
    * @param Returns block number of the file found.
    * @returns Error code: 0 = OK, 1 = Empty fs
    */
    char findNextFile(uint32_t block, char *filenameOut, uint32_t *blockOut);
    
    /** Get block number of the given file
    *
    * Returns block number of the block flaged with FBOF flag.
    *
    * @param filename[20] Filename input.
    * @param Returns block number of the first block of the given file.
    * @returns Error code: 0 = OK, 1 = File not found
    */
    char getFirstBlockOfFile(char filename[20], uint32_t *blockOut);
    
    /** Create a new empty file
    *
    * Reserves one block for the file created.
    *
    * @param filename[20] Filename input.
    * @returns Error code: 0 = OK, 1 = File exists already, 2 = Out of space
    */
    char createFile(char filename[20]);
    
    /** Remove a file from the file system
    *
    * @param filename[20] Filename input.
    * @returns Error code: 0 = OK, 1 = File doesn't exists, 2 = RO file
    */
    char removeFile(char filename[20]);
    
    /** Rename a file
    *
    * @param oldFilename[20] Old filename.
    * @param newFilename[20] New file name.
    * @returns Error code: 0 = OK, 1 = File doesn't exists, 2 = RO file, 3 = fs is corrupted
    */
    char renameFile(char oldFilename[20], char newFilename[20]);
    
    /** Set user modifiable flags.
    *
    * \code
    * desc RO|HIDDEN|LOCK
    * bit  3    2       1
    * \endcode
    *
    * @param *flags Flag input
    * @param filename[20] Filename input.
    * @returns Error code: 0 = OK, 1 = File doesn't exists, 2 = File system is corrupted
    */
    char setFileFlags(char *flags, char filename[20]);
    
    /** Read user modifiable flags.
    *
    * \code
    * desc RO|HIDDEN|LOCK
    * bit  3    2       1
    * \endcode
    *
    * @param *flags Flag output
    * @param filename[20] Filename input.
    * @returns Error code: 0 = OK, 1 = File doesn't exists
    */
    char getFileFlags(char *flags, char filename[20]);
    
    /** Get number of free blocks
    *
    * @returns Number of free blocks.
    */
    uint32_t free();
    
    /** Format new file system
    *
    * \note Keep in mind that only first byte is checked for functionality and
    * if it's broken the who file system is useless.
    *
    * @param createLabel Create volume label at the begining of the file system. (there is no specified use for volume labels atm).
    * @returns Number of bad block headers.
    */
    uint32_t mkfs(bool createLabel);
};

enum BlockLinkType {NEXT, PREV};
enum FileOpenMode {RO, AWRITE, DWRITE};

/** mFS File handle class
 * 
 * This class provides a file handle and data manipulation methods to be
 * used for files stored in mFS files system.
 */
class file {
private:
    mfs *fs;   // Reference to the file system in use
    FileOpenMode fMode;
    char buffer[BUF]; // Write buffer
    uint32_t bufPos; // "Cursor" position in buffer
    uint32_t firstBlock; // First block of the file
    uint32_t currBlock; // Current block in use
    uint32_t blockPos; // "head" position on the current block
    uint32_t byteCount; // Stores current "cursor" position in file for seek
    // Private functions
    char getBlockLink(BlockLinkType linkSelection, uint32_t *blockOut);
    char removeFollowingBlocks(uint32_t block); // Offers destructive write/very simple wear levelling
public:
    /** Create file handle
    *
    * \warning File must be created before it can be opened!
    * Opening non-existing file will trip the system to error();
    * If read only file is opened in rw mode system will trip to error().
    *
    * \b AWRITE is a file access mode where cursor can be moved along the file
    * and write can be started at any point. write() function will overwrite
    * only as many bytes as you chosen to write.
    *
    * \b DWRITE is a file access mode similiar to AWRITE but when you start
    * writing all the data after cursor will be removed permanently and flush()
    * will set a new EOF marker.
    *
    * @param filename[20] Filename input.
    * @param operation RO = Read only, AWRITE = read and write, DWRITE = read + destructive write.
    */
    file(mfs *fs_ref, char filename[20], FileOpenMode operation);
    
    /** Close file handle
    *
    * Flushes the file and closes the handle.
    */
    ~file(); // Close file handle and flush
    
    /** Rewind to the start postion of the file
    *
    */
    void rewind();
    
    /** Reverse n bytes back
    *
    * @param n Number of bytes.
    * @returns Error code: 0 = OK, 1 = First byte of file.
    */
    char rewind(uint32_t n);
    
    /** Forward one byte
    *
    * @returns Error code: 0 = OK, 1 = End of file.
    */
    char forward();
    
    /** Forward n bytes
    *
    * @param n Number of blocks.
    * @returns Error code: 0 = OK, 1 = End of file.
    */
    char forward(uint32_t n);
    
    /** Seek to byte given
    *
    * @param byte Byte number where to seek.
    * @returns Error code: 0 = OK, 1 = End of file or already at first byte.
    */
    char seek(uint32_t byte);
    
    /** Reads a string of bytes
    *
    * Always places '\0' at the end of string.
    *
    * @param data Output buffer.
    * @param n Number of bytes to be read.
    * @returns Error code. 0 = OK, 1 = Last block of the file
    */
    void read(char *data, uint32_t n);
    
    /** Reads a binary array of bytes
    *
    * Doesn't add '\0' at the end of data array and doesn't respect mEOF byte.
    *
    * @param data Output buffer.
    * @param n Number of bytes to be read.
    */   
    void readBin(char *data, uint32_t n);
    
    /** Write byte array to a file (buffer)
    *
    * @param data Input data.
    * @param n Number of bytes to be read.
    * @returns Error code: 0 = OK, 1 = Flush failed.
    */    
    char write(char *data, uint32_t n);
    
    /** Flush file buffer
    * Writes buffer to the EEPROM chip in use.
    *
    * @returns Error code: 0 = OK, 1 = Out of free space, 2 = Destructive operation failed (fs is corrupted).
    */
    char flush();
};

#endif