/**
 * @author Abraham Marsen & Allison Ashlock
 * Georgia Institute of Technology 
 * ECE 4180 Embedded Systems Design
 * Professor Hamblen
 * Spring 2015
 * 
 * @section LICENSE
 * ----------------------------------------------------------------------------
 * "THE BEER-WARE LICENSE" (Revision 42):
 * <amarsen3@gmail.com> wrote this file. As long as you retain this notice you
 * can do whatever you want with this stuff. If we meet some day, and you think
 * this stuff is worth it, you can buy me a beer in return.
 * ----------------------------------------------------------------------------
 *
 *
 * @section DESCRIPTION
 *
 * Avago Digital Proximity, Ambient Light, RGB and Gesture Sensor APDS-9960
 *
 * Datasheet, specs, and information:
 *
 * https://www.sparkfun.com/products/12787
 */



#ifndef APDS9960_H
#define APDS9960_H

#include "mbed.h"

//Location of I2C slave
#define APDS9960_I2C_ADDR   0x72

/* Error code for returned values */
#define ERROR 0xFF

/* Acceptable device IDs */
#define APDS9960_ID_1 0xAB
#define APDS9960_ID_2 0x9C

//reused wait time
#define FIFO_TIME           0.30

//Gest param
#define GEST_THRESHOLD_OUT 5
#define GEST_SENSITIVITY_1 50
#define GEST_SENSITIVITY_2 20

//LOOK AT ALL MY ADDRESSES
#define APDS9960_ENABLE     0x80
#define APDS9960_ATIME      0x81
#define APDS9960_WTIME      0x83
#define APDS9960_AILTL      0x84
#define APDS9960_AILTH      0x85
#define APDS9960_AIHTL      0x86
#define APDS9960_AIHTH      0x87
#define APDS9960_PILT       0x89
#define APDS9960_PIHT       0x8B
#define APDS9960_PERS       0x8C
#define APDS9960_CONFIG1    0x8D
#define APDS9960_PPULSE     0x8E
#define APDS9960_CONTROL    0x8F
#define APDS9960_CONFIG2    0x90
#define APDS9960_ID         0x92
#define APDS9960_STATUS     0x93
#define APDS9960_CDATAL     0x94
#define APDS9960_CDATAH     0x95
#define APDS9960_RDATAL     0x96
#define APDS9960_RDATAH     0x97
#define APDS9960_GDATAL     0x98
#define APDS9960_GDATAH     0x99
#define APDS9960_BDATAL     0x9A
#define APDS9960_BDATAH     0x9B
#define APDS9960_PDATA      0x9C
#define APDS9960_POFFSET_UR 0x9D
#define APDS9960_POFFSET_DL 0x9E
#define APDS9960_CONFIG3    0x9F
#define APDS9960_GPENTH     0xA0
#define APDS9960_GEXTH      0xA1
#define APDS9960_GCONF1     0xA2
#define APDS9960_GCONF2     0xA3
#define APDS9960_GOFFSET_U  0xA4
#define APDS9960_GOFFSET_D  0xA5
#define APDS9960_GOFFSET_L  0xA7
#define APDS9960_GOFFSET_R  0xA9
#define APDS9960_GPULSE     0xA6
#define APDS9960_GCONF3     0xAA
#define APDS9960_GCONF4     0xAB
#define APDS9960_GFLVL      0xAE
#define APDS9960_GSTATUS    0xAF
#define APDS9960_IFORCE     0xE4
#define APDS9960_PICLEAR    0xE5
#define APDS9960_CICLEAR    0xE6
#define APDS9960_AICLEAR    0xE7
#define APDS9960_GFIFO_U    0xFC
#define APDS9960_GFIFO_D    0xFD
#define APDS9960_GFIFO_L    0xFE
#define APDS9960_GFIFO_R    0xFF

//bitmasks
#define APDS9960_PON        0x01
#define APDS9960_AEN        0x02
#define APDS9960_PEN        0x04
#define APDS9960_WEN        0x08
#define APSD9960_AIEN       0x10
#define APDS9960_PIEN       0x20
#define APDS9960_GEN        0x40
#define APDS9960_GVALID     0x01

/* On/Off definitions */
#define OFF                 0
#define ON                  1

// Acceptable parameters for setMode
enum {
    POWER,
    AMBIENT_LIGHT,
    PROXIMITY,
    WAIT,
    AMBIENT_LIGHT_INT,
    PROXIMITY_INT,
    GESTURE,
    ALL
};

/* LED Drive values */
enum {
    LED_DRIVE_100MA,
    LED_DRIVE_50MA,
    LED_DRIVE_25MA,
    LED_DRIVE_12_5MA
};

/* Proximity Gain (PGAIN) values */
enum {
    PGAIN_1X,
    PGAIN_2X,
    PGAIN_4X,
    PGAIN_8X
};

/* ALS Gain (AGAIN) values */
enum {
    AGAIN_1X,
    AGAIN_4X,
    AGAIN_16X,
    AGAIN_64X
};

/* Gesture Gain (GGAIN) values */
enum {
    GGAIN_1X,
    GGAIN_2X,
    GGAIN_4X,
    GGAIN_8X
};
/* LED Boost values */
enum {
    LED_BOOST_100,
    LED_BOOST_150,
    LED_BOOST_200,
    LED_BOOST_300
};

/* Gesture wait time values */
enum {
    GWTIME_0MS,
    GWTIME_2_8MS,
    GWTIME_5_6MS,
    GWTIME_8_4MS,
    GWTIME_14_0MS,
    GWTIME_22_4MS,
    GWTIME_30_8MS,
    GWTIME_39_2MS
};

/* Default values */
#define DEFAULT_ATIME           219 // 103ms
#define DEFAULT_WTIME           246 // 27ms
#define DEFAULT_PROX_PPULSE     0x87 // 16us, 8 pulses
#define DEFAULT_GEST_PPULSE     0x89 // 16us, 10 pulses
#define DEFAULT_POFFSET_UR      0 // 0 offset
#define DEFAULT_POFFSET_DL      0 // 0 offset
#define DEFAULT_CONFIG1         0x60 // No 12x wait (WTIME) factor
#define DEFAULT_LDRIVE          LED_DRIVE_100MA
#define DEFAULT_PGAIN           PGAIN_4X
#define DEFAULT_AGAIN           AGAIN_4X
#define DEFAULT_PILT            0 // Low proximity threshold
#define DEFAULT_PIHT            50 // High proximity threshold
#define DEFAULT_AILT            0xFFFF // Force interrupt for calibration
#define DEFAULT_AIHT            0
#define DEFAULT_PERS            0x11 // 2 consecutive prox or ALS for int.
#define DEFAULT_CONFIG2         0x01 // No saturation interrupts or LED boost
#define DEFAULT_CONFIG3         0 // Enable all photodiodes, no SAI
#define DEFAULT_GPENTH          40 // Threshold for entering gesture mode
#define DEFAULT_GEXTH           30 // Threshold for exiting gesture mode
#define DEFAULT_GCONF1          0x40 // 4 gesture events for int., 1 for exit
#define DEFAULT_GGAIN           GGAIN_4X
#define DEFAULT_GLDRIVE         LED_DRIVE_100MA
#define DEFAULT_GWTIME          GWTIME_2_8MS
#define DEFAULT_GOFFSET         0 // No offset scaling for gesture mode
#define DEFAULT_GPULSE          0xC9 // 32us, 10 pulses
#define DEFAULT_GCONF3          0 // All photodiodes active during gesture
#define DEFAULT_GIEN            0 // Disable gesture interrupts

//Directions X, Y, Z
enum {
    DIR_NA,
    DIR_W,
    DIR_E,
    DIR_N,
    DIR_S,
    DIR_I,
    DIR_O,
    DIR_NW,
    DIR_SW,
    DIR_IW,
    DIR_OW,
    DIR_NE,
    DIR_SE,
    DIR_IE,
    DIR_OE,
    DIR_NI,
    DIR_NO,
    DIR_SI,
    DIR_SO,
    DIR_AN
};

/* State definitions */
enum {
  NA_STATE,
  NEAR_STATE,
  FAR_STATE,
  ALL_STATE
};

/* Container for gesture data */
typedef struct gesture_data_type {
    uint8_t u_data[32];
    uint8_t d_data[32];
    uint8_t l_data[32];
    uint8_t r_data[32];
    uint8_t index_gest;
    uint8_t total_gests;
    uint8_t in_threshold;
    uint8_t out_threshold;
} gest_data_type;

/* APDS9960 Class */
class APDS9960 {
public:
/* Initialization methods */
    APDS9960(PinName sda, PinName scl);
    bool init();
    uint8_t getMode();
    bool setMode(uint8_t mode, uint8_t enable);
/* Turn the APDS-9960 on and off */
    bool enPower();
    bool disPower();
/* Enable or disable specific sensors */
    bool enLightSens(bool interrupts = false);
    bool disLightSens();
    bool enProxSens(bool interrupts = false);
    bool disProxSens();
    bool enGestSens(bool interrupts = true);
    bool disGestSens();
/* LED drive strength control */
    uint8_t getLEDDrive();
    bool setLEDDrive(uint8_t drive);
    uint8_t getGestLEDDrive();
    bool setGestLEDDrive(uint8_t drive);
/* Gain control */
    uint8_t getAmbientLightGain();
    bool setAmbientLightGain(uint8_t gain);
    uint8_t getProxGain();
    bool setProxGain(uint8_t gain);
    uint8_t getGestGain();
    bool setGestGain(uint8_t gain);
/* Get and set light interrupt thresholds */
    bool getLightIntLowThreshold(uint16_t &threshold);
    bool setLightIntLowThreshold(uint16_t threshold);
    bool getLightIntHighThreshold(uint16_t &threshold);
    bool setLightIntHighThreshold(uint16_t threshold);
/* Get and set proximity interrupt thresholds */
    bool getProxIntLowThreshold(uint8_t &threshold);
    bool setProxIntLowThreshold(uint8_t threshold);
    bool getProxIntHighThreshold(uint8_t &threshold);
    bool setProxIntHighThreshold(uint8_t threshold);
/* Get and set interrupt enables */
    uint8_t getAmbientLightIntEnable();
    bool setAmbientLightIntEnable(uint8_t enable);
    uint8_t getProxIntEnable();
    bool setProxIntEnable(uint8_t enable);
    uint8_t getGestIntEnable();
    bool setGestIntEnable(uint8_t enable);
/* Clear interrupts */
    bool clrAmbientLightInt();
    bool clrProxInt();
/* Ambient light methods */
    bool readAmbientLight(uint16_t &val);
    bool readRedLight(uint16_t &val);
    bool readGreenLight(uint16_t &val);
    bool readBlueLight(uint16_t &val);
/* Prox methods */
    bool readProx(uint8_t &val);
/* Gesture methods */
    bool isGestAvailable();
    int readGest();
/*Destructor*/
    ~APDS9960();


private:
/* Gesture processing */
    I2C i2c_;
    void rstGestParameters();
    bool procGestData();
    bool decodeGest();
/* Proximity Interrupt Threshold */
    uint8_t getProxIntLowThresh();
    bool setProxIntLowThresh(uint8_t threshold);
    uint8_t getProxIntHighThresh();
    bool setProxIntHighThresh(uint8_t threshold);
/* LED Boost Control */
    uint8_t getLEDBoost();
    bool setLEDBoost(uint8_t boost);
/* Proximity photodiode select */
    uint8_t getProxGainCompEnable();
    bool setProxGainCompEnable(uint8_t enable);
    uint8_t getProxPhotoMask();
    bool setProxPhotoMask(uint8_t mask);
/* Gesture threshold control */
    uint8_t getGestEnterThresh();
    bool setGestEnterThresh(uint8_t threshold);
    uint8_t getGestExitThresh();
    bool setGestExitThresh(uint8_t threshold);
/* Gesture LED, gain, and time control */
    uint8_t getGestWaitTime();
    bool setGestWaitTime(uint8_t time);
/* Gesture mode */
    uint8_t getGestMode();
    bool setGestMode(uint8_t mode);
/* Raw I2C Commands */
    bool WriteByte(uint8_t val);
    int WriteDataByte(char reg, char val);
    bool WriteDataBlock(uint8_t reg, uint8_t *val, unsigned int len);
    uint8_t ReadDataByte(char reg);
    int ReadDataBlock(uint8_t reg, uint8_t *val, unsigned int len);
/* Members */
    gest_data_type gest_data_;
    int gest_ud_delta_;
    int gest_lr_delta_;
    int gest_ud_count_;
    int gest_lr_count_;
    int gest_near_count_;
    int gest_far_count_;
    int gest_state_;
    int gest_motion_;
};

#endif