/* mbed USBHost Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#pragma once
#include "rtos.h"
#include "FunctionPointer.h"
#include "USBHostTypes.h"
#include "USBDeviceConnected.h"

class USBDeviceConnected;

#define HCTD_QUEUE_SIZE 3

struct HCTD;
struct HCED;
struct HCITD;
class USBHost;

/**
* USBEndpoint class
*/
class USBEndpoint {
public:
    /**
    * Constructor
    */
    USBEndpoint(USBDeviceConnected* _dev) {
        init(CONTROL_ENDPOINT, IN, 8, 0);
        dev = _dev;
    }

    /**
    * Initialize an endpoint
    *
    * @param type endpoint type
    * @param dir endpoint direction
    * @param size endpoint size
    * @param ep_number endpoint number
    */
    void init(ENDPOINT_TYPE _type, ENDPOINT_DIRECTION _dir, uint32_t size, uint8_t ep_number) {
        setState(USB_TYPE_FREE);
        setType(_type);
        dir = _dir;
        MaxPacketSize = size;
        address = ep_number;
        m_pED = NULL;
        //data01_toggle = DATA0; // for KL46Z
    }  

    /**
     *  Attach a member function to call when a transfer is finished
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void attach(T* tptr, void (T::*mptr)(void)) {
        if((mptr != NULL) && (tptr != NULL)) {
            rx.attach(tptr, mptr);
        }
    }

    /**
     * Attach a callback called when a transfer is finished
     *
     * @param fptr function pointer
     */
    void attach(void (*fptr)(void)) {
        if(fptr != NULL) {
            rx.attach(fptr);
        }
    }

    /**
    * Call the handler associted to the end of a transfer
    */
    void call() {
        rx.call();
    };

    void irqWdhHandler(HCTD* td) {m_queue.put(td);} // WDH
    HCTD* get_queue_HCTD(uint32_t millisec=osWaitForever);
    HCED* m_pED;
    // report
    uint8_t m_ConditionCode;
    int m_report_queue_error;

    void setType(ENDPOINT_TYPE _type) { type = _type; };
    void setState(USB_TYPE st){ state = st; };
    void setLengthTransferred(int len) { transferred = len; };
    void setBuffer(uint8_t* buf, int size) { buf_start = buf, buf_size = size; }
    void setSize(int size) { MaxPacketSize = size; }
    void setNextEndpoint(USBEndpoint* ep) { nextEp = ep; };

    USBDeviceConnected* getDevice() { return dev; }
    ENDPOINT_TYPE getType() { return type; };
    USB_TYPE getState() { return state; }
    int getLengthTransferred() { return transferred; }
    uint8_t *getBufStart() { return buf_start; }
    int getBufSize() { return buf_size; }
    uint8_t getAddress(){ return address; };
    int getSize() { return MaxPacketSize; }
    ENDPOINT_DIRECTION getDir() { return dir; }
    USBEndpoint* nextEndpoint() { return nextEp; };

private:
    ENDPOINT_TYPE type;
    USB_TYPE state;
    ENDPOINT_DIRECTION dir;
    USBDeviceConnected* dev;
    uint8_t address;
    int transferred;
    uint8_t * buf_start;
    int buf_size;
    FunctionPointer rx;
    int MaxPacketSize;
    USBEndpoint* nextEp;

protected:
    Queue<HCTD, HCTD_QUEUE_SIZE> m_queue; // TD done queue
    int m_td_queue_count;
};

class EndpointQueue {
public:
    EndpointQueue():head(NULL),tail(NULL) {}
    void push(USBEndpoint* ep) {
        if (head) {
            tail->setNextEndpoint(ep);
        } else {
            head = ep;
        }
        tail = ep;
        ep->setNextEndpoint(NULL);
    }
    USBEndpoint* pop() {
        USBEndpoint* ep = head;
        if (ep) {
            head = ep->nextEndpoint();
        }
        return ep;
    }
    bool empty() { return head == NULL; }

private:
    USBEndpoint* head;
    USBEndpoint* tail;
};


