/*
 * DMX512 send/recv library
 * Copyright (c) 2013 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief DMX512 send/recv
 */
 
#ifndef DMX_H
#define DMX_H

#include "mbed.h"
#include "RawSerial.h"

//#define DMX_UART_DIRECT

#define DMX_TIME_BREAK 100 // 100us (88us-1s)
#define DMX_TIME_MAB 10 // 10us (8us-1s)
#define DMX_TIME_BETWEEN 200 // 10us (0-1s)
#define DMX_START_CODE 0

enum DMX_MODE
{
    DMX_MODE_BEGIN,
    DMX_MODE_START,
    DMX_MODE_BREAK,
    DMX_MODE_MAB,
    DMX_MODE_DATA,
    DMX_MODE_ERROR,
    DMX_MODE_STOP,
};

/** DMX512 class (sender/client)
 */
class DMX 
{
public:
    /** init DMX class
     * @param p_tx TX serial port (PC_2, PA_2)
     * @param p_rx RX serial port (PC_2, PA_2)
     * @param size dmx size, default = 512
     */
    DMX(PinName p_tx, PinName p_rx, unsigned int size = 512);
    ~DMX();

    /** Send the data
     * @param addr DMX data address (0-511)
     * @param data DMX data (0-255)
     */
    void put(int addr, char data);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void put(char *buf, int addr = 0, int len = 0);

    /** Send the data
     * @param addr DMX data address (0-511)
     * @return DMX data (0-255) or -1 on error
     */
    int get(int addr);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void get(char *buf, int addr = 0, int len = 0);

    /** Start DMX send operation
     */
    void start();
    /** Stop DMX send operation
     */
    void stop();
    /** Clear DMX data
     */
    void clear();

    volatile int is_recived;
    volatile int is_sent;

protected:

    void int_timer();
    void int_tx();
    void int_rx();

//    Serial _dmx;
    RawSerial _dmx;
    Timeout timeout01;
    volatile DMX_MODE mode_tx;
    volatile DMX_MODE mode_rx;
    volatile int addr_tx;
    volatile int addr_rx;
    char *data_tx;
    char *data_rx;

private:
    unsigned int myDMXSize;
    // struct recreated from LPC port -> must be converted to STM32 port
    typedef struct
    {
      uint8_t LCR;  // ?
      uint8_t LSR;  // flg
      uint8_t RBR;  // read port
      uint8_t THR;  // ?
    }DUMMY_UART_TypeDef;
    
    DUMMY_UART_TypeDef *_uart;
};

#endif