#ifndef MDEBUGGER_H
#define MDEBUGGER_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <cmsis.h>

#define MDEBUGGER_CIRCULAR_BUFFER_SIZE_IN_BYTES     32  
#define MDEBUGGER_NUMBER_OF_ADDRESS_CHANNELS        2

/* @brief Function which user has to provide to put a character to the stream associated with given channel number.

    To make the MDebugger working this function has to be provided by the user. 
    It will be called every time the debug library wants to send a character from the message which has to be passed.
    
    @param[in] data_byte A characted byte which has to be sent.
    @param[in] channel Channel number where the character byte has to be sent.
    
*/
void mdebug_putc(unsigned char data_byte, uint32_t channel);


/*
    @brief Function which writes data to the output buffer of the MDebugger library.
    
    This function should be used if the user wants to send a binary data to the debug channel.
    Saying more precisely - to send data which can contain NULL character.
    This function is not parsing anything and is fastest than mdebug_printf version.
    Underneath this function there is a circular buffer which is used in the first place to store the data.
    If the buffer will fill to the maximum during copying data the mdebug_flush function will be called to write the data to the stream.
    
    @param[in] in_buffer Pointer to buffer which contains data to be send.
    @param[in] in_buffer_size Size of data in the passed buffer.
    @param[in] channel Channel number where the data has to be sent.
*/
void mdebug_write(unsigned char *in_buffer, uint32_t in_buffer_size, uint32_t channel);

/*
    @brief Function which writes data straight to the output stream of the MDebugger library.
    
    This function should be used if the user wants to send a binary data to the debug channel.
    Saying more precisely - to send data which can contain NULL character.
    This function is not parsing anything and is fastest than mdebug_printf version.
    This is an unbuffered version of mdebug_write funtion which means that there is no circular buffer underneath it.
    This function will be passing the data straight to the stream.
    
    @param[in] in_buffer Pointer to buffer which contains data to be send.
    @param[in] in_buffer_size Size of data in the passed buffer.
    @param[in] channel Channel number where the data has to be sent.
*/
void mdebug_write_unbuffered(unsigned char *in_buffer, uint32_t in_buffer_size, uint32_t channel);

/*
    @brief Function which parses the formatting string and after that writes it to the output buffer of the MDebugger library.
    
    This function should be used if the user wants to send a string message to the debug channel.
    This function parses the formatting string together with passed arguments fimilary to printf function.
    Because of parsing the string function is slower than the mdebug_write.
    Underneath this function there is a circular buffer which is used in the first place to store the data.
    If the buffer will fill to the maximum during copying data the mdebug_flush function will be called to write the data to the stream.
    
    @param[in] channel Channel number where the data has to be sent.
    @param[in] formatting_string Formatting string which will be used to construct the message together with passed arguments after it.
*/
void mdebug_printf(uint32_t channel, const char *formatting_string, ...);

/*
    @brief Function which parses the formatting string and after that writes data straight to the output stream of the MDebugger library.
    
    This function should be used if the user wants to send a string message to the debug channel.
    This function parses the formatting string together with passed arguments fimilary to printf function.
    Because of parsing the string function is slower than the mdebug_write.
    This is an unbuffered version of mdebug_printf funtion which means that there is no circular buffer underneath it.
    This function will be passing the data straight to the stream.
    
    @param[in] channel Channel number where the data has to be sent.
    @param[in] formatting_string Formatting string which will be used to construct the message together with passed arguments after it.
*/
void mdebug_printf_unbuffered(uint32_t channel, const char *formatting_string, ...);

/*
    @brief Function which flushes the circular buffer assigned to given channel (sends the data to the stream).
    
    @param[in] channel Channel number which circular buffer will be flushed.
*/
void mdebug_flush(uint32_t channel);

/*
    @brief Function which showes how much space is used within the circular buffer on a given channel.
    
    @param[in] channel Channel number to calculate used space of circular buffer.
    @returns Used space in circular buffer assigned to given channel in bytes.
*/
uint32_t mdebug_output_bufer_used_space(uint32_t channel);

/*
    @brief Function which showes how much space is free within the circular buffer on a given channel.
    
    @param[in] channel Channel number to calculate free space of circular buffer.
    @returns Free space in circular buffer assigned to given channel in bytes.
*/
uint32_t mdebug_output_buffer_free_space(uint32_t channel);


/*
    @brief Function which checks is the circular buffer of a given channel empty.
    
    @param[in] channel Channel number to check is it empty.
    @return 0 Circular buffer assigned to given channel is not empty.
    @return 1 Circular buffer assigned to given channel is empty.
*/
uint32_t mdebug_output_buffer_is_empty(uint32_t channel);

/*
    @brief Function which prints information about the Cortex core.
    
    This function prints to default channel 0 all information about the core at the stage of calling this function.
*/
void mdebug_print_system_info(void);

#ifdef __cplusplus
}
#endif

#endif // MDEBUGGER_H
