#include "mbed.h"
#include "pwm_tone.h"   /* Tone library for Buzzer Click */
#include "Hexi_OLED_SSD1351.h" /* Header file for OLED Display */
#include "string.h"
#include "Hexi_KW40Z.h"   /* Driver for button presses and BLE */
#include <math.h>       /* Used for ADC conversion algorithm */
#include "images.h"     /* BMPs of the images drawn to the OLED */

/*****Function Prototypes*****/
void readSensors();
void ledColor();
void led_out();
void buzzer();
void CalculatePPM();
void dataQueue();
void initModules();
void StartHaptic(void);
void StopHaptic(void const *n);
void txTask(void);

PwmOut Buzzer(PTA10);   /* Buzzer is attached to docking station port 1 */
AnalogIn AQSensor(PTB3);   /* Air Quality sensor is attached to docking station port 2 */
AnalogIn COSensor(PTB6);   /* Carbon Monoxide sensor is attached to docking station port 3 */

/* Instantiate the SSD1351 OLED Driver */ 
SSD1351 oled(PTB22,PTB21,PTC13,PTB20,PTE6, PTD15);

BusOut led(PTC8, PTD0, PTC9);   /* RGB Port Configurations */
DigitalOut haptic(PTB9);        /* Port Configuration for vibrations */
/* Define timer for haptic feedback */
RtosTimer hapticTimer(StopHaptic, osTimerOnce);

int screen = 1;         //detects which screen the user is currently on
bool is_drawn1 = 0;      //detects if the screen is already drawn to the screen
bool is_drawn2 = 0;
bool is_drawn3 = 0;
bool is_drawn4 = 0;
bool is_ble = 0;        //detects if the device is currently running BLE mode

/* Screens */
const uint8_t *image1;          //homescreen
    
const uint8_t *image2;         //air quality screen

const uint8_t *image3;         //co screen

const int green = 5,
          red = 6,          /* LED Colors */
          black = 7,
          yellow = 4;
          
bool is_sounding = false;       /* Set to high if the buzzer is going off */
          
int led_color;      /* Variable to hold the current LED color */

Ticker sensor_read,      /* Used for the read sensor subroutine call */
       buzzer_sound,     /* Used for the sound buzzer subroutine call */
       led_flash,        /* Used to determine the flash rate */
       ledcolor,        /* Used to determine the LED color */
       A2D_Convert,    /* Used to convert the analog signal to digital */
       push;            /* Used to push values into the data array */

float B_4 = 1000000/Ti4,    /* Tones that will play on the buzzer click */
      B_5 = 1000000/Ti5;
      
char text[20];  /* Static Text Buffer for OLED Display */
char AQ_level[20]; /* Dynamic Text Buffer for AQ total */
char CO_level[20]; /* Dynamic Text Buffer for CO total */
char average[20]; /* Dynamic Text Buffer for the Average total */
char total_total[20]; /* Dynamic Text Buffer for the Total total */
char ble_status[20];  /* Informs the user if BLE is activated or disabled */

/* Either unit16_t or double */
double     total,    /* Variable to store the total total */
           aq_ppm,   /* Variable used to store the AQ PPM */
           co_ppm,   /* Variable used to store the CO PPM */ 
           prev_value;  /* Used for ARC Calculation */
double ARC;
         
uint16_t aq_total, /* Variable to store the AQ total */
         co_total; /* Variable to store the CO total */  
         
/*Create a Thread to handle sending BLE Sensor Data */ 
Thread txThread;

/* Variables to handle the circular array algorithm */
const int SIZE = 500;
double dataSet [SIZE];
int pushIndex = 0;
bool calc = false;  
double avg = 0, sum = 0;     

/* Instantiate the Hexi KW40Z Driver (UART TX, UART RX) */ 
KW40Z kw40z_device(PTE24, PTE25);

/* Below are the functions to handle button presses and screens */
void ButtonRight(void)//CO screen button
{   
    StartHaptic();
    screen = 3;
}

void ButtonLeft(void)//AQ screen button
{
    StartHaptic();
    screen = 2;
}

void ButtonDown(void)//home screen button
{
    StartHaptic();
    screen = 1;
}

void ButtonUp(void) {//toggles bluetooth mode
    StartHaptic();
    screen = 4;
    kw40z_device.ToggleAdvertisementMode();
    is_ble = !is_ble;
}

void PassKey(void)
{
    StartHaptic();
    strcpy((char *) text,"PAIR CODE");
    oled.TextBox((uint8_t *)text,0,25,95,18);
  
    /* Display Bond Pass Key in a 95px by 18px textbox at x=0,y=40 */
    sprintf(text,"%d", kw40z_device.GetPassKey());
    oled.TextBox((uint8_t *)text,0,40,95,18);
} 
      
int main(){
    
    initModules();
    
    /* Subroutine Calls */
    sensor_read.attach(&readSensors, 1);   /* Read the sensor on a time interval */
    A2D_Convert.attach(&CalculatePPM, 1);  /* Convert the values read from the sensors to floating point ppm values */
    push.attach(&dataQueue, 1);            /* Push the value into the set and compute the average */
    ledcolor.attach(&ledColor, 0.5);       /* Determine the LED color */
    led_flash.attach(&led_out, 0.25);      /* Flash LED based on sensor data */
    buzzer_sound.attach(&buzzer, 0.25);    /* Sensor values are sent to buzzer function */
    
    /* Register callbacks to button presses */
    kw40z_device.attach_buttonDown(&ButtonDown);
    kw40z_device.attach_buttonLeft(&ButtonLeft);
    kw40z_device.attach_buttonRight(&ButtonRight);
    kw40z_device.attach_buttonUp(&ButtonUp);
    kw40z_device.attach_passkey(&PassKey);
    
    txThread.start(txTask); /*Start transmitting Sensor Tag Data */
    
    while (1) { /* Loop to process and display data to the OLED */
    
        /* Get OLED Class Default Text Properties */
        oled_text_properties_t textProperties = {0};
        oled.GetTextProperties(&textProperties);
        
        /* Set text properties to green and right aligned for the dynamic text */
        textProperties.fontColor = COLOR_GREEN;
        textProperties.alignParam = OLED_TEXT_ALIGN_RIGHT;
        oled.SetTextProperties(&textProperties);
        
        if (screen == 2){   //Air Quality Screen
            if(!is_drawn2){
                is_drawn1 = 0;
                is_drawn3 = 0;
                is_drawn4 = 0;
                oled.DrawImage(image2,0,0);
                is_drawn2 = 1;
            }
            if (aq_ppm >= 50 and aq_ppm < 100){
                textProperties.fontColor = COLOR_YELLOW;  //color the font yellow
                oled.SetTextProperties(&textProperties);
            }
            else if (aq_ppm >= 100){
                textProperties.fontColor = COLOR_RED;   //color the font red
                oled.SetTextProperties(&textProperties);
            }
            sprintf(AQ_level,"%.2f",aq_ppm);    /* Print the AQ PPM to the screen */
            oled.TextBox((uint8_t *)AQ_level,35,76,35,15);
        }
        else if (screen == 3){  //Carbon Monoxide Screen
            if (!is_drawn3){
                is_drawn2 = 0;
                is_drawn1 = 0;
                is_drawn4 = 0;
                oled.DrawImage(image3,0,0);
                is_drawn3 = 1;
            }
            if (co_ppm >= 50 and co_ppm < 100){
                textProperties.fontColor = COLOR_YELLOW;  //color the font yellow
                oled.SetTextProperties(&textProperties);
            }
            
            else if (co_ppm >= 100) {
                textProperties.fontColor = COLOR_RED;  //color the font red
                oled.SetTextProperties(&textProperties);
            }
            sprintf(CO_level,"%.2f",co_ppm);    /* Print the CO PPM to the screen */        
            oled.TextBox((uint8_t *)CO_level,35,76,35,15);
            
        }
        else if (screen == 1) {   //Home Screen
            if (!is_drawn1){
                is_drawn3 = 0;
                is_drawn2 = 0;
                is_drawn4 = 0;
                oled.DrawImage(image1,0,0);
                is_drawn1 = 1;
            }
            if (avg >= 50 and avg <100) {
                textProperties.fontColor = COLOR_YELLOW;  //color the font yellow
                oled.SetTextProperties(&textProperties);
            }
            else if (avg >=100){
                textProperties.fontColor = COLOR_RED;  //color the font red
                oled.SetTextProperties(&textProperties); 
            }
            
            sprintf(average, "%.2f", avg);
            oled.TextBox((uint8_t *)average,53,23,35,15);
            
            if (total >=50 and total < 100){
                textProperties.fontColor = COLOR_YELLOW;  //color the font yellow
                oled.SetTextProperties(&textProperties); 
            }
            else if (total >= 100) {
                textProperties.fontColor = COLOR_RED;  //color the font red
                oled.SetTextProperties(&textProperties);
            }
            else {
                textProperties.fontColor = COLOR_GREEN; //color the font green
                oled.SetTextProperties(&textProperties);
            }
            sprintf(total_total, "%.2f", total);   /* Print the total to the screen */
            oled.TextBox((uint8_t *)total_total,53,39,35,15);
        }
        
        else if (screen == 4){   //BLE Pairing Screen
            if (!is_drawn4){
                is_drawn3 = 0;
                is_drawn2 = 0;
                is_drawn1 = 0; 
                oled.FillScreen(COLOR_BLACK);
                is_drawn4 = 1;  
            }
            textProperties.alignParam = OLED_TEXT_ALIGN_CENTER;
            textProperties.fontColor = COLOR_BLUE;
            oled.SetTextProperties(&textProperties);
            strcpy((char *) text,"BLUETOOTH");
            oled.TextBox((uint8_t *)text,0,25,95,18);  
            if (is_ble) {
                textProperties.fontColor = COLOR_GREEN;
                oled.SetTextProperties(&textProperties);
                strcpy((char *) ble_status, "On");
                oled.TextBox((uint8_t *)ble_status,0,55,95,18);
            }
            else {
                textProperties.fontColor = COLOR_RED;
                oled.SetTextProperties(&textProperties);
                strcpy((char *) ble_status, "Off");
                oled.TextBox((uint8_t *)ble_status,0,55,95,18);
            }
        }
        Thread::wait(500);
    }
    
    return 0;
}
void txTask(void){
    while (1){
        /*Notify Hexiwear App that it is running Sensor Tag mode*/
        kw40z_device.SendSetApplicationMode(GUI_CURRENT_APP_SENSOR_TAG);   
        kw40z_device.SendTemperature(100*aq_ppm); //send ppm Air Quality Click value
        kw40z_device.SendHumidity(100*co_ppm);  //send ppm CO click value
        kw40z_device.SendPressure(100*total);     //send total
        Thread::wait(1000);
    }
}
void initModules() {    /* Function to initialize the system */

    /* Turns on the OLED Display*/
    oled.PowerON();
    
    /* Sets the pointers to their respective images */
    image1 = homescreen_bmp;
    image2 = airquality_bmp;
    image3  = co_bmp;
}

void readSensors(){  /* Function to read sensors */
    
    /* Grab the analog signal from the sensors as 16 bit unsigned integers */
    aq_total = AQSensor.read_u16();
    co_total = COSensor.read_u16();
    /* Grab the analog signal from the sensors as floats */
    //aq_total = AQSensor.read();
    //co_total = AQSensor.read();
  
}

void ledColor(){    /* Function to determine the LED color */
    
    if ((total - avg) <= 10 and total < 50 and ARC < 10 ) {
        led = green;
        led_color = led;     /* Store the LED color for the flash function */
    }
    else if ((total - avg) > 10 and (total - avg) < 50 and total < 100 or (ARC >= 10 and ARC < 20)) {
        led = yellow;
        led_color = led;     /* Store the LED color for the flash function */
    }
    else if ((total - avg >= 50) or total >= 100 or avg >= 100 or ARC >= 20) {
        led = red;
        led_color = led;     /* Store the LED color for the flash function */
    }
}

void led_out() { /* Function to flash LEDs */

    if (led == green) {return;}   /* If green, don't blink */
    else if (led == black) {led = led_color;}
    else {led = black;}
     
}

void buzzer() { /* Function to handle the buzzer sound */

    if ((total - avg) <=25 and ARC < 10){            /* No buzzer sound if PPM is under 10 */
        Buzzer.period_us(0);
        is_sounding = false;  
    }

    else if ((total - avg) > 25 and (total - avg) < 50 and total < 100 or (ARC >=10 and ARC < 20)) {
        if (is_sounding == false){      /* If the buzzer is not triggered, trigger */
            Buzzer.period_us(B_4);      /* Period of B4 */
            Buzzer.write(0.50f);        /* 50% Duty Cycle on the Tone */
            is_sounding = true;         /* Signal that the buzzer is currently triggering */
        }
        else {   
            Buzzer.period_us(0);        /* Turn off the buzzer (Period = 0us)*/
            is_sounding = false;        /* Signal that the buzzer is no longer triggering */
        }        
    }
    
    else if ((total - avg) >= 50 or total >= 100 or avg >= 100 or ARC >= 20){
        if (is_sounding == false){
            Buzzer.period_us(B_5);      /* Period of B5 */
            Buzzer.write(0.50f);        /* 50% Duty Cycle on the Tone */
            is_sounding = true;
        }
        else {   
            Buzzer.period_us(0);        /* Turn off the buzzer */
            is_sounding = false;
        }
    }   
}

void CalculatePPM(){ /* Function to convert the analog signals to digital based on 16 bit adc steps */
    
    const double AQ_Rl = 20000.0;               // AQ_Rl (20kOhm) - Load resistance for AQ sensor
    const double CO_Rl = 18500.0;               // CO_Rl (18.5kOhm) - Load resistance for CO sensor
    const double Vadc_33 = 0.0000503548;         // ADC step 3,3V/65535 0.00503mV (16bit ADC)
    //const double Vadc_5 = 5.0/65535;          // ADC step 5.0V/2^16 (65535, 16bit ADC)
    double Vrl;                                  // Output voltage
    double Rs;                                   // Rs (Ohm) - Sensor resistance
    double ratio;                                // Rs/Rl ratio
    double lgPPM;

    if (aq_total > 65533)       //prevents NAN error from overflow of 16 bits
        aq_total = 65530;
    Vrl = (double)aq_total * Vadc_33;            // For 3.3V Vcc use Vadc_33
    Rs = AQ_Rl * (3.3 - Vrl)/Vrl;                 // Calculate sensor resistance
    ratio = Rs/AQ_Rl;                             // Calculate ratio
    lgPPM = (log10(ratio) * -0.8) + 0.9;      
    aq_ppm = 6* pow(10,lgPPM) - 12;                 // Calculate air quality ppm
    
    if (co_total > 65533)   //prevents NAN error from overflow of 16 bits
        co_total = 65530;
    Vrl = (double)co_total * Vadc_33;            // For 3.3V Vcc use Vadc_33
    Rs = CO_Rl * (3.3 - Vrl)/Vrl;                 // Calculate sensor resistance
    ratio = Rs/CO_Rl;                             // Calculate ratio
    lgPPM = (log10(ratio) * -0.8) + 0.9;       
    co_ppm = 6* pow(10,lgPPM) - 12;                 // Calculate carbon monoxide ppm
    
    if (aq_ppm < 0)
        aq_ppm = 0;             //prevents underflow
    else if (aq_ppm >= 1000)
        aq_ppm = 999.99;        //prevents overflow
    
    if (co_ppm <0)
        co_ppm = 0;
    else if (co_ppm >= 1000)
        co_ppm = 999.99;
      
    /* Calculate the total */
    total = co_ppm + aq_ppm;    
    if (total < 0)
        total = 0;
    else if (total >= 1000)
        total = 999.99; 
}

void dataQueue()
{   /* Beginning of function */
    if (pushIndex != SIZE and !calc){    /* Initially pushing values into the set */
        dataSet[pushIndex] = total;          //push value into the queue
        sum += dataSet[pushIndex];  //add the value to the sum
        pushIndex++;                    //increment the push index
        
        if (pushIndex == SIZE){
            avg = sum / SIZE;       //compute the average once the queue is full
            calc = true;                //flag calc
            ARC = dataSet[SIZE - 1] - dataSet[SIZE - 2];
            pushIndex = 0;              //reset the push index back to 0
        }
    }
    
    else if (pushIndex != SIZE and calc){   /* Pushing values into the set once the set is full */
        sum -= dataSet[pushIndex];          //subtract the value to be overriden from the sum
        dataSet[pushIndex] = total;           //push the value into the set
        sum += dataSet[pushIndex];          //add the value to the sum
        avg = sum / SIZE;                   //compute average
        if (pushIndex == 0){
            prev_value = dataSet[SIZE - 1];
            ARC = dataSet[pushIndex] - prev_value;
        }
        else {
            prev_value = dataSet[pushIndex - 1];
            ARC = dataSet[pushIndex] - prev_value;
        }
        pushIndex++;                        //increment the push index
        
        if (pushIndex == SIZE)
            pushIndex = 0;    //reset the push index back to 0
    }

} /* End of function */

void StartHaptic(void)
{
    hapticTimer.start(50);
    haptic = 1;
}

void StopHaptic(void const *n) {
    haptic = 0;
    hapticTimer.stop();
}