#include "mbed.h"
#include "DS1820.h"
#include "SHT21_ncleee.h"
#include "Adafruit_TCS34725.h"
#include "ssd1306.h"

#define DUREE_MESURE 10         // Durée en seconde entre deux mesures
#define DUREE_ECRAN_ON 30       // Durée en seconde d'éveil de l'écran

Serial pc(SERIAL_TX, SERIAL_RX);
Serial nucleo(D1,D0);

I2C i2c(D4, D5);
SSD1306 oled(D12, A6);

Ticker timeScreen;
Ticker capture;

// capteur temperature sol
DS1820 DS(D3); 

// capteur humidité sol
AnalogIn capteur_humidity_sol(A0); 

// capteur humidité + température air
SHT21 sht(&i2c);

// capteur RGB
Adafruit_TCS34725 RGBsens = Adafruit_TCS34725(&i2c, TCS34725_INTEGRATIONTIME_154MS, TCS34725_GAIN_1X);

// capteur lumière
AnalogIn ain(A1); 

//Interruption Bouton
InterruptIn bouton(D10);

// Flag OLED on
bool oled_on = 0;

// Définition de fonctions
float   temp_sol(void);
int     fct_humidity_sol(void);
void    fct_RGB(void);
void    sendDataSigfox(void);
void    oledData(void);
void    readData(void);
void    interruption_bouton(void);
void    turnOffScreen(void);

float temperature_sol;
unsigned char humidity_sol;
float temperature_air;
unsigned char humidity_air;
unsigned char pr, pg, pb;
unsigned short lum;
    
    
int main() {
    // Affichage logo pour initialisation
    oled.off();
    oled.on();
    oled.init();
    oled.cls(0,1);
    oled.locate(4,4);
    oled.printf("2PA2S");
    oled.redraw();
    wait(1);
    oled.cls();
    oled.off();
    // Initialisation des mesures 
    capture.attach(&readData,DUREE_MESURE);    
    
    //Initialisation de l'interruption
    bouton.fall(interruption_bouton);
    readData();
    
    while(1) {
    }
}


float temp_sol()
{
    DS.convertTemperature(true, DS1820::all_devices);
    if (DS.unassignedProbe(D3)){
        pc.printf( "D3 not assigned\n\r");
    }
    return DS.temperature();
}

int fct_humidity_sol(void)
{
    float val_min = 0.377;
    float val_max = 0.772;
    float mesure, mesure_etalonnee;
    mesure = capteur_humidity_sol.read();
    mesure_etalonnee = (1-((mesure - val_min)/(val_max - val_min)))*100;
    return (int) mesure_etalonnee;
}

void fct_RGB(void)
{
    int somme;
    uint16_t clear, red, green, blue;
    if (!RGBsens.begin())
    {
        pc.printf("No TCS34725 found ... check your connections");
        //while (1); // halt!
    }
    RGBsens.getRawData(&red, &green, &blue, &clear);
    somme = red + green + blue;
    pr = red*100/somme;
    pg = green*100/somme;
    pb = blue*100/somme;
    lum = clear;
}

void sendDataSigfox(void){
    short tempSol_short, tempAir_short;
    tempSol_short = (short)(temperature_sol*10);
    tempAir_short = (short)(temperature_air*10);

    nucleo.printf("AT$SF=%04x%02x%04x%02x%04x%02x%02x%02x\r\n",tempSol_short, humidity_sol, tempAir_short, humidity_air, lum, pr, pg, pb);
}
    
void oledData(void){
    if(!oled_on){
        pc.printf("Turning on screen\n\r");
        oled.on();
        oled.speed (SSD1306::Medium);
        oled.init();
        oled.set_contrast(200);
        oled_on = 1;
    }
    pc.printf("Displaying data\n\r");
    oled.cls(0,1);
    oled.locate(0,0);
    oled.printf("AIR T : %.1f", temperature_air);
    oled.locate(1,0);
    oled.printf("AIR H : %d", humidity_air);
    oled.locate(3,0);
    oled.printf("FLOOR T : %.1f", temperature_sol);
    oled.locate(4,0);
    oled.printf("FLOOR H : %d", humidity_sol);
    oled.locate(6,0);
    oled.printf("Light : %d", lum);
    oled.locate(7,0);
    oled.printf("R %d G %d B %d", pr, pg, pb);
    oled.redraw();
}

void readData(void){
    pc.printf("reading data\n\r");
    temperature_sol = temp_sol();
    humidity_sol = fct_humidity_sol();
    temperature_air = sht.readTemp();
    humidity_air = sht.readHumidity();
    fct_RGB();
    sendDataSigfox();
    if(oled_on)
        oledData();
}

void interruption_bouton(){
    bouton.disable_irq();
    if(!oled_on){
        pc.printf("Interruption avec ecran eteint\n\r");
        oledData();
        timeScreen.attach(&turnOffScreen,DUREE_ECRAN_ON);
    }
    else{
        pc.printf("Interruption avec ecran allume\n\r");
        readData();
    }
    bouton.enable_irq();
}

void turnOffScreen(void){
    pc.printf("Extinction ecran\n\r");
    timeScreen.detach();
    oled_on = 0;
    oled.off();
}
