/* mbed library for 240*320 pixel display TFT based on ILI9341 LCD Controller
 * Copyright (c) 2013 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
// 12.06.13 fork from SPI_TFT code because controller is different ...
// 14.07.13 Test with real display and bugfix 
// 18.10.13 Better Circle function from Michael Ammann
// 22.10.13 Fixes for Kinetis Board - 8 bit spi
// 26.01.14 Change interface for BMP_16 to also use SD-cards
// 01-09-14 Geremia:
// using always 16bit spi. Now 8bits are wasted every cmd, but slower cpus takes more time to send 3x8bits than 2x16bits with 8bit wasted
// added read stuff, like read ID, read status, read pixel

#include "SPI_TFT_ILI9341.h"
#include "mbed.h"

#define BPP         16                  // Bits per pixel    

// if display is the only slave into the SPI bus, we can skip driving CS at every cmd
//#define USE_CS  

SPI_TFT_ILI9341::SPI_TFT_ILI9341(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset, PinName dc, const char *name)
    : _spi(mosi, miso, sclk), _cs(cs), _reset(reset), _dc(dc), GraphicsDisplay(name)
{
    clk = sclk;
    orientation = 0;
    char_x = 0;
    _reset = 0;
    _spi.format(16,0);                  // 16 bit spi mode 0
    _spi.frequency(10000000);          // 10 Mhz SPI clock
    tft_reset();
}

int SPI_TFT_ILI9341::width()
{
    if (orientation == 0 || orientation == 2) return 240;
    else return 320;
}


int SPI_TFT_ILI9341::height()
{
    if (orientation == 0 || orientation == 2) return 320;
    else return 240;
}


void SPI_TFT_ILI9341::set_orientation(unsigned int o)
{
    orientation = o;
    wr_cmd(0x36);                     // MEMORY_ACCESS_CONTROL
    switch (orientation) {
        case 0:
            wr_8(0x48);
            break;
        case 1:
            wr_8(0x28);
            break;
        case 2:
            wr_8(0x88);
            break;
        case 3:
            wr_8(0xE8);
            break;
    }
#ifdef USE_CS
    _cs = 1;
#endif 
    WindowMax();
} 


// write command to tft register
void SPI_TFT_ILI9341::wr_cmd(unsigned char cmd)
{
    _dc = 0;
#ifdef USE_CS
    _cs = 0;
#endif
    _spi.write(cmd);      // write 16bit, 00 (NOP cmd) then 8bit cmd
    _dc = 1;
}

// write 8bit data/parameter
void SPI_TFT_ILI9341::wr_8(unsigned char value)
{
    _spi.format(8,0);   // changing format takes time but wr_8 is used only for init and set orientation
    _spi.write(value);
    _spi.format(16,0);
}

// write 16bit data/parameter
void SPI_TFT_ILI9341::wr_16(unsigned int value)
{
    _spi.write(value);
}

// the ILI9341 can read 
// for cmd that returs 8bits (dummy cycle not needed)
unsigned char SPI_TFT_ILI9341::wr_cmd_rd8(unsigned char cmd)
{
    unsigned char r;
    _dc = 0;
#ifdef USE_CS
    _cs = 0;
#endif
    _spi.write(cmd);      // mbed lib
 //   _dc = 1; //not needed for read cmds
    r = _spi.write(0) >> 8;
#ifdef USE_CS
    _cs = 1;
#endif    
    return(r);
}
// for cmd that return 24bits (need a dummy cycle after cmd)
unsigned int SPI_TFT_ILI9341::wr_cmd_rd24(unsigned char cmd)
{
    unsigned int d = 0;
    unsigned short r;
    _dc = 0;
#ifdef USE_CS
    _cs = 0;
#endif
    _spi.write(cmd); 
//    _dc = 1; //not needed for read cmds
    r = _spi.write(0); // we get only 15bit valid, first bit was the dummy cycle
    d = r;
    r = _spi.write(0);
    d = (d << 16) | r;
    d = d >> 7;
    // we clocked 7 more bit so ILI waiting for 8th, we need to reset spi bus with CS
    _cs = 1;
#ifndef USE_CS //if CS is not used, force fixed LOW again
    _cs = 0;
#endif    
    return(d);
}
// for cmd that return 32bits (need a dummy cycle after cmd)
unsigned int SPI_TFT_ILI9341::wr_cmd_rd32(unsigned char cmd)
{
    unsigned int d = 0;
    unsigned short r;
    _dc = 0;
#ifdef USE_CS
    _cs = 0;
#endif
    _spi.write(cmd); 
 //   _dc = 1; //not needed for read cmds
    r = _spi.write(0); // we get only 15bit valid, first bit was the dummy cycle
    d = r;
    r = _spi.write(0);
    d = (d << 16) | r;
    r = _spi.write(0);
    d = (d << 1) | (r >> 15);
    // we clocked 15 more bit so ILI waiting for 16th, we need to rest spi
    _cs = 1;
#ifndef USE_CS //if CS is not used, force fixed LOW again
    _cs = 0;
#endif  
    return(d);
}
// for read cmds that needs a dummy byte (not a dummy bit) like 2E ram read
unsigned int SPI_TFT_ILI9341::wr_cmd_rd32_dbyte(unsigned char cmd)
{
    unsigned int r;
    _dc = 0;
#ifdef USE_CS
    _cs = 0;
#endif
    _spi.write(cmd<<8); // send cmd then 00, cmd is processed immediatly, so the second byte will be the dummy byte
  // _dc = 1; //not needed for read cmds
    r = _spi.write(0)<<16; // Byte3,2
    r = r | _spi.write(0); // Byte1,0
    // now need to reset because some cmds would expect to xfer more data
    _cs = 1;
#ifndef USE_CS //if CS is not used, force fixed LOW again
    _cs = 0;
#endif    
    return(r);
}
// undocumented cmd 0xD9 found in some code example
unsigned char SPI_TFT_ILI9341::Read_Register(char Addr, char xParameter)
{
    char data=0;
    wr_cmd(0xd9); /* ext command */
    wr_8(0x10+xParameter); /* 0x11 is the first Parameter */
#ifdef USE_CS
    _cs = 1;
#endif
    data = wr_cmd_rd8(Addr);
    return data;
}
int  SPI_TFT_ILI9341::Read_ID4ext(void){
    char i;
    char r;
    int d=0;
    for(i=0;i<3;i++)
    {
        r=Read_Register(0xd3,i+1);  // D3 for ID4
        d = (d << 8) | r;
    }
    
    return(d);
}

// broken on 9341
int  SPI_TFT_ILI9341::Read_ID123(void){
    int r;
    r = wr_cmd_rd24(0x04);
    return(r);
}
// broken on 9341
int  SPI_TFT_ILI9341::Read_ID4(void){
    int r;
    r = wr_cmd_rd24(0xD3);  // BF for ili9481, give a try
    return(r);
}
// read status register
int  SPI_TFT_ILI9341::Read_STS(void){
    int r;
    r = wr_cmd_rd32(0x09);
    return(r);
}

// HW reset
void SPI_TFT_ILI9341::tft_reset()
{
//    _spi.format(8,0);                  // 8 bit spi mode 0
//    _spi.frequency(10000000);          // 10 Mhz SPI clock
    _cs = 1;                           // cs high
    _dc = 1;                           // dc high 
    
    _reset = 0;                        // display reset
    wait_ms(50);
    
    _reset = 1;                       // end hardware reset
    wait_ms(150);                       // or 5?!?
    
    _cs = 0;
    
 //   wr_cmd(0x01);                     // SW reset  
 //   wait_ms(150);                       // or 5?!?
    tft_init_redPCBtm22();
}


 // init for 2.2" red PCB 1580005661C with Tianma TM022HDH26 display
void SPI_TFT_ILI9341::tft_init_redPCBtm22()
{
    /* Start Initial Sequence ----------------------------------------------------*/
    wr_cmd(0xCB);  // POWER_ON_SEQ_CONTROL             
     wr_8(0x39);
     wr_8(0x2C);
     wr_8(0x00);
     wr_8(0x34);
     wr_8(0x02);
     
    wr_cmd(0xCF);  // POWER_CONTROL_B              
     wr_8(0x00);
     wr_8(0xC1);  // Applic Notes 81, was 83, C1 enables PCEQ: PC and EQ operation for power saving
     wr_8(0x30);
     
     wr_cmd(0xE8);  // DRIVER_TIMING_CONTROL_A               
     wr_8(0x85);
     wr_8(0x00);  // AN 10, was 01
     wr_8(0x78);  // AN 7A, was 79
     
     wr_cmd(0xEA);  // DRIVER_TIMING_CONTROL_B                    
     wr_8(0x00);
     wr_8(0x00);
     
     wr_cmd(0xED);                     
     wr_8(0x64);
     wr_8(0x03);
     wr_8(0x12);
     wr_8(0x81);
     
     wr_cmd(0xF7);  // PUMP_RATIO_CONTROL                   
     wr_8(0x20);
     
     wr_cmd(0xC0);                     // POWER_CONTROL_1
     wr_8(0x23);  // AN 21, was 26
     
     wr_cmd(0xC1);                     // POWER_CONTROL_2
     wr_8(0x10);  // AN 11, was 11
     
     wr_cmd(0xC5);                     // VCOM_CONTROL_1
     wr_8(0x3E);  // AN 3F, was 35
     wr_8(0x28);  // AN 3C, was 3E
     
     wr_cmd(0xC7);                     // VCOM_CONTROL_2
     wr_8(0x86);  // AN A7, was BE
     
     wr_cmd(0x36);                     // MEMORY_ACCESS_CONTROL
     wr_8(0x48);
     
     wr_cmd(0x3A);                     // COLMOD_PIXEL_FORMAT_SET, not present in AN
     wr_8(0x55);                 // 16 bit pixel 
     
     wr_cmd(0xB1);                     // Frame Rate
     wr_8(0x00);
     wr_8(0x18);  // AN 1B, was 1B  1B=70hz             
     
     wr_cmd(0xB6);                       // display function control, INTERESTING
     wr_8(0x08);  // AN 0A, was 0A
     wr_8(0x82);  // AN A2
     wr_8(0x27);  // AN not present
  //   wr_8(0x00);  // was present
     
     wr_cmd(0xF2);                     // Gamma Function Disable
     wr_8(0x00);  // AN 00, was 08
     
     wr_cmd(0x26);                     
     wr_8(0x01);                 // gamma set for curve 01/2/04/08
     
     wr_cmd(0xE0);                     // positive gamma correction
     wr_8(0x0F); 
     wr_8(0x31); 
     wr_8(0x2B); 
     wr_8(0x0C); 
     wr_8(0x0E); 
     wr_8(0x08); 
     wr_8(0x4E); 
     wr_8(0xF1); 
     wr_8(0x37); 
     wr_8(0x07); 
     wr_8(0x10); 
     wr_8(0x03); 
     wr_8(0x0E);
     wr_8(0x09); 
     wr_8(0x00);
     
     wr_cmd(0xE1);                     // negativ gamma correction
     wr_8(0x00); 
     wr_8(0x0E); 
     wr_8(0x14); 
     wr_8(0x03); 
     wr_8(0x11); 
     wr_8(0x07); 
     wr_8(0x31); 
     wr_8(0xC1); 
     wr_8(0x48); 
     wr_8(0x08); 
     wr_8(0x0F); 
     wr_8(0x0C); 
     wr_8(0x31);
     wr_8(0x36); 
     wr_8(0x0F);
     
     //wr_cmd(0x34);                     // tearing effect off
     
     //wr_cmd(0x35);                     // tearing effect on
      
  //   wr_cmd(0xB7);                       // ENTRY_MODE_SET
  //   wr_8(0x07);
         
     wr_cmd(0x11);                     // sleep out
     
     wait_ms(150);
     
     wr_cmd(0x29);                     // display on
     
     wait_ms(150);
#ifdef USE_CS
    _cs = 1;
#endif
     
     WindowMax ();
     
 }
void SPI_TFT_ILI9341::pixel(int x, int y, int color)
{
    wr_cmd(0x2A);
    wr_16(x);   // set only start column, end colum set by window() funct
    wr_cmd(0x2B); // set only start page
    wr_16(y);
    wr_cmd(0x2C);  // send pixel
    wr_16(color);
#ifdef USE_CS
    _cs = 1;
#endif
}
unsigned short SPI_TFT_ILI9341::pixelread(int x, int y)
{
    unsigned int color;
    wr_cmd(0x2A);
    wr_16(x);   // set only start column, end colum set by window() funct
    wr_cmd(0x2B); // set only start page
    wr_16(y); 
#ifdef USE_CS
    _cs = 1;
#endif
    color= wr_cmd_rd32_dbyte(0x2E)>>8;  // read ram, returns 18bit pixel, rrrrrr00_gggggg00_bbbbbb00_nexpixelred
    color= RGB((color&0xFC0000)>>16, (color&0xFC00)>>8, color&0xFC);
    return (unsigned short)color;
}

void SPI_TFT_ILI9341::window (unsigned int x, unsigned int y, unsigned int w, unsigned int h)
{
    wr_cmd(0x2A);
    wr_16(x);
    wr_16(x+w-1);

    wr_cmd(0x2B);
    wr_16(y);
    wr_16(y+h-1);
#ifdef USE_CS
    _cs = 1;
#endif
}


void SPI_TFT_ILI9341::WindowMax (void)
{
    window (0, 0, width(),  height());
}



void SPI_TFT_ILI9341::cls (void)
{
    int pixels = ( width() * height());
    WindowMax();
    wr_cmd(0x2C);  // send pixel
    for (int i = 0; i < pixels; i++)
        {
        wr_16(_background);
        }
#ifdef USE_CS
    _cs = 1;
#endif
}


void SPI_TFT_ILI9341::circle(int x0, int y0, int r, int color)
{

    int x = -r, y = 0, err = 2-2*r, e2;
    do {
        pixel(x0-x, y0+y,color);
        pixel(x0+x, y0+y,color);
        pixel(x0+x, y0-y,color);
        pixel(x0-x, y0-y,color);
        e2 = err;
        if (e2 <= y) {
            err += ++y*2+1;
            if (-x == y && e2 <= x) e2 = 0;
        }
        if (e2 > x) err += ++x*2+1;
    } while (x <= 0);

}

void SPI_TFT_ILI9341::fillcircle(int x0, int y0, int r, int color)
{
    int x = -r, y = 0, err = 2-2*r, e2;
    do {
        vline(x0-x, y0-y, y0+y, color);
        vline(x0+x, y0-y, y0+y, color);
        e2 = err;
        if (e2 <= y) {
            err += ++y*2+1;
            if (-x == y && e2 <= x) e2 = 0;
        }
        if (e2 > x) err += ++x*2+1;
    } while (x <= 0);
}


void SPI_TFT_ILI9341::hline(int x0, int x1, int y, int color)
{
    int w;
    w = x1 - x0 + 1;
    window(x0,y,w,1);
    wr_cmd(0x2C);  // send pixel
    for (int j=0; j<w; j++) {
        wr_16(color);
    }
    WindowMax();
    return;
}

void SPI_TFT_ILI9341::vline(int x, int y0, int y1, int color)
{
    int h;
    h = y1 - y0 + 1;
    window(x,y0,1,h);
    wr_cmd(0x2C);  // send pixel
    for (int y=0; y<h; y++) {
        wr_16(color);
    }
    WindowMax();
    return;
}



void SPI_TFT_ILI9341::line(int x0, int y0, int x1, int y1, int color)
{
    //WindowMax();
    int   dx = 0, dy = 0;
    int   dx_sym = 0, dy_sym = 0;
    int   dx_x2 = 0, dy_x2 = 0;
    int   di = 0;

    dx = x1-x0;
    dy = y1-y0;

    if (dx == 0) {        /* vertical line */
        if (y1 > y0) vline(x0,y0,y1,color);
        else vline(x0,y1,y0,color);
        return;
    }

    if (dx > 0) {
        dx_sym = 1;
    } else {
        dx_sym = -1;
    }
    if (dy == 0) {        /* horizontal line */
        if (x1 > x0) hline(x0,x1,y0,color);
        else  hline(x1,x0,y0,color);
        return;
    }

    if (dy > 0) {
        dy_sym = 1;
    } else {
        dy_sym = -1;
    }

    dx = dx_sym*dx;
    dy = dy_sym*dy;

    dx_x2 = dx*2;
    dy_x2 = dy*2;

    if (dx >= dy) {
        di = dy_x2 - dx;
        while (x0 != x1) {

            pixel(x0, y0, color);
            x0 += dx_sym;
            if (di<0) {
                di += dy_x2;
            } else {
                di += dy_x2 - dx_x2;
                y0 += dy_sym;
            }
        }
        pixel(x0, y0, color);
    } else {
        di = dx_x2 - dy;
        while (y0 != y1) {
            pixel(x0, y0, color);
            y0 += dy_sym;
            if (di < 0) {
                di += dx_x2;
            } else {
                di += dx_x2 - dy_x2;
                x0 += dx_sym;
            }
        }
        pixel(x0, y0, color);
    }
    return;
}


void SPI_TFT_ILI9341::rect(int x0, int y0, int x1, int y1, int color)
{

    if (x1 > x0) hline(x0,x1,y0,color);
    else  hline(x1,x0,y0,color);

    if (y1 > y0) vline(x0,y0,y1,color);
    else vline(x0,y1,y0,color);

    if (x1 > x0) hline(x0,x1,y1,color);
    else  hline(x1,x0,y1,color);

    if (y1 > y0) vline(x1,y0,y1,color);
    else vline(x1,y1,y0,color);

    return;
}



void SPI_TFT_ILI9341::fillrect(int x0, int y0, int x1, int y1, int color)
{
    int tmp;
    if(x0 > x1)
    {
        tmp=x0; // swap them
        x0=x1;
        x1=tmp;
    }
    if(y0 > y1)
    {
        tmp=y0; // swap them
        y0=y1;
        y1=tmp;
    }
        
    int h = y1 - y0 + 1;
    int w = x1 - x0 + 1;
    int pixels = h * w;
    window(x0,y0,w,h);
    wr_cmd(0x2C);  // send pixel 
    for (int p=0; p<pixels; p++) {
        wr_16(color);
    }
    WindowMax();
    return;
}


void SPI_TFT_ILI9341::locate(int x, int y)
{
    char_x = x;
    char_y = y;
}



int SPI_TFT_ILI9341::columns()
{
    return width() / font[1];
}



int SPI_TFT_ILI9341::rows()
{
    return height() / font[2];
}



int SPI_TFT_ILI9341::_putc(int value)
{
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
    }
    return value;
}


void SPI_TFT_ILI9341::character(int x, int y, int c)
{
    unsigned int hor,vert,offset,bpl,j,i,b;
    unsigned char* zeichen;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }
    window(char_x, char_y,hor,vert); // char box
    wr_cmd(0x2C);  // send pixel
    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
     for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) { 
                wr_16(_background);
            } else {
                wr_16(_foreground);
            }
        }
    }
    WindowMax();    // maybe we can skip this
    if ((w + 2) < hor) {                   // x offset to next char
        char_x += w + 2;
    } else char_x += hor;
}


void SPI_TFT_ILI9341::set_font(unsigned char* f)
{
    font = f;
}



void SPI_TFT_ILI9341::Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h,unsigned char *bitmap)
{
    unsigned int  j;
    int padd;
    unsigned short *bitmap_ptr = (unsigned short *)bitmap;    
    unsigned int i;
    
    // the lines are padded to multiple of 4 bytes in a bitmap
    padd = -1;
    do {
        padd ++;
    } while (2*(w + padd)%4 != 0);
    window(x, y, w, h);
    bitmap_ptr += ((h - 1)* (w + padd));
    wr_cmd(0x2C);  // send pixel
    for (j = 0; j < h; j++) {         //Lines
        for (i = 0; i < w; i++) {     // one line
                wr_16(*bitmap_ptr);
                bitmap_ptr++;
        }
        bitmap_ptr -= 2*w;
        bitmap_ptr -= padd;
    }
    WindowMax();
}


// local filesystem is not implemented in kinetis board , but you can add a SD card

int SPI_TFT_ILI9341::BMP_16(unsigned int x, unsigned int y, const char *Name_BMP)
{

#define OffsetPixelWidth    18
#define OffsetPixelHeigh    22
#define OffsetFileSize      34
#define OffsetPixData       10
#define OffsetBPP           28

    char filename[50];
    unsigned char BMP_Header[54];
    unsigned short BPP_t;
    unsigned int PixelWidth,PixelHeigh,start_data;
    unsigned int    i,off;
    int padd,j;
    unsigned short *line;

    // get the filename
    i=0;
    while (*Name_BMP!='\0') {
        filename[i++]=*Name_BMP++;
    }
    filename[i] = 0;  
    
    FILE *Image = fopen((const char *)&filename[0], "rb");  // open the bmp file
    if (!Image) {
        return(0);      // error file not found !
    }

    fread(&BMP_Header[0],1,54,Image);      // get the BMP Header

    if (BMP_Header[0] != 0x42 || BMP_Header[1] != 0x4D) {  // check magic byte
        fclose(Image);
        return(-1);     // error no BMP file
    }

    BPP_t = BMP_Header[OffsetBPP] + (BMP_Header[OffsetBPP + 1] << 8);
    if (BPP_t != 0x0010) {
        fclose(Image);
        return(-2);     // error no 16 bit BMP
    }

    PixelHeigh = BMP_Header[OffsetPixelHeigh] + (BMP_Header[OffsetPixelHeigh + 1] << 8) + (BMP_Header[OffsetPixelHeigh + 2] << 16) + (BMP_Header[OffsetPixelHeigh + 3] << 24);
    PixelWidth = BMP_Header[OffsetPixelWidth] + (BMP_Header[OffsetPixelWidth + 1] << 8) + (BMP_Header[OffsetPixelWidth + 2] << 16) + (BMP_Header[OffsetPixelWidth + 3] << 24);
    if (PixelHeigh > height() + y || PixelWidth > width() + x) {
        fclose(Image);
        return(-3);      // to big
    }

    start_data = BMP_Header[OffsetPixData] + (BMP_Header[OffsetPixData + 1] << 8) + (BMP_Header[OffsetPixData + 2] << 16) + (BMP_Header[OffsetPixData + 3] << 24);

    line = (unsigned short *) malloc (2 * PixelWidth); // we need a buffer for a line
    if (line == NULL) {
        return(-4);         // error no memory
    }

    // the bmp lines are padded to multiple of 4 bytes
    padd = -1;
    do {
        padd ++;
    } while ((PixelWidth * 2 + padd)%4 != 0);

    window(x, y,PixelWidth ,PixelHeigh);
    wr_cmd(0x2C);  // send pixel
    for (j = PixelHeigh - 1; j >= 0; j--) {               //Lines bottom up
        off = j * (PixelWidth  * 2 + padd) + start_data;   // start of line
        fseek(Image, off ,SEEK_SET);
        fread(line,1,PixelWidth * 2,Image);       // read a line - slow 
        for (i = 0; i < PixelWidth; i++) {        // copy pixel data to TFT
            wr_16(line[i]);                  // one 16 bit pixel   
        } 
     }
    free (line);
    fclose(Image);
    WindowMax();
    return(1);
}
