#include "AES.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

static const uint8_t sbox[] =
{
   0x63, 0x7C, 0x77, 0x7B, 0xF2, 0x6B, 0x6F, 0xC5, 0x30, 0x01, 0x67, 0x2B, 0xFE, 0xD7, 0xAB, 0x76,
   0xCA, 0x82, 0xC9, 0x7D, 0xFA, 0x59, 0x47, 0xF0, 0xAD, 0xD4, 0xA2, 0xAF, 0x9C, 0xA4, 0x72, 0xC0,
   0xB7, 0xFD, 0x93, 0x26, 0x36, 0x3F, 0xF7, 0xCC, 0x34, 0xA5, 0xE5, 0xF1, 0x71, 0xD8, 0x31, 0x15,
   0x04, 0xC7, 0x23, 0xC3, 0x18, 0x96, 0x05, 0x9A, 0x07, 0x12, 0x80, 0xE2, 0xEB, 0x27, 0xB2, 0x75,
   0x09, 0x83, 0x2C, 0x1A, 0x1B, 0x6E, 0x5A, 0xA0, 0x52, 0x3B, 0xD6, 0xB3, 0x29, 0xE3, 0x2F, 0x84,
   0x53, 0xD1, 0x00, 0xED, 0x20, 0xFC, 0xB1, 0x5B, 0x6A, 0xCB, 0xBE, 0x39, 0x4A, 0x4C, 0x58, 0xCF,
   0xD0, 0xEF, 0xAA, 0xFB, 0x43, 0x4D, 0x33, 0x85, 0x45, 0xF9, 0x02, 0x7F, 0x50, 0x3C, 0x9F, 0xA8,
   0x51, 0xA3, 0x40, 0x8F, 0x92, 0x9D, 0x38, 0xF5, 0xBC, 0xB6, 0xDA, 0x21, 0x10, 0xFF, 0xF3, 0xD2,
   0xCD, 0x0C, 0x13, 0xEC, 0x5F, 0x97, 0x44, 0x17, 0xC4, 0xA7, 0x7E, 0x3D, 0x64, 0x5D, 0x19, 0x73,
   0x60, 0x81, 0x4F, 0xDC, 0x22, 0x2A, 0x90, 0x88, 0x46, 0xEE, 0xB8, 0x14, 0xDE, 0x5E, 0x0B, 0xDB,
   0xE0, 0x32, 0x3A, 0x0A, 0x49, 0x06, 0x24, 0x5C, 0xC2, 0xD3, 0xAC, 0x62, 0x91, 0x95, 0xE4, 0x79,
   0xE7, 0xC8, 0x37, 0x6D, 0x8D, 0xD5, 0x4E, 0xA9, 0x6C, 0x56, 0xF4, 0xEA, 0x65, 0x7A, 0xAE, 0x08,
   0xBA, 0x78, 0x25, 0x2E, 0x1C, 0xA6, 0xB4, 0xC6, 0xE8, 0xDD, 0x74, 0x1F, 0x4B, 0xBD, 0x8B, 0x8A,
   0x70, 0x3E, 0xB5, 0x66, 0x48, 0x03, 0xF6, 0x0E, 0x61, 0x35, 0x57, 0xB9, 0x86, 0xC1, 0x1D, 0x9E,
   0xE1, 0xF8, 0x98, 0x11, 0x69, 0xD9, 0x8E, 0x94, 0x9B, 0x1E, 0x87, 0xE9, 0xCE, 0x55, 0x28, 0xDF,
   0x8C, 0xA1, 0x89, 0x0D, 0xBF, 0xE6, 0x42, 0x68, 0x41, 0x99, 0x2D, 0x0F, 0xB0, 0x54, 0xBB, 0x16
};

static const uint8_t inv_s[] = 
{
   0x52, 0x09, 0x6A, 0xD5, 0x30, 0x36, 0xA5, 0x38, 0xBF, 0x40, 0xA3, 0x9E, 0x81, 0xF3, 0xD7, 0xFB,
   0x7C, 0xE3, 0x39, 0x82, 0x9B, 0x2F, 0xFF, 0x87, 0x34, 0x8E, 0x43, 0x44, 0xC4, 0xDE, 0xE9, 0xCB,
   0x54, 0x7B, 0x94, 0x32, 0xA6, 0xC2, 0x23, 0x3D, 0xEE, 0x4C, 0x95, 0x0B, 0x42, 0xFA, 0xC3, 0x4E,
   0x08, 0x2E, 0xA1, 0x66, 0x28, 0xD9, 0x24, 0xB2, 0x76, 0x5B, 0xA2, 0x49, 0x6D, 0x8B, 0xD1, 0x25,
   0x72, 0xF8, 0xF6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xD4, 0xA4, 0x5C, 0xCC, 0x5D, 0x65, 0xB6, 0x92,
   0x6C, 0x70, 0x48, 0x50, 0xFD, 0xED, 0xB9, 0xDA, 0x5E, 0x15, 0x46, 0x57, 0xA7, 0x8D, 0x9D, 0x84,
   0x90, 0xD8, 0xAB, 0x00, 0x8C, 0xBC, 0xD3, 0x0A, 0xF7, 0xE4, 0x58, 0x05, 0xB8, 0xB3, 0x45, 0x06,
   0xD0, 0x2C, 0x1E, 0x8F, 0xCA, 0x3F, 0x0F, 0x02, 0xC1, 0xAF, 0xBD, 0x03, 0x01, 0x13, 0x8A, 0x6B,
   0x3A, 0x91, 0x11, 0x41, 0x4F, 0x67, 0xDC, 0xEA, 0x97, 0xF2, 0xCF, 0xCE, 0xF0, 0xB4, 0xE6, 0x73,
   0x96, 0xAC, 0x74, 0x22, 0xE7, 0xAD, 0x35, 0x85, 0xE2, 0xF9, 0x37, 0xE8, 0x1C, 0x75, 0xDF, 0x6E,
   0x47, 0xF1, 0x1A, 0x71, 0x1D, 0x29, 0xC5, 0x89, 0x6F, 0xB7, 0x62, 0x0E, 0xAA, 0x18, 0xBE, 0x1B,
   0xFC, 0x56, 0x3E, 0x4B, 0xC6, 0xD2, 0x79, 0x20, 0x9A, 0xDB, 0xC0, 0xFE, 0x78, 0xCD, 0x5A, 0xF4,
   0x1F, 0xDD, 0xA8, 0x33, 0x88, 0x07, 0xC7, 0x31, 0xB1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xEC, 0x5F,
   0x60, 0x51, 0x7F, 0xA9, 0x19, 0xB5, 0x4A, 0x0D, 0x2D, 0xE5, 0x7A, 0x9F, 0x93, 0xC9, 0x9C, 0xEF,
   0xA0, 0xE0, 0x3B, 0x4D, 0xAE, 0x2A, 0xF5, 0xB0, 0xC8, 0xEB, 0xBB, 0x3C, 0x83, 0x53, 0x99, 0x61,
   0x17, 0x2B, 0x04, 0x7E, 0xBA, 0x77, 0xD6, 0x26, 0xE1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0C, 0x7D
};


static const uint32_t rcon[10]=
{
    0x01000000, 0x02000000, 0x04000000, 0x08000000,
    0x10000000, 0x20000000, 0x40000000, 0x80000000,
    0x1B000000, 0x36000000
};

AES::AES(const AES_TYPE t, uint8_t *key):
BlockCipher(16,ECB_MODE),
state()
{
    switch(t)
    {
        case AES_128:
            nr = 10;
            nk = 4;
        break;
        
        case AES_192:
            nr = 12;
            nk = 6;
        break;
        
        case AES_256:
            nr = 14;
            nk = 8;
        break;
    }
    
    keyExpansion(key);
}

AES::AES(const AES_TYPE t, uint8_t *key, uint8_t *iv, BLOCK_CIPHER_MODE m):
BlockCipher(16,m, iv),
state()
{
    switch(t)
    {
        case AES_128:
            nr = 10;
            nk = 4;
        break;
        
        case AES_192:
            nr = 12;
            nk = 6;
        break;
        
        case AES_256:
            nr = 14;
            nk = 8;
        break;
    }
    
    keyExpansion(key);
}

void AES::keyExpansion(uint8_t *key)
{
    uint32_t temp;
    int i = 0;
    
    while(i < nk)
    {
        w[i] = (key[4*i] << 24) + (key[4*i+1] << 16) + (key[4*i+2] << 8) + key[4*i+3];
        i++;
    }
    i = nk;
    while(i < 4*(nr+1))
    {
        temp = w[i-1];
        if(i % nk == 0)
        {
            temp = rotWord(temp);
            temp = subWord(temp);
            temp ^= rcon[i/nk-1];
        }
        else if(nk > 6 && i % nk == 4)
            temp = subWord(temp);
        w[i] = w[i-nk] ^ temp;
        i++;
    }
}

uint32_t AES::rotWord(uint32_t w)
{
    return (w << 8) + (w >> 24);
}

uint32_t AES::invRotWord(uint32_t w)
{
    return (w >> 8) + (w << 24);
}

uint32_t AES::subWord(uint32_t w)
{
    uint32_t out = 0;
    for(int i = 0; i < 4; ++i)
    {
        uint8_t temp = (w & 0xFF);
        out |= (sbox[temp] << (8*i));
        w = (w >> 8);
    }
    return out;
}

void AES::subBytes()
{
    for(int i = 0; i < 16; ++i)
        state[i] = sbox[state[i]];
}

void AES::invSubBytes()
{
    for(int i = 0; i < 16; ++i)
        state[i] = inv_s[state[i]];
}

void AES::shiftRows()
{
    for(int r = 0; r < 4; ++r)
    {   
        uint32_t temp = (state[r] << 24) + (state[r+4] << 16) + (state[r+8] << 8) + state[r+12];
        int i = r;
        while(i > 0)
        {
            temp = rotWord(temp);
            --i;
        }
        state[r] = temp >> 24;
        state[r+4] = temp >> 16;
        state[r+8] = temp >> 8;
        state[r+12] = temp;
    }
}

void AES::invShiftRows()
{
    for(int r = 0; r < 4; ++r)
    {
        uint32_t temp = (state[r] << 24) + (state[r+4] << 16) + (state[r+8] << 8) + state[r+12];
        int i = r;
        while(i > 0)
        {
            temp = invRotWord(temp);
            --i;
        }
        state[r] = temp >> 24;
        state[r+4] = temp >> 16;
        state[r+8] = temp >> 8;
        state[r+12] = temp;
    }
}

/* Multiply two numbers in the GF(2^8) finite field defined 
 * by the polynomial x^8 + x^4 + x^3 + x + 1 */
uint8_t gmul(uint8_t a, uint8_t b) 
{
    uint8_t p = 0;
    uint8_t counter;
    uint8_t carry;
    for (counter = 0; counter < 8; counter++) {
        if (b & 1) 
            p ^= a;
        carry = (a & 0x80);
        a <<= 1;
        if (carry) 
            a ^= 0x001B; /* what x^8 is modulo x^8 + x^4 + x^3 + x^2 + 1 */
        b >>= 1;
    }
    return p;
}

void AES::mul(uint8_t *r) 
{
    uint8_t tmp[4];
    memcpy(tmp, r, 4);
    r[0] = gmul(tmp[0],2) ^ gmul(tmp[1],3) ^ tmp[2] ^ tmp[3];
    r[1] = tmp[0] ^ gmul(tmp[1],2) ^ gmul(tmp[2],3) ^ tmp[3];
    r[2] = tmp[0] ^ tmp[1] ^ gmul(tmp[2],2) ^ gmul(tmp[3],3);
    r[3] = gmul(tmp[0],3) ^ tmp[1] ^ tmp[2] ^ gmul(tmp[3],2);
}

void AES::invMul(uint8_t *r)
{
    uint8_t tmp[4];
    memcpy(tmp, r, 4);
    r[0] = gmul(tmp[0],0x0e) ^ gmul(tmp[1],0x0b) ^ gmul(tmp[2],0x0d) ^ gmul(tmp[3],9);
    r[1] = gmul(tmp[0],9) ^ gmul(tmp[1],0x0e) ^ gmul(tmp[2],0x0b) ^ gmul(tmp[3],0x0d);
    r[2] = gmul(tmp[0],0x0d) ^ gmul(tmp[1],9) ^ gmul(tmp[2],0x0e) ^ gmul(tmp[3],0x0b);
    r[3] = gmul(tmp[0],0x0b) ^ gmul(tmp[1],0x0d) ^ gmul(tmp[2],9) ^ gmul(tmp[3],0x0e);
}

void AES::mixColumns()
{
    for(int c = 0; c < 4; ++c)
        mul(&state[4*c]);
}

void AES::invMixColumns()
{
    for(int c = 0; c < 4; ++c)
        invMul(&state[4*c]);
}

void AES::addRoundKey(int round)
{
    for(int c = 0; c < 4; ++c)
    {
        uint32_t temp = (state[4*c] << 24) + (state[4*c+1] << 16) + (state[4*c+2] << 8) + state[4*c+3];
        temp ^= w[round*4+c];
        state[4*c] = temp >> 24;
        state[4*c+1] = temp >> 16;
        state[4*c+2] = temp >> 8;
        state[4*c+3] = temp;
    }
}

void AES::decryptBlock(uint8_t *out, uint8_t *in)
{
    memcpy(state,in,16);
    
    addRoundKey(nr);
    
    for(int round = nr-1; round > 0; --round)
    {
        invShiftRows();
        invSubBytes();
        addRoundKey(round);
        invMixColumns();
    }
    invShiftRows();
    invSubBytes();
    addRoundKey(0);
    
    memcpy(out, state, 16);
}

void AES::encryptBlock(uint8_t *out, uint8_t *in)
{
    memcpy(state,in,16);
    
    addRoundKey(0);
    
    for(int round = 1; round < nr; ++round)
    {
        subBytes();
        shiftRows();
        mixColumns();
        addRoundKey(round);
    }
    subBytes();
    shiftRows();
    addRoundKey(nr);
    
    memcpy(out, state, 16);
}
