#ifndef _SERVO_
#define _SERVO_

#include "mbed.h"

/**
 *  A Servo controller library :)s
 *
 * @author CA Bezuidenhout
 */
class Servo {
public:
    /**
    * @param pwmPin : PWM pin of servo / orange wire
    * @param zeroAngle : The zero angle of the servo
    * @param minAngle : The minimum angle allowed for the servo
    * @param maxAngle : The maximum angle allowed for the servo
    */
    Servo(PinName pwmPin, float zeroAngle = 0, float minAngle = -90.0f, float maxAngle = 90.0f);
  
    /**
     * Saves the current position as the zero position.
     */
    void SaveZero();
  
    /**
     * Sets the zero position without moving the servo
     *
     * @param angle : The angle to set as the zero position
     */
    void SetZero(float angle);
  
    /**
     * Calibrate the servo by changing the pwm/deg ratio
     * @param actualAngle : The actual angle from zero of the current position in degrees.
     */
    void Calibrate(float actualAngle);
  
    /**
     * Sets the angular position of the servo
     * @param angle : Desired angle from the zero position in degrees
     */
    void Set(float angle);
  
    /**
     * Moves to an relative angular position from the current position
     * @param relAngle : The relative angle to move in degrees
     */
    void Move(float relAngle);
  
    /**
     * Disables control signal to servo
     */
    void Off();

    /**
     * Gets the current angle of the servo
     * @returns Current angle of the servo
     */ 
    float GetAngle();

    /**
     * Sets the minimum angle allowed for the servo
     * @params minAngle : minimum angle allowed
     */
    void SetMin(float minAngle);

    /**
     * Sets the maximum angle allowed for the servo
     * @params maxAngle : maximum angle allowed
     */
    void SetMax(float maxAngle);

    /**
     * Sets the servo to minimum position
     */
    void GotoMin();

    /**
     * Sets the servo to maximum position
     */
    void GotoMax();

    /**
     * Gets the current minimum angle of the servo
     * @returns Minimum angle of servo
     */
    float GetMin();

    /**
     * Gets the current maximum angle of the servo
     * @returns Maximum angle of servo
     */
    float GetMax();

private:
    PwmOut _servo;
    float _angle;
    float _ref;
    float _minAngle;
    float _maxAngle;
    
    float PW_PER_DEG;
    float ZEROPW;
    static const float MIN_PW = 0.0005;
    static const float MAX_PW = 0.0025f;
  
    void SetServo();
};
#endif